/*
 * @(#)LoginManager.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect.admin.login ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.io.IOCom;
import org.maachang.commons.sys.NamingManager;
import org.maachang.commons.sys.user.UserManager;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.UserInfo;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.Digest;
import org.maachang.queue.config.MqDefine;

/**
 * ログインユーザマネージャ.
 *  
 * @version 2006/12/31
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class LoginManager {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( LoginManager.class ) ;
    
    /**
     * ネーミングマネージャ登録名.
     */
    public static final String NAMING_MANAGER = "MANAGER@maachangq.login" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public LoginManager() { }
    
    
    /**
     * 初期化処理.
     * <BR><BR>
     * 初期化処理を行います.
     */
    public void init() {
        
        String dir = new StringBuffer().
            append( MqDefine.EMV_NAME ).
            append( MqDefine.USER_DIR ).
            toString() ;
        
        if( IOCom.isDirExists( dir ) == false ) {
            IOCom.mkdirs( dir ) ;
        }
        
        String file = new StringBuffer().
            append( dir ).
            append( "/" ).
            append( MqDefine.USER_FILE_NAME ).
            toString() ;
        dir = null ;
        
        UserManager man = new UserManager( false,file ) ;
        
        this.destroy( false ) ;
        
        LOG.info( "#### MaachangQログインユーザ情報生成処理 ####" ) ;
        
        // ネーミングマネージャに登録.
        synchronized( SYNC ) {
            
            NamingManager.add( NAMING_MANAGER,man ) ;
            
        }
        
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理を行います.
     */
    public void destroy() {
        this.destroy( true ) ;
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理を行います.
     * <BR>
     * @param mode ログ表示モードを設定します.
     */
    public void destroy( boolean mode ) {
        
        if( mode == true ) {
            LOG.info( "#### MaachangQログインユーザ情報破棄処理 ####" ) ;
        }
        
        synchronized( SYNC ) {
            
            try {
                UserManager man = LoginManager.getUserManager() ;
                if( man != null ) {
                    man.destroy() ;
                }
            } catch( Exception e ) {
            }
            
            NamingManager.remove( NAMING_MANAGER ) ;
            
        }
        
    }
    
    /**
     * ログイン認証.
     * <BR><BR>
     * ログイン認証を行います.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @param user 対象のログインユーザ名を設定します.
     * @param passwd SHA1変換されたパスワードを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void login( LoginSession session,
        String user,byte[] passwd )
        throws MaachangQException {
        
        if( session == null ) {
            throw new InputException( "設定セッションは不正です" ) ;
        }
        
        if( user == null || ( user = UtilCom.trimPlus( user ) ).length() <= 0 ) {
            throw new MaachangQException(
                "指定ユーザ名は不正です",
                MaachangQErrorCode.ERROR_USER ) ;
        }
        
        try {
            
            synchronized( SYNC ) {
                
                UserManager man = LoginManager.getUserManager() ;
                
                if( man.isUser( user ) == false ) {
                    throw new MaachangQException(
                        "ユーザは存在しません",
                        MaachangQErrorCode.ERROR_NOT_USER ) ;
                }
                else if( man.getNowUserCount( user ) == -1 ) {
                    // ゲストユーザ対応.
                    throw new MaachangQException(
                        "ユーザは存在しません",
                        MaachangQErrorCode.ERROR_NOT_USER ) ;
                }
                
                String passwdString = man.getPasswd( user ) ;
                if( passwdString == null || passwdString.length() <= 0 ) {
                    passwdString = CommonProtocol.DUMMY ;
                }
                
                byte[] sha1 = Digest.convertBinary(
                    CommonProtocol.DIGEST,
                    passwdString.getBytes( CommonProtocol.CHARSET ) ) ;
                
                int len = sha1.length ;
                for( int i = 0 ; i < len ; i ++ ) {
                    if( passwd[ i ] != sha1[ i ] ) {
                        throw new MaachangQException(
                            "パスワードが違います",
                            MaachangQErrorCode.ERROR_NOT_PASSWD ) ;
                    }
                }
                
                boolean owner = man.getRootOwner( user ) ;
                int id = man.getUserNameByUserID( user ) ;
                passwdString = man.getPasswd( user ) ;
                if( passwdString == null || passwdString.length() <= 0 ) {
                    passwdString = null ;
                }
                
                if( session.isSession() == true ) {
                    LoginManager.logout( session ) ;
                }
                
                session.create( id,user,passwdString,owner ) ;
                man.addUserCount( user ) ;
                
            }
        } catch( MaachangQException fa ) {
            throw fa ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "ログイン処理に失敗",
                MaachangQErrorCode.ERROR_LOGIN ) ;
        }
    }
    
    /**
     * ログアウト処理.
     * <BR><BR>
     * ログアウト処理を行います.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     */
    public static final void logout( LoginSession session ) {
        
        if( session == null || session.isSession() == false ) {
            return ;
        }
        
        try {
            synchronized( SYNC ) {
                
                String user = session.getUser() ;
                session.clear() ;
                UserManager man = LoginManager.getUserManager() ;
                
                man.removeUserCount( user ) ;
                
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * ユーザ追加.
     * <BR><BR>
     * 対象ユーザを追加します.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @param owner 対象の権限を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void add( LoginSession session,
        String user,String passwd,boolean owner )
        throws MaachangQException {
        
        LoginManager.checkLoginUser( true,session ) ;
        
        if( user == null || ( user = UtilCom.trimPlus( user ) ).length() <= 0 ) {
            throw new MaachangQException(
                "指定ユーザ名は不正です",
                MaachangQErrorCode.ERROR_USER ) ;
        }
        
        // ゲストユーザ新規追加せず、無制限ログインに設定.
        if( UserManager.GUEST_USER.equals( user ) == true ) {
            synchronized( SYNC ) {
                UserManager man = LoginManager.getUserManager() ;
                man.renewMaxUserCount( user,0 ) ;
            }
            return ;
        }
        
        if( passwd == null ||
            ( passwd = UtilCom.trimPlus( passwd ) ).length() <= 0 ) {
            passwd = null ;
        }
        
        synchronized( SYNC ) {
            
            UserManager man = LoginManager.getUserManager() ;
            
            if( man.isUser( user ) == true ) {
                throw new MaachangQException(
                    "指定ユーザ(" + user + ")は既に存在します",
                    MaachangQErrorCode.ERROR_USE_USER ) ;
            }
            
            try {
                man.addUser( user,passwd,owner,false,0,null ) ;
                man.save() ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "指定ユーザ(" + user + ")の追加に失敗しました",
                    MaachangQErrorCode.ERROR_ADD_USER ) ;
            }
            
        }
        
    }
    
    /**
     * ユーザ削除.
     * <BR><BR>
     * 対象のユーザを削除します.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @param user 対象のユーザ名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void remove( LoginSession session,String user )
        throws MaachangQException {
        
        LoginManager.checkLoginUser( true,session ) ;
        
        if( user == null || ( user = UtilCom.trimPlus( user ) ).length() <= 0 ) {
            throw new MaachangQException(
                "指定ユーザ名は不正です",
                MaachangQErrorCode.ERROR_USER ) ;
        }
        
        // ルートユーザは削除できない.
        if( UserManager.ROOT_USER.equals( user ) == true ) {
            throw new MaachangQException(
                "スーパーユーザは削除できません",
                MaachangQErrorCode.ERROR_NOT_DELETE_USER ) ;
        }
        
        // ゲストユーザは削除せず、ログインできないようにする.
        if( UserManager.GUEST_USER.equals( user ) == true ) {
            synchronized( SYNC ) {
                UserManager man = LoginManager.getUserManager() ;
                man.renewMaxUserCount( user,-1 ) ;
            }
            return ;
        }
        
        synchronized( SYNC ) {
            
            UserManager man = LoginManager.getUserManager() ;
            
            if( man.isUser( user ) == false ) {
                throw new MaachangQException(
                    "指定ユーザは存在しません",
                    MaachangQErrorCode.ERROR_NOT_USER ) ;
            }
            
            try {
                man.removeUser( user ) ;
                man.save() ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "ユーザ削除に失敗",
                    MaachangQErrorCode.ERROR_REMOVE_USER ) ;
            }
            
        }
        
    }
    
    /**
     * 指定ユーザのパスワードを変更.
     * <BR><BR>
     * 指定ユーザのパスワードを変更します.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void changePasswd( LoginSession session,
        String user,String passwd )
        throws MaachangQException {
        
        LoginManager.checkLoginUser( true,session ) ;
        
        if( user == null || ( user = UtilCom.trimPlus( user ) ).length() <= 0 ) {
            throw new MaachangQException(
                "指定ユーザ名は不正です",
                MaachangQErrorCode.ERROR_USER ) ;
        }
        
        // ルートユーザはパスワード変更できない.
        if( UserManager.ROOT_USER.equals( user ) == true ) {
            throw new MaachangQException(
                "rootユーザは、スーパーユーザログインでのみ変更可能です",
                MaachangQErrorCode.ERROR_EXEC_SUPER_USER ) ;
        }
        
        if( passwd == null ||
            ( passwd = UtilCom.trimPlus( passwd ) ).length() <= 0 ) {
            passwd = null ;
        }
        
        synchronized( SYNC ) {
            
            UserManager man = LoginManager.getUserManager() ;
            
            if( man.isUser( user ) == false ) {
                throw new MaachangQException(
                    "指定ユーザは存在しません",
                    MaachangQErrorCode.ERROR_NOT_USER ) ;
            }
            
            try {
                man.renewPasswd( user,passwd ) ;
                man.save() ;
                if( session.getUser().equals( user ) == true ) {
                    session.setPasswd( passwd ) ;
                }
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "ユーザパスワード変更に失敗",
                    MaachangQErrorCode.ERROR_CHANNGE_PASSWD ) ;
            }
            
        }
    }
    
    /**
     * 現在ログインユーザのパスワードを変更.
     * <BR><BR>
     * 現在ログインユーザのパスワードを変更します.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @param oldPasswd 対象の変更前パスワードを設定します.
     * @param newPasswd 対象の新規パスワードを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void changeThisPasswd( LoginSession session,
        String oldPasswd,String newPasswd )
        throws MaachangQException {
        
        LoginManager.checkLoginUser( false,session ) ;
        
        if( oldPasswd == null ||
            ( oldPasswd = UtilCom.trimPlus( oldPasswd ) ).length() <= 0 ) {
            oldPasswd = "" ;
        }
        if( newPasswd == null ||
            ( newPasswd = UtilCom.trimPlus( newPasswd ) ).length() <= 0 ) {
            newPasswd = null ;
        }
        
        synchronized( SYNC ) {
            
            UserManager man = LoginManager.getUserManager() ;
            
            if( man.isUser( session.getUser() ) == false ) {
                throw new MaachangQException(
                    "現在ログイン中のユーザ(" + session.getUser() +
                    "は既に削除されています",
                    MaachangQErrorCode.ERROR_NOT_USER ) ;
            }
            
            String nowPasswd = man.getPasswd( session.getUser() ) ;
            if( nowPasswd == null ||
                ( nowPasswd = UtilCom.trimPlus( nowPasswd ) ).length() <= 0 ) {
                nowPasswd = "" ;
            }
            
            if( nowPasswd.equals( oldPasswd ) == false ) {
                throw new MaachangQException(
                    "パスワードが一致しません",
                    MaachangQErrorCode.ERROR_NOT_PASSWD ) ;
            }
            
            try {
                man.renewPasswd( session.getUser(),newPasswd ) ;
                man.save() ;
                session.setPasswd( newPasswd ) ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "ユーザパスワード変更に失敗",
                    MaachangQErrorCode.ERROR_CHANNGE_PASSWD ) ;
            }
            
        }
        
    }
    
    /**
     * 権限変更.
     * <BR><BR>
     * 権限変更します.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @param user 対象のユーザ名を設定します.
     * @param owner 対象の権限を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void changeOwner( LoginSession session,
        String user,boolean owner )
        throws MaachangQException {
        
        LoginManager.checkLoginUser( true,session ) ;
        
        if( user == null || ( user = UtilCom.trimPlus( user ) ).length() <= 0 ) {
            throw new MaachangQException(
                "指定ユーザ名は不正です",
                MaachangQErrorCode.ERROR_USER ) ;
        }
        
        // ルートユーザは、権限を変更できない.
        if( UserManager.ROOT_USER.equals( user ) == true ) {
            throw new MaachangQException(
                "rootユーザは、権限を変更できません",
                MaachangQErrorCode.ERROR_EXEC_SUPER_USER ) ;
        }
        // ゲストユーザは、権限変更を行えない.
        else if( UserManager.GUEST_USER.equals( user ) == true ) {
            // この場合は無視する.
            return ;
        }
        
        synchronized( SYNC ) {
            
            UserManager man = LoginManager.getUserManager() ;
            
            if( man.isUser( user ) == false ) {
                throw new MaachangQException(
                    "指定ユーザは存在しません",
                    MaachangQErrorCode.ERROR_NOT_USER ) ;
            }
            
            try {
                man.renewRootOwner( user,owner ) ;
                man.save() ;
                if( session.getUser().equals( user ) == true ) {
                    session.setOwner( owner ) ;
                }
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "ユーザ権限変更に失敗",
                    MaachangQErrorCode.ERROR_CHANGE_OWNER ) ;
            }
            
        }
        
    }
    
    /**
     * ユーザ情報一覧を取得.
     * <BR><BR>
     * ユーザ情報一覧が返されます.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @retun UserInfo[] ユーザ情報群が返されます
     * @exception MaachangQException MaachangQ例外.
     */
    public static final UserInfo[] getUsers( LoginSession session )
        throws MaachangQException {
        
        LoginManager.checkLoginUser( true,session ) ;
        
        synchronized( SYNC ) {
            UserManager man = LoginManager.getUserManager() ;
            
            try {
                int len = 0 ;
                String[] names = man.getUsers() ;
                if( names != null && ( len = names.length ) > 0 ) {
                    UserInfo[] ret = new UserInfo[ len ] ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        
                        UserInfo info = new UserInfo() ;
                        info.setUser( names[ i ] ) ;
                        info.setRootOwner( man.getRootOwner( names[ i ] ) ) ;
                        
                        ret[ i ] = info ;
                        
                    }
                    return ret ;
                }
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "ユーザ一覧取得に失敗",
                    MaachangQErrorCode.ERROR_GET_USER_LIST ) ;
            }
        }
        
        return null ;
    }
    
    /**
     * ユーザ数を取得.
     * <BR><BR>
     * 登録されているユーザ数が返されます.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @return int ユーザ数が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final int getSize( LoginSession session )
        throws MaachangQException {
        LoginManager.checkLoginUser( true,session ) ;
        
        synchronized( SYNC ) {
            UserManager man = LoginManager.getUserManager() ;
            
            try {
                int ret = man.getUserLength() ;
                return ret ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "ユーザ数取得に失敗",
                    MaachangQErrorCode.ERROR_GET_USER_SIZE ) ;
            }
        }
    }
    
    /**
     * 指定ユーザのログイン数を取得.
     * <BR><BR>
     * 登録されている指定ユーザの現在ログイン数を取得します.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @param user 対象のユーザ名を設定します.
     * @return int 現在ログイン中のユーザ数が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final int getLoginUserCount( LoginSession session,
        String user )
        throws MaachangQException {
        LoginManager.checkLoginUser( true,session ) ;
        
        if( user == null || ( user = UtilCom.trimPlus( user ) ).length() <= 0 ) {
            throw new MaachangQException(
                "指定ユーザ名は不正です",
                MaachangQErrorCode.ERROR_USER ) ;
        }
        
        synchronized( SYNC ) {
            UserManager man = LoginManager.getUserManager() ;
            
            if( man.isUser( user ) == false ) {
                throw new MaachangQException(
                    "指定ユーザは存在しません",
                    MaachangQErrorCode.ERROR_NOT_USER ) ;
            }
            
            try {
                int ret = man.getNowUserCount( user ) ;
                return ret ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "ログインユーザ数取得に失敗",
                    MaachangQErrorCode.ERROR_GET_LOGIN_USER_SIZE ) ;
            }
        }
    }
    
    /**
     * ユーザに対する権限を取得.
     * <BR><BR>
     * ユーザに対する権限を取得します.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @param user 対象のユーザ名を設定します.
     * @return boolean 対象の権限が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final boolean getOwner( LoginSession session,
        String user )
        throws MaachangQException {
        LoginManager.checkLoginUser( true,session ) ;
        
        if( user == null || ( user = UtilCom.trimPlus( user ) ).length() <= 0 ) {
            throw new MaachangQException(
                "指定ユーザ名は不正です",
                MaachangQErrorCode.ERROR_USER ) ;
        }
        
        synchronized( SYNC ) {
            UserManager man = LoginManager.getUserManager() ;
            
            if( man.isUser( user ) == false ) {
                throw new MaachangQException(
                    "指定ユーザは存在しません",
                    MaachangQErrorCode.ERROR_NOT_USER ) ;
            }
            
            try {
                boolean ret = man.getRootOwner( user ) ;
                return ret ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "権限取得に失敗",
                    MaachangQErrorCode.ERROR_GET_USER_OWNER ) ;
            }
        }
    }
    
    /**
     * 対象ユーザが存在するか取得.
     * <BR><BR>
     * 対象ユーザが存在するか取得します.
     * <BR>
     * @param session 対象のログインセッションを設定します.
     * @param user 対象のユーザ名を設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                  [true]が返された場合、対象ユーザは存在します.<BR>
     *                  [false]が返された場合、対象ユーザは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final boolean isUser( LoginSession session,
        String user )
        throws MaachangQException {
        LoginManager.checkLoginUser( true,session ) ;
        
        if( user == null || ( user = UtilCom.trimPlus( user ) ).length() <= 0 ) {
            throw new MaachangQException(
                "指定ユーザ名は不正です",
                MaachangQErrorCode.ERROR_USER ) ;
        }
        
        synchronized( SYNC ) {
            UserManager man = LoginManager.getUserManager() ;
            
            if( man.isUser( user ) == false ) {
                throw new MaachangQException(
                    "指定ユーザは存在しません",
                    MaachangQErrorCode.ERROR_NOT_USER ) ;
            }
            
            try {
                boolean ret = man.isUser( user ) ;
                return ret ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "権限取得に失敗",
                    MaachangQErrorCode.ERROR_IS_USER ) ;
            }
        }
    }
    
    /**
     * 現在ログインユーザの状態チェック.
     * <BR><BR>
     * 現在ログインユーザの状態をチェックします.
     * <BR>
     * @param mode 権限チェックを行うか設定します.<BR>
     *             [true]を設定した場合、権限チェックを行います.<BR>
     *             [false]を設定した場合、権限チェックを行いません.
     * @param session 対象のログインセッションを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void checkLoginUser( boolean mode,LoginSession session )
        throws MaachangQException {
        
        if( session == null || session.isSession() == false ) {
            throw new MaachangQException(
                "ログインされていません",
                MaachangQErrorCode.ERROR_NO_LOGIN ) ;
        }
        
        String passwd = session.getPasswd() ;
        if( passwd == null ||
            ( passwd = UtilCom.trimPlus( passwd ) ).length() <= 0 ) {
            passwd = "" ;
        }
        
        try {
            
            synchronized( SYNC ) {
                UserManager man = LoginManager.getUserManager() ;
                
                if( man.isUser( session.getUser() ) == false ) {
                    throw new MaachangQException(
                        "ログインユーザは削除されています",
                        MaachangQErrorCode.ERROR_NOT_USER ) ;
                }
                
                String nowPasswd = man.getPasswd( session.getUser() ) ;
                if( nowPasswd == null ||
                    ( nowPasswd = UtilCom.trimPlus( nowPasswd ) ).length() <= 0 ) {
                    nowPasswd = "" ;
                }
                
                if( nowPasswd.equals( passwd ) == false ) {
                    throw new MaachangQException(
                        "他ユーザによりパスワードが変更されました",
                        MaachangQErrorCode.WARNING_CHANGE_PASSWD ) ;
                }
                
                boolean owner = man.getRootOwner( session.getUser() ) ;
                session.setOwner( owner ) ;
            }
            
        } catch( MaachangQException fm ) {
            LoginManager.logout( session ) ;
            throw fm ;
        }
        
        if( mode == true && session.isOwner() == false ) {
            throw new MaachangQException(
                "権限が足りません",
                MaachangQErrorCode.ERROR_LOGIN_NOT_OWNWE ) ;
        }
        
    }
    
    /**
     * ユーザ管理オブジェクトを取得.
     */
    private static final UserManager getUserManager() {
        
        UserManager ret = null ;
        
        synchronized( SYNC ) {
            ret = ( UserManager )NamingManager.get(
                NAMING_MANAGER ) ;
        }
        
        return ret ;
        
    }
    
}

