/*
 * @(#)ClientConnectService.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect.service ;

import java.net.ServerSocket;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.sys.NamingManager;
import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.net.CreateSocket;
import org.maachang.queue.access.net.SSLOption;
import org.maachang.queue.config.MqConfig;
import org.maachang.queue.config.SSLConfig;
import org.maachang.queue.config.service.ConfigService;
import org.maachang.queue.connect.ExecutionServerAdmin;
import org.maachang.queue.connect.ExecutionServerMessage;
import org.maachang.queue.connect.admin.ExecutionAdminCache;
import org.maachang.queue.connect.admin.ExecutionAdminChannel;
import org.maachang.queue.connect.admin.ExecutionAdminConnect;
import org.maachang.queue.connect.admin.ExecutionAdminLogin;
import org.maachang.queue.connect.admin.ExecutionAdminQueue;
import org.maachang.queue.connect.admin.ExecutionAdminQueueManager;
import org.maachang.queue.connect.admin.ExecutionAdminResultLogin;
import org.maachang.queue.connect.admin.ExecutionAdminUser;
import org.maachang.queue.connect.admin.login.LoginManager;
import org.maachang.queue.connect.common.ServerConnectThread;

/**
 * クライアントコネクションサービス.
 *  
 * @version 2007/01/25
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class ClientConnectService {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ClientConnectService.class ) ;
    
    /**
     * ネーミングマネージャ登録名.
     */
    public static final String NAMING_MANAGER = "CLIENT@maachangq.connect.service" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public ClientConnectService() { }
    
    
    /**
     * 初期化処理.
     * <BR><BR>
     * 初期化処理を行います.
     * <BR>
     * @exception Exception 例外.
     */
    public void init()
        throws Exception {
        
        LOG.info( "#### MaachangQクライアントコネクションサービス生成処理 ####" ) ;
        
        // 現在の条件を破棄.
        this.destroy( false ) ;
        
        ClientConnectChild cl = new ClientConnectChild() ;
        
        ServerSocket sock = null ;
        MqConfig conf = ConfigService.getMqConfig() ;
        
        // 通常データ電文用コネクションを生成.
        sock = CreateSocket.serverSocket( false,false,
            conf.getBindPort( 0 ),conf.getBindAddress( 0 ),
            conf.getMaxAccess() ) ;
        
        cl.setDataConnect( this.dataConnect( sock ) ) ;
        
        // 通常管理者用コネクションを生成.
        sock = CreateSocket.serverSocket( false,false,
            conf.getAdminPort(),conf.getAdminAddress(),
            MaachangQAccessDefine.MAX_ACCESS_ADMIN_PORT ) ;
        
        cl.setAdminConnect( this.adminConnect( sock ) ) ;
        
        SSLConfig sslConfig = ConfigService.getSSLConfig() ;
        
        // SSLが有効な場合.
        if( sslConfig.isSSL() == true ) {
            
            // SSL初期化処理.
            SSLOption opt = new SSLOption() ;
            opt.setKeyManagerAlgorithm( sslConfig.getKeyManagerAlgorithm() ) ;
            opt.setKeyStore( sslConfig.getKeyStore() ) ;
            opt.setKeyStoreFile( sslConfig.getKeyFile() ) ;
            opt.setKeyStorePasswd( sslConfig.getKeyPasswd() ) ;
            opt.setRandomAlgorithm( sslConfig.getRandomAlgorithm() ) ;
            opt.setSslProtocol( sslConfig.getSSlProtocol() ) ;
            
            if( sslConfig.isTrust() == true ) {
                opt.setTrustFile( sslConfig.getTrustFile() ) ;
                opt.setTrustKeyManagerAlgorithm( sslConfig.getTrustManagerAlgorithm() ) ;
                opt.setTrustPassword( sslConfig.getTrustPasswd() ) ;
                opt.setTrustStore( sslConfig.getTrustStore() ) ;
            }
            
            CreateSocket.initServerSSL( opt ) ;
            
            // SSLデータ電文用コネクションを生成.
            sock = CreateSocket.serverSocket( true,sslConfig.isClientAuth(),
                conf.getSSLBindPort( 0 ),conf.getSSLBindAddress( 0 ),
                conf.getMaxAccess() ) ;
            
            cl.setDataSSLConnect( this.dataConnect( sock ) ) ;
            
            // SSL管理者用コネクションを生成.
            sock = CreateSocket.serverSocket( true,sslConfig.isClientAuth(),
                conf.getSSLAdminPort(),conf.getSSLAdminAddress(),
                MaachangQAccessDefine.MAX_ACCESS_ADMIN_PORT ) ;
            
            cl.setAdminSSLConnect( this.adminConnect( sock ) ) ;
            
        }
        
        // ネーミングマネージャに登録.
        synchronized( SYNC ) {
            
            NamingManager.add( NAMING_MANAGER,cl ) ;
            
        }
        
        new LoginManager().init() ;
        
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理を行います.
     */
    public void destroy() {
        this.destroy( true ) ;
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理を行います.
     * <BR>
     * @param mode ログ表示モードを設定します.
     */
    public void destroy( boolean mode ) {
        
        if( mode == true ) {
            LOG.info( "#### MaachangQクライアントコネクションサービス破棄処理 ####" ) ;
        }
        
        // ネーミングマネージャを破棄.
        synchronized( SYNC ) {
            
            NamingManager.remove( NAMING_MANAGER ) ;
            
        }
        
        new LoginManager().destroy( mode ) ;
        
    }
    
    /**
     * Adminコネクションオブジェクトを取得.
     */
    private ServerConnectThread adminConnect( ServerSocket sock ) {
        
        ExecutionServerAdmin ch = new ExecutionServerAdmin() ;
        ch.add( new ExecutionAdminCache() ) ;
        ch.add( new ExecutionAdminChannel() ) ;
        ch.add( new ExecutionAdminConnect() ) ;
        ch.add( new ExecutionAdminLogin() ) ;
        ch.add( new ExecutionAdminQueue() ) ;
        ch.add( new ExecutionAdminQueueManager() ) ;
        ch.add( new ExecutionAdminResultLogin() ) ;
        ch.add( new ExecutionAdminUser() ) ;
        
        ServerConnectThread ret = new ServerConnectThread(
            sock,ch ) ;
        
        return ret ;
    }
    
    /**
     * 通常コネクションオブジェクトを取得.
     */
    private ServerConnectThread dataConnect( ServerSocket sock ) {
        return new ServerConnectThread(
            sock,new ExecutionServerMessage() ) ;
    }
    
}

/**
 * クライアントコネクションチャイルド.
 */
class ClientConnectChild {
    
    private ServerConnectThread dataConnect = null ;
    private ServerConnectThread dataSSLConnect = null ;
    private ServerConnectThread adminConnect = null ;
    private ServerConnectThread adminSSLConnect = null ;
    
    /**
     * adminConnectを取得.
     * <BR><BR>
     * @return adminConnect が返されます。
     */
    public ServerConnectThread getAdminConnect() {
        return adminConnect;
    }
    /**
     * adminConnectを設定.
     * <BR><BR>
     * @param adminConnect 対象のadminConnectを設定.
     */
    public void setAdminConnect(ServerConnectThread adminConnect) {
        this.adminConnect = adminConnect;
    }
    /**
     * adminSSLConnectを取得.
     * <BR><BR>
     * @return adminSSLConnect が返されます。
     */
    public ServerConnectThread getAdminSSLConnect() {
        return adminSSLConnect;
    }
    /**
     * adminSSLConnectを設定.
     * <BR><BR>
     * @param adminSSLConnect 対象のadminSSLConnectを設定.
     */
    public void setAdminSSLConnect(ServerConnectThread adminSSLConnect) {
        this.adminSSLConnect = adminSSLConnect;
    }
    /**
     * dataConnectを取得.
     * <BR><BR>
     * @return dataConnect が返されます。
     */
    public ServerConnectThread getDataConnect() {
        return dataConnect;
    }
    /**
     * dataConnectを設定.
     * <BR><BR>
     * @param dataConnect 対象のdataConnectを設定.
     */
    public void setDataConnect(ServerConnectThread dataConnect) {
        this.dataConnect = dataConnect;
    }
    /**
     * dataSSLConnectを取得.
     * <BR><BR>
     * @return dataSSLConnect が返されます。
     */
    public ServerConnectThread getDataSSLConnect() {
        return dataSSLConnect;
    }
    /**
     * dataSSLConnectを設定.
     * <BR><BR>
     * @param dataSSLConnect 対象のdataSSLConnectを設定.
     */
    public void setDataSSLConnect(ServerConnectThread dataSSLConnect) {
        this.dataSSLConnect = dataSSLConnect;
    }
    
}
