/*
 * @(#)LastReceiveIdManage.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel ;

import org.maachang.queue.main.queue.base.core.QSendSeq;


/**
 * 送信ID管理１要素.
 * <BR><BR>
 * 最終受信IDの１要素を管理するオブジェクト.
 *  
 * @version 2006/01/29
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class LastReceiveIdChild {
    
    /**
     * キューマネージャ名.
     */
    private String queueManager = null ;
    
    /**
     * キュー名.
     */
    private String queue = null ;
    
    /**
     * 通信先マックアドレス.
     */
    private long macAddress = 0L ;
    
    /**
     * 送信ID.
     */
    private long sendId = 0L ;
    
    /**
     * 更新時間.
     */
    private long updateTime = -1L ;
    
    /**
     * コンストラクタ.
     */
    protected LastReceiveIdChild() {
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 新しい情報を生成します.
     * <BR>
     * @param queueManager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param macAddress 接続先のマックアドレスを設定します.
     */
    protected synchronized void create( String queueManager,
        String queue,long macAddress ) {
        this.queueManager = queueManager ;
        this.queue = queue ;
        this.macAddress = macAddress ;
        this.sendId = 0L ;
        this.updateTime = System.currentTimeMillis() ;
    }
    
    /**
     * 情報破棄.
     * <BR><BR>
     * 情報を破棄します.
     */
    protected synchronized void destroy() {
        queueManager = null ;
        queue = null ;
        macAddress = 0L ;
        updateTime = -1L ;
        
        this.clear() ;
    }
    
    /**
     * 新しく生成した場合に、通信IDを設定.
     * <BR><BR>
     * 新しく生成した場合に、通信IDを設定します.
     * <BR>
     * @param id 対象の通信IDを設定します.
     */
    public synchronized void firstSendId( long id ) {
        if( sendId == 0L ) {
            this.sendId = id ;
            this.updateTime = System.currentTimeMillis() ;
        }
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 情報を削除します.
     */
    protected synchronized void clear() {
        sendId = 0L ;
    }
    
    /**
     * キューマネージャ名を取得.
     * <BR><BR>
     * 設定されているキューマネージャ名を取得します.
     * <BR>
     * @return String キューマネージャ名が返されます.
     */
    public synchronized String getQueueManager() {
        return queueManager ;
    }
    
    /**
     * キュー名を取得.
     * <BR><BR>
     * キュー名を取得します.
     * <BR>
     * @return String キュー名が返されます.
     */
    public synchronized String getQueue() {
        return queue ;
    }
    
    /**
     * マックアドレスを取得.
     * <BR><BR>
     * マックアドレスを取得します.
     * <BR>
     * @return long マックアドレスが返されます.
     */
    public synchronized long getMacAddress() {
        return macAddress ;
    }
    
    /**
     * 最終更新時間を取得.
     * <BR><BR>
     * 現在の最終更新時間を取得します.
     * <BR>
     * @return long 現在の最終更新時間が返されます.
     */
    public synchronized long getUpdateTime() {
        return updateTime ;
    }
    
    /**
     * 受信可能かチェック.
     * <BR><BR>
     * 対象IDの情報が受信可能かチェックします.
     * <BR>
     * @param id 対象の受信IDを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、受信情報設定処理が
     *                 処理可能です.
     */
    public synchronized boolean isReceive( long id ) {
        
        int nextId = QSendSeq.getNowNextIdByNextId( sendId ) ;
        int checkNowId = QSendSeq.getNowNextIdByNowId( id ) ;
        
        if( sendId == 0L || nextId == checkNowId ) {
            this.sendId = id ;
            this.updateTime = System.currentTimeMillis() ;
            return true ;
        }
        
        return false ;
    }
    
}

