/*
 * @(#)SendChannelImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel ;

import java.net.InetAddress;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.SequenceIDEx;
import org.maachang.queue.access.status.ChannelStatus;
import org.maachang.queue.main.connect.Connect;
import org.maachang.queue.main.connect.ConnectFactory;

/**
 * チャネルオブジェクト.
 * 
 * @version 2006/09/02
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class SendChannelImple extends ChannelImple implements SendChannel,Channel {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( SendChannelImple.class ) ;
    
    /**
     * 接続先のキューマネージャ名.
     */
    protected String queueManager = null ;
    
    /**
     * 接続先のIPアドレス.
     */
    protected InetAddress inetAddress = null ;
    
    /**
     * 接続先のポート番号.
     */
    protected int port = -1 ;
    
    /**
     * 通信オブジェクト名.
     */
    protected String connectName = null ;
    
    /**
     * チャネルID発行オブジェクト.
     */
    protected SequenceIDEx sequence = null ;
    
    /**
     * オプション情報.
     */
    protected Object option = null ;
    
    /**
     * 利用暗号ワード.
     */
    protected String useCb32Word = null ;
    
    /**
     * コンストラクタ.
     */
    protected SendChannelImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を設定して、オブジェクトを生成します.
     * <BR>
     * @param name チャネル名を設定します.
     * @param queueManager 対象のキューマネージャ名を設定します.
     * @param addr 接続先アドレスを設定します.
     * @param port 接続先ポート番号を設定します.
     * @param connectName 通信処理で利用する通信オブジェクト名を設定します.
     * @param cb32Word 対象の利用暗号ワードを設定します.
     * @exception Exception 例外.
     */
    public SendChannelImple(
        String name,String queueManager,InetAddress addr,int port,
        String connectName,String useCb32Word )
        throws Exception {
            
        if(
            name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ||
            addr == null || port < 0 || port > 65535 ||
            connectName == null ||
            ( connectName = connectName.trim().toLowerCase() ).length() <= 0 ||
            queueManager == null ||
            ( queueManager = queueManager.trim().toLowerCase() ).length() <= 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        sync.create() ;
        
        try {
            int id = ( ( ( name.hashCode() & 0x3fffffff ) / 2 ) & 0x3fffffff ) ;
            synchronized( ChannelFactory.getSync() ) {
                for( ;; ) {
                    if( id != -1 && ChannelFactory.getToID( id ) != null ) {
                        id ++ ;
                    }
                    else {
                        break ;
                    }
                }
                
                synchronized( sync.get() ) {
                    this.name = name ;
                    this.queueManager = queueManager ;
                    this.id = id ;
                    this.type = Channel.TYPE_SEND ;
                    this.inetAddress = addr ;
                    this.port = port ;
                    this.connectName = connectName ;
                    this.state = ChannelStatus.STATE_STARTUP ;
                    this.sequence = new SequenceIDEx() ;
                    this.channelKey = new ChannelKey( Channel.TYPE_SEND,name ) ;
                    this.setUseCb32Word( useCb32Word ) ;
                }
            }
        } catch( Exception e ) {
            this.destroy() ;
            throw e ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * チャネルオブジェクトを破棄します.
     */
    public void destroy() {
        
        try {
            synchronized( sync.get() ) {
                name = null ;
                queueManager = null ;
                id = -1 ;
                inetAddress = null ;
                port = -1 ;
                connectName = null ;
                state = ChannelStatus.STATE_DOWN ;
                option = null ;
                channelKey = null ;
                useCb32Word = null ;
            }
        } catch( Exception e ) {
        }
        
        name = null ;
        queueManager = null ;
        id = -1 ;
        inetAddress = null ;
        port = -1 ;
        connectName = null ;
        state = ChannelStatus.STATE_DOWN ;
        option = null ;
        channelKey = null ;
        
        sync.clear() ;
    }
    
    /**
     * キューマネージャ名を取得.
     * <BR><BR>
     * キューマネージャ名を取得します.
     * <BR>
     * @param name 対象のキューマネージャ名を設定します.
     */
    public void setQueueManager( String name ) {
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                queueManager = name ;
                this.resetInfo() ;
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * キューマネージャ名を取得.
     * <BR><BR>
     * キューマネージャ名を取得します.
     * <BR>
     * @param String 接続先のキューマネージャ名を取得します.
     */
    public String getQueueManager() {
        
        String ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = queueManager ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 接続先IPアドレスを設定.
     * <BR><BR>
     * 接続先のIPアドレスを設定します.
     * <BR>
     * @param addr 接続先のIPアドレスを設定します.
     */
    public void setInetAddress( InetAddress addr ) {
        
        if( addr == null ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                inetAddress = addr ;
                this.resetInfo() ;
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * 接続先IPアドレスを取得.
     * <BR><BR>
     * 接続先のIPアドレスを取得します.
     * <BR>
     * @return InetAddress 接続先のIPアドレスが返されます.
     */
    public InetAddress getInetAddress() {
        
        InetAddress ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = inetAddress ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 接続先ポート番号を設定.
     * <BR><BR>
     * 接続先のポート番号を設定します.
     * <BR>
     * @param port 接続先のポート番号を設定します.
     */
    public void setPort( int port ) {
        
        if( port < 0 || port > 65535 ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                this.port = port ;
                this.resetInfo() ;
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * 接続先ポート番号を取得.
     * <BR><BR>
     * 接続先のポート番号を取得します.
     * <BR>
     * @return int 接続先のポート番号が返されます.
     */
    public int getPort() {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                ret = port ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 通信オブジェクト名を設定.
     * <BR><BR>
     * 通信処理を行うオブジェクト名を設定します.
     * <BR>
     * @return connectName 通信オブジェクトを設定します.
     */
    public void setConnectName( String connectName ) {
        
        if(
            connectName == null ||
            ( connectName = connectName.trim().toLowerCase() ).length() <= 0
        ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                this.connectName = connectName ;
                this.resetInfo() ;
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * 通信オブジェクト名を取得.
     * <BR><BR>
     * 通信処理を行うオブジェクト名を取得します.
     * <BR>
     * @return String 通信オブジェクト名が返されます.
     */
    public String getConnectName() {
        
        String ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = connectName ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 利用暗号ワードを設定.
     * <BR><BR>
     * 利用暗号ワードを設定します.
     * <BR>
     * @param useCb32Word 対象の利用暗号ワードを設定します.
     */
    public void setUseCb32Word( String useCb32Word ) {
        
        if( useCb32Word == null || ( useCb32Word = useCb32Word.trim() ).length() <= 0 ) {
            useCb32Word = null ;
        }
        else if( this.isCb32Word( connectName,useCb32Word ) == false ) {
            useCb32Word = null ;
        }
        
        try {
            synchronized( sync.get() ) {
                this.useCb32Word = useCb32Word ;
                this.resetInfo() ;
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * 利用暗号ワードを取得.
     * <BR><BR>
     * 利用暗号ワードを取得します.
     * <BR>
     * @return String String 利用暗号ワードが返されます.
     */
    public String getUseCb32Word() {
        String ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.useCb32Word ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * シーケンスIDを取得.
     * <BR><BR>
     * シーケンスID情報を取得します.
     * <BR>
     * @return long シーケンスIDが返されます.
     */
    public long getSequenceID() {
        
        long ret = -1L ;
        
        try {
            synchronized( sync.get() ) {
                ret = sequence.getID() ;
            }
        } catch( Exception e ) {
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * チャネルBeanオブジェクトを取得.
     * <BR><BR>
     * チャネルBeanオブジェクトを取得します.
     * <BR>
     * @return ChannelBean チャネルBeanオブジェクトが返されます.
     */
    public ChannelBean getChannelBean() {
        
        ChannelBean ret = null ;
        
        try {
            synchronized( sync.get() ) {
                
                ret = new ChannelBean() ;
                ret.setType( this.type ) ;
                ret.setChannelName( this.name ) ;
                ret.setQueueManager( this.queueManager ) ;
                
                if( this.inetAddress == null ) {
                    ret.setConnectAddress( null ) ;
                }
                else {
                    ret.setConnectAddress( this.inetAddress.getHostName() ) ;
                }
                ret.setConnectObjectName( this.connectName ) ;
                ret.setConnectPort( this.port ) ;
                
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * チャネルBeanオブジェクトを取得.
     * <BR><BR>
     * チャネルBeanオブジェクトを取得します.
     * <BR>
     * @param bean 設定対象のBeanを設定します.
     */
    public void getBean( ChannelBean bean ) {
        try {
            synchronized( sync.get() ) {
                
                super.getBean( bean ) ;
                
                bean.setType( this.type ) ;
                bean.setChannelName( this.name ) ;
                bean.setQueueManager( this.queueManager ) ;
                
                if( this.inetAddress == null ) {
                    bean.setConnectAddress( null ) ;
                }
                else {
                    bean.setConnectAddress( this.inetAddress.getHostName() ) ;
                }
                bean.setConnectObjectName( this.connectName ) ;
                bean.setConnectPort( this.port ) ;
                
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * ステータスを設定.
     * <BR><BR>
     * チャネルステータスを設定します.
     * <BR>
     * @param state 対象のチャネルステータスを設定します.
     */
    public void setState( int state ) {
        
        if(
            state != ChannelStatus.STATE_SUCCESS &&
            state != ChannelStatus.STATE_DOWN &&
            state != ChannelStatus.STATE_STARTUP &&
            state != ChannelStatus.STATE_SHUTDOWN &&
            state != ChannelStatus.STATE_ERROR &&
            state != ChannelStatus.STATE_NOT_HEART_BEAT &&
            state != ChannelStatus.STATE_NOT_CONNECT_CHANNEL
        ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                if( this.state != state ) {
                    LOG.info(
                        new StringBuffer().
                        append( "送信チャネル(" ).
                        append( name ).
                        append( ")のステータスは " ).
                        append( ChannelStatus.getStateByString( this.state ) ).
                        append( " から " ).
                        append( ChannelStatus.getStateByString( state ) ).
                        append( " に変更されました" ).
                        toString() ) ;
                    this.state = state ;
                }
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * オプション情報を設定.
     * <BR><BR>
     * オプション情報を設定します.
     * <BR>
     * @param option 対象のオプション情報を設定します
     */
    public void setOption( Object option ) {
        try {
            synchronized( sync.get() ) {
                this.option = option ;
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * オプション情報を取得.
     * <BR><BR>
     * オプション情報を取得します.
     * <BR>
     * @return Object オプション情報が返されます.
     */
    public Object getOption() {
        Object ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.option ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 最終受信ID管理オブジェクトを取得.
     * <BR><BR>
     * 最終受信ID管理オブジェクトを取得します.
     * <BR>
     * @return LastReceiveIdManage 最終受信ID管理オブジェクトが返されます.
     */
    public LastReceiveIdManage getLastReceiveIdManage() {
        return null ;
    }
    
    /**
     * CB32ワードが設定可能かチェック.
     * <BR><BR>
     * CB32ワードが設定可能かチェックします.
     * <BR>
     * @param connect 対象のコネクション名を設定します.
     * @param word チェック対象のワードを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、設定可能です.<BR>
     *                 [false]が返された場合、設定不可です.
     */
    public boolean isCb32Word( String connect,String word ) {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                if( word != null && ( word = word.trim() ).length() > 0 ) {
                    
                    if(
                        connect == null ||
                        ( connect = connect.trim().toLowerCase() ).length() <= 0
                    ) {
                        connect = this.getConnectName() ;
                    }
                    
                    Connect conn = ConnectFactory.get( connect ) ;
                    
                    if( conn != null ) {
                        String[] names = conn.getCb32Word() ;
                        
                        if( names == null || names.length <= 0 ) {
                            return false ;
                        }
                        
                        int len = names.length ;
                        for( int i = 0 ; i < len ; i ++ ) {
                            if( word.equals( names[ i ] ) == true ) {
                                ret = true ;
                                break ;
                            }
                        }
                    }
                    
                }
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * 情報再設定時のステータス変更.
     */
    private final void resetInfo() {
        
        if( this.state == ChannelStatus.STATE_SUCCESS ) {
            this.state = ChannelStatus.STATE_STARTUP ;
        }
        
    }
    
}

