/*
 * @(#)ChannelProtocol.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.protocol ;

import org.maachang.commons.conv.CodeBase32;
import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.net.NetDef;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.BufferedBinResource;
import org.maachang.commons.resource.ConvertResourceParam;
import org.maachang.commons.resource.Resource;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.util.ConvertParam;
import org.maachang.commons.util.UtilCom;
import org.maachang.commons.util.zip.GZIPBinary;
import org.maachang.queue.config.MqDefine;
import org.maachang.queue.main.queue.QueueMessage;

/**
 * チャネル通信プロトコル.
 * 
 * @version 2006/08/30
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class ChannelProtocol
{
    
    /**
     * プロトコルタイプ : データ電文.
     */
    public static final int PROTOCOL_DATA = 0x000000da ;
    
    /**
     * プロトコルタイプ : ハートビート電文.
     */
    public static final int PROTOCOL_HEARTBEAT = 0x000000bb ;
    
    /**
     * プロトコルタイプ : 正常戻り電文.
     */
    public static final int PROTOCOL_SUCCESS = 0x000000ac ;
    
    /**
     * プロトコルタイプ : エラー戻り電文.
     */
    public static final int PROTOCOL_ERROR = 0x000000e0 ;
    
    
    
    /**
     * データヘッダ.
     */
    private static final byte HEAD_DATA = ( byte )PROTOCOL_DATA ;
    
    /**
     * ハートビートヘッダ.
     */
    private static final byte HEAD_HEARTBEAT = ( byte )PROTOCOL_HEARTBEAT ;
    
    /**
     * 正常戻り値ヘッダ.
     */
    private static final byte HEAD_SUCCESS = ( byte )PROTOCOL_SUCCESS ;
    
    /**
     * エラー戻り値ヘッダ.
     */
    private static final byte HEAD_ERROR = ( byte )PROTOCOL_ERROR ;
    
    /**
     * 非暗号ヘッダ.
     */
    private static final byte HEAD_NO_CB32 = ( byte )0x000000ff ;
    
    /**
     * 暗号ヘッダ.
     */
    private static final byte HEAD_CB32 = ( byte )0x00000032 ;
    
    
    
    /**
     * ヘッダ情報 : データ電文(非暗号).
     */
    private static final byte[] HEADER_DATA = {
        HEAD_DATA,HEAD_NO_CB32
    } ;
    
    /**
     * ヘッダ情報 : データ電文(CB32暗号).
     */
    private static final byte[] HEADER_DATA_BY_CB32 = {
        HEAD_DATA,HEAD_CB32
    } ;
    
    /**
     * ヘッダ情報 : ハートビート電文(非暗号).
     */
    private static final byte[] HEADER_HEARTBEAT = {
        HEAD_HEARTBEAT,HEAD_NO_CB32
    } ;
    
    /**
     * ヘッダ情報 : 正常電文(非暗号).
     */
    private static final byte[] HEADER_SUCCESS = {
        HEAD_SUCCESS,HEAD_NO_CB32
    } ;
    
    /**
     * ヘッダ情報 : エラー電文(非暗号).
     */
    private static final byte[] HEADER_ERROR = {
        HEAD_ERROR,HEAD_NO_CB32
    } ;
    
    /**
     * 圧縮条件.
     */
    private static final int GZIP_LENGTH = MqDefine.CHANNEL_GZIP_LENGTH ;
    
    /**
     * 圧縮フラグ最大値.
     */
    private static final int GZIP_FLG = 0x000000ff - 2 ;
    
    
    
    /**
     * 電文からタイプを取得.
     * <BR><BR>
     * 指定電文からタイプを取得します.
     * <BR>
     * @param telegram チェック対象の電文を設定します.
     * @return int 電文タイプが返されます.<BR>
     *             [PROTOCOL_DATA]が返された場合、[ProtocolData]オブジェクトで処理します.<BR>
     *             [PROTOCOL_HEARTBEAT]が返された場合、[ProtocolHeartBeat]オブジェクトで処理します.<BR>
     *             [PROTOCOL_SUCCESS]が返された場合、[ProtocolResultSuccess]オブジェクトで処理します.<BR>
     *             [PROTOCOL_ERROR]が返された場合、[ProtocolResultError]オブジェクトで処理します.<BR>
     *             [-1]が返された場合、対象電文は不正です.
     */
    public static final int getTelegramType( BinResource telegram ) {
        
        int hd ;
        int ret = -1 ;
        
        if( telegram == null || telegram.size() <= 0 ) {
            return -1 ;
        }
        
        hd = ( int )( telegram.get( 0 ) & 0x000000ff ) ;
        
        if(
            hd == PROTOCOL_DATA ||
            hd == PROTOCOL_HEARTBEAT ||
            hd == PROTOCOL_SUCCESS ||
            hd == PROTOCOL_ERROR
        ) {
            ret = hd ;
        }
        
        return ret ;
    }
    
    /**
     * 電文が暗号電文であるか取得.
     * <BR><BR>
     * 指定電文が暗号電文であるか取得します.
     * <BR>
     * @param telegram チェック対象の電文を設定します.
     * @return boolean 暗号電文であるか返されます.<BR>
     *                  [true]が返された場合、暗号電文です.<BR>
     *                  [false]が返された場合、暗号電文ではありません.
     */
    public static final boolean getTelegramCb32( BinResource telegram ) {
        
        int hd ;
        boolean ret = false ;
        
        if( telegram == null || telegram.size() <= 0 ) {
            return false ;
        }
        
        hd = ( int )( telegram.get( 1 ) & 0x000000ff ) ;
        
        if( hd == ( int )( HEAD_CB32 & 0x000000ff ) ) {
            ret = true ;
        }
        
        return ret ;
    }
    
    /**
     * 指定電文からキューマネージャ名を取得.
     * <BR><BR>
     * 指定された電文からキューマネージャ名を取得します.
     * <BR>
     * @param telegram 対象の電文を設定します.
     * @return String 対象のキューマネージャ名が返されます.
     */
    public static final String getQueueManager( BinResource telegram ) {
        
        int pnt ;
        int len ;
        
        byte[] bin = null ;
        String ret = null ;
        
        try {
            
            // フル基本ヘッダ条件の場合.
            if( telegram.get( 0 ) == PROTOCOL_DATA ) {
                // データプロトコル.
                
                pnt = 2 + 4 + 2 + NetDef.MAC_ADDRESS_LENGTH + 32 ;// ヘッダシフト.
                
            }
            // ミニ基本ヘッダ条件の場合.
            else {
                // 戻り値系プロトコル.
                
                pnt = 2 + 2 + 4 + 4 ;// ヘッダシフト.
                
            }
            
            // キューマネージャ名長取得.
            len = ConvertResourceParam.convertInt( pnt,telegram ) ;
            pnt += 4 ;
            
            bin = new byte[ len ] ;
            Resource.arraycopy( telegram,pnt,bin,0,len ) ;
            
            // キューマネージャ名取得.
            ret = new String( bin,BaseDef.UTF8 ) ;
            
       } catch( Exception e ) {
            ret = null ;
        } finally {
            bin = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * プロトコルオブジェクトから、電文タイプを取得.
     * <BR><BR>
     * プロトコルオブジェクトから、電文タイプを取得します.
     * <BR>
     * @param protocolObject チェック対象のプロトコルオブジェクトを設定します.
     * @return int 電文タイプが返されます.<BR>
     *             [PROTOCOL_DATA]が返された場合、プロトコルデータ([ProtocolData]オブジェクト)です.<BR>
     *             [PROTOCOL_HEARTBEAT]が返された場合、プロトコル正常戻り([ProtocolHeartBeat]オブジェクト)です.<BR>
     *             [PROTOCOL_SUCCESS]が返された場合、プロトコル正常戻り([ProtocolResultSuccess]オブジェクト)です.<BR>
     *             [PROTOCOL_ERROR]が返された場合、プロトコルエラー戻り値([ProtocolResultError]オブジェクト)です.<BR>
     *             [-1]が返された場合、プロトコルオブジェクトは不正です.
     */
    public static final int getProtocolObjectType( ProtocolObject protocolObject ) {
        
        int ret = -1 ;
        
        
        if( protocolObject == null ) {
            return -1 ;
        }
        
        switch( protocolObject.getType() ) {
            
            case ProtocolObject.PROTOCOL_DATA : ret = PROTOCOL_DATA ; break ;
            case ProtocolObject.PROTOCOL_HEARTBEAT : ret = PROTOCOL_HEARTBEAT ; break ;
            case ProtocolObject.PROTOCOL_SUCCESS : ret = PROTOCOL_SUCCESS ; break ;
            case ProtocolObject.PROTOCOL_ERROR : ret = PROTOCOL_ERROR ; break ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * プロトコルオブジェクトから電文を作成.
     * <BR><BR>
     * プロトコルオブジェクトから電文を作成します.
     * <BR>
     * @param in データプロトコルオブジェクトを設定します.
     * @param resType リソースタイプを設定します.<BR>
     *                [ProtocolData]オブジェクト以外では意味を持ちません.
     * @param cb32 暗号フラグを設定します.<BR>
     *             [ProtocolData]オブジェクト以外では意味を持ちません.
     * @return BinResource バイナリリソースが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BinResource getProtocolByTelegram(
        ProtocolObject in,ResourceType resType,boolean cb32 )
        throws InputException,AccessException {
        
        BinResource ret = null ;
        
        if( in == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            
            switch( ChannelProtocol.getProtocolObjectType( in ) ) {
                case PROTOCOL_DATA :
                    ret = ChannelProtocol.getProtocolDataByTelegram(
                        ( ProtocolData )in,resType,cb32
                    ) ;
                    break ;
                case PROTOCOL_HEARTBEAT :
                    ret = ChannelProtocol.getProtocolHeartBeatByTelegram(
                        ( ProtocolHeartBeat )in ) ;
                    break ;
                case PROTOCOL_SUCCESS :
                    ret = ChannelProtocol.getProtocolResultSuccessByTelegram(
                        ( ProtocolResultSuccess )in ) ;
                    break ;
                case PROTOCOL_ERROR :
                    ret = ChannelProtocol.getProtocolResultErrorByTelegram(
                        ( ProtocolResultError )in ) ;
                    break ;
                default :
                    throw new AccessException(
                        "プロトコルオブジェクトは不正です[" +
                        in.getClass().getName() + "]" ) ;
            }
            
        } catch( InputException ie ) {
            throw ie ;
        } catch( AccessException ae ) {
            throw ae ;
        }
        
        return ret ;
        
    }
    
    /**
     * 電文からプロトコルオブジェクトを生成.
     * <BR><BR>
     * 電文からプロトコルオブジェクトを生成します.
     * <BR>
     * @param resType データを確保するリソースタイプを設定します.
     * @param telegram 対象の電文を設定します.
     * @return ProtocolObject プロトコルオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final ProtocolObject getTelegramByProtocolObject(
        ResourceType resType,BinResource telegram )
        throws InputException,AccessException {
        
        ProtocolObject ret = null ;
        
        if( telegram == null || telegram.size() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            
            switch( ChannelProtocol.getTelegramType( telegram ) ) {
                case PROTOCOL_DATA :
                    ret = new ProtocolData() ;
                    ChannelProtocol.getTelegramByProtocolData(
                        ( ProtocolData )ret,resType,telegram
                    ) ;
                    break ;
                case PROTOCOL_HEARTBEAT :
                    ret = new ProtocolHeartBeat() ;
                    ChannelProtocol.getTelegramByProtocolHeartBeat(
                        ( ProtocolHeartBeat )ret,telegram ) ;
                    break ;
                case PROTOCOL_SUCCESS :
                    ret = new ProtocolResultSuccess() ;
                    ChannelProtocol.getTelegramByProtocolResultSuccess(
                        ( ProtocolResultSuccess )ret,telegram ) ;
                    break ;
                case PROTOCOL_ERROR :
                    ret = new ProtocolResultError() ;
                    ChannelProtocol.getTelegramByProtocolResultError(
                        ( ProtocolResultError )ret,telegram ) ;
                    break ;
                default :
                    throw new AccessException(
                        "プロトコルタイプは不正です[" +
                        telegram.get( 0 ) + "]" ) ;
            }
            
        } catch( InputException in ) {
            throw in ;
        } catch( AccessException ae ) {
            throw ae ;
        }
        
        return ret ;
    }
    
    /**
     * データ電文作成.
     * <BR><BR>
     * データ電文を作成します.
     * <BR>
     * @param in データプロトコルオブジェクトを設定します.
     * @param resType リソースタイプを設定します.
     * @param cb32 暗号フラグを設定します.
     * @return BinResource バイナリリソースが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BinResource getProtocolDataByTelegram(
        ProtocolData in,ResourceType resType,boolean cb32 )
        throws InputException,AccessException {
        
        int len ;
        int sz ;
        int pnt ;
        int cb32Pnt ;
        int dataLen ;
        int zFlg ;
        int step ;
        boolean gzipFlg = false ;
        
        String tmp = null ;
        byte[] header = null ;
        byte[] tmpBin = null ;
        byte[] cb32pub = null ;
        Object o = null ;
        CodeBase32 enc = null ;
        BinResource ret = null ;
        
        if(
            in == null ||
            ( o = in.getData() ) == null ||
            ( len = in.getLength() ) <= 0 ) {
            
            throw new InputException( "引数は不正です" ) ;
        }
        
        // リソースタイプが未設定.
        if( resType == null ) {
            resType = new ResourceType() ;
        }
        
        // 暗号コードをセット.
        cb32pub = CodeBase32.getPublicKey() ;
        
        // 暗号ありの場合.
        if( cb32 == true ) {
            // 暗号識別子のヘッダコードで基本ヘッダ情報を生成.
            header = ChannelProtocol.createHeader( HEADER_DATA_BY_CB32,cb32pub,in,-1 ) ;
            pnt = header.length ;
            cb32Pnt = pnt ;
        }
        else {
            // 非暗号識別子のヘッダコードで基本ヘッダ情報を生成.
            header = ChannelProtocol.createHeader( HEADER_DATA,cb32pub,in,-1 ) ;
            pnt = header.length ;
            cb32Pnt = -1 ;
        }
        
        try {
            
            // 戻り値となるリソース情報を作成.
            ret = Resource.createBinResource( resType,header ) ;
            
            // キーコードが存在しない場合は、長さ0の情報で処理
            if( ( tmp = in.getKeyCode() ) == null || tmp.trim().length() <= 0 ) {
                sz = 0 ;
            }
            // キーコードが存在する場合は、対象の長さ分の情報で処理.
            else {
                tmpBin = tmp.getBytes( BaseDef.UTF8 ) ;
                if( tmpBin == null || ( sz = tmpBin.length ) <= 0 ) {
                    sz = 0 ;
                }
            }
            
            // キーコード長をセット.
            ConvertResourceParam.convertInt( ret,pnt,sz ) ;
            pnt += 4 ;
            
            // キーコード情報が存在する場合は、その条件をセット.
            if( sz > 0 ) {
                ret.setBinary( pnt,tmpBin,0,sz ) ;
                pnt += sz ;
            }
            tmpBin = null ;
            
            // プロセスIDが存在しない場合は、長さ0の情報で処理
            if( ( tmp = in.getProcessId() ) == null || tmp.trim().length() <= 0 ) {
                sz = 0 ;
            }
            // プロセスIDが存在する場合は、対象の長さ分の情報で処理.
            else {
                tmpBin = tmp.getBytes( BaseDef.UTF8 ) ;
                if( tmpBin == null || ( sz = tmpBin.length ) <= 0 ) {
                    sz = 0 ;
                }
            }
            
            // プロセスID長をセット.
            ConvertResourceParam.convertInt( ret,pnt,sz ) ;
            pnt += 4 ;
            
            // プロセスID情報が存在する場合は、その条件をセット.
            if( sz > 0 ) {
                ret.setBinary( pnt,tmpBin,0,sz ) ;
                pnt += sz ;
            }
            tmpBin = null ;
            
            // プライオリティを設定.
            ConvertResourceParam.convertInt( ret,pnt,in.getPriority() ) ;
            pnt += 4 ;
            
            // expireを設定.
            ConvertResourceParam.convertInt( ret,pnt,in.getExpire() ) ;
            pnt += 4 ;
            
            // 圧縮フラグを乱数で取得.
            zFlg = UtilCom.random( GZIP_FLG )  ;
            zFlg += ( zFlg & 0x00000001 ) ;
            
            // 圧縮条件の場合.
            if(
                len >= GZIP_LENGTH ||
                (
                    ( o instanceof QueueMessage ) == true &&
                    ( gzipFlg = ( ( QueueMessage )o ).isGzipFlag() ) == true
                )
            ) {
                
                // 処理対象条件が既に圧縮されている場合.
                if( gzipFlg == true ) {
                    
                    QueueMessage qmsg = null ;
                    
                    // 電文長を設定.
                    ConvertResourceParam.convertInt( ret,pnt,len ) ;
                    pnt += 4 ;
                    
                    // 圧縮条件をOn.
                    // 奇数条件をセット.
                    ret.set( pnt,zFlg + 1 ) ;
                    pnt += 1 ;
                    
                    qmsg = ( QueueMessage )o ;
                    o = qmsg.getMessage() ;
                    
                    // 電文を設定.
                    if( o instanceof byte[] ) {
                        ret.setBinary( pnt,( byte[] )o,0,len ) ;
                    }
                    else {
                        ret.setBinary( pnt,( BinResource )o,0,len ) ;
                    }
                    
                    pnt += len ;
                    
                }
                // 処理対象が圧縮されていない場合.
                else {
                    
                    BinResource data = null ;
                    BinResource zDat = null ;
                    
                    // データ情報がQueueMessageオブジェクトの場合.
                    if( o instanceof QueueMessage ) {
                        o = ( ( QueueMessage )o ).getMessage() ;
                    }
                    
                    // 圧縮対象の電文を設定.
                    if( o instanceof byte[] ) {
                        data = Resource.createBinResource( resType,( byte[] )o ) ;
                    }
                    else {
                        data = ( BinResource )o ;
                    }
                    
                    // 圧縮処理.
                    zDat = Resource.createBinResource( resType,1 ) ;
                    GZIPBinary.getInstance().convertBinaryByGZIP( zDat,data ) ;
                    if( o instanceof byte[] ) {
                        data.clear() ;
                    }
                    data = null ;
                    
                    dataLen = zDat.size() ;
                    
                    // 電文を設定.
                    ret.setBinary( pnt+5,zDat,0,dataLen ) ;
                    zDat.clear() ;
                    zDat = null ;
                    
                    // 電文長を設定.
                    ConvertResourceParam.convertInt( ret,pnt,dataLen ) ;
                    pnt += 4 ;
                    
                    // 圧縮条件をOn.
                    // 奇数条件をセット.
                    ret.set( pnt,zFlg + 1 ) ;
                    pnt += 1 + dataLen ;
                    
                }
                
            }
            // 非圧縮条件の場合.
            else {
                
                // 電文長を設定.
                ConvertResourceParam.convertInt( ret,pnt,len ) ;
                pnt += 4 ;
                
                // 圧縮条件をOff.
                // 偶数条件をセット.
                ret.set( pnt,zFlg ) ;
                pnt += 1 ;
                
                // データ情報がQueueMessageオブジェクトの場合.
                if( o instanceof QueueMessage ) {
                    o = ( ( QueueMessage )o ).getMessage() ;
                }
                
                // 電文を設定.
                if( o instanceof byte[] ) {
                    ret.setBinary( pnt,( byte[] )o,0,len ) ;
                }
                else {
                    ret.setBinary( pnt,( BinResource )o,0,len ) ;
                }
                
                pnt += len ;
                
            }
            
            o = null ;
            
            // 暗号ありの場合.
            if( cb32Pnt != -1 ) {
                
                // 暗号オブジェクト生成.
                enc = new CodeBase32(
                    CodeBase32.getUserPasswdByKey(
                        in.getQueueManagerName(),in.getName()
                    )
                ) ;
                
                // 暗号処理.
                step = enc.encryption( cb32pub,ret,cb32Pnt,ret.size()-cb32Pnt ) ;
                
                // チェックコードを生成.
                byte chk = ( byte )( ChannelProtocol.getNoCb32StepCode(
                    ret,ret.size() ) & 0x000000ff ) ;
                
                // ステップコードを設定.
                ret.set( pnt,step ) ;
                
                // チェックデジッドを設定.
                ret.set( pnt+1,chk ) ;
                
            }
            // 暗号なしの場合.
            else {
                
                // チェックコードを生成.
                byte chk = ( byte )( ChannelProtocol.getNoCb32StepCode(
                    ret,ret.size() ) & 0x000000ff ) ;
                
                // ランダムな数値をセット.
                ret.set( pnt,UtilCom.random( 0x000000ff ) ) ;
                
                // チェックデジッドを設定.
                ret.set( pnt+1,chk ) ;
                
            }
            
        } catch( AccessException ae ) {
            if( ret != null ) {
                ret.clear() ;
            }
            ret = null ;
            throw ae ;
        } catch( Exception e ) {
            if( ret != null ) {
                ret.clear() ;
            }
            ret = null ;
            throw new AccessException( e ) ;
        } finally {
            
            tmp = null ;
            header = null ;
            tmpBin = null ;
            cb32pub = null ;
            o = null ;
            enc = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * データ電文からProtocolDataオブジェクトを生成.
     * <BR><BR>
     * データ電文からProtocolDataオブジェクトを生成します.
     * <BR>
     * @param out データを受け取るProtocolDataオブジェクトを設定します.
     * @param resType データを確保するリソースタイプを設定します.
     * @param telegram 対象の電文を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void getTelegramByProtocolData(
        ProtocolData out,ResourceType resType,BinResource telegram )
        throws InputException,AccessException {
        
        int len ;
        int pnt ;
        int zFlg ;
        int chDg ;
        
        byte[] bin = null ;
        BinResource data = null ;
        
        if( out == null || telegram == null || telegram.size() <= 0 ) {
            
            throw new InputException( "引数は不正です" ) ;
            
        }
        else if( telegram.get( 0 ) != PROTOCOL_DATA ) {
            
            throw new InputException(
                "指定電文はプロトコルデータ電文ではありません[src:" + telegram.get( 0 ) +
                " dest:" + PROTOCOL_DATA + "]" ) ;
            
        }
        
        // チェックデジッドを計算.
        chDg = ChannelProtocol.getNoCb32StepCode( telegram,telegram.size()-2 ) ;
        
        // チェックデジッドチェック.
        if( telegram.get( telegram.size()-1 ) != chDg ) {
            throw new AccessException( "チェックデジットが一致しません[src:" +
                telegram.get( telegram.size()-1 ) + " dest:" + chDg + "]" ) ;
        }
        
        try {
            
            // 基本ヘッダを取得.
            pnt = ChannelProtocol.convertHeader( out,telegram ) ;
            
            // 暗号電文である場合.
            if( telegram.get( 1 ) == HEAD_CB32 ) {
                
                int step ;
                byte[] cb32pub = null ;
                CodeBase32 ana = null ;
                
                // パブリックキーを取得.
                cb32pub = new byte[ 32 ] ;
                Resource.arraycopy( telegram,4,cb32pub,0,cb32pub.length ) ;
                
                // ステップコードを取得.
                step = telegram.get( telegram.size()-2 ) ;
                
                // 暗号オブジェクト生成.
                ana = new CodeBase32(
                    CodeBase32.getUserPasswdByKey(
                        out.getQueueManagerName(),out.getName()
                    )
                ) ;
                
                // 暗号解析.
                ana.analysis( cb32pub,step,telegram,pnt,telegram.size()-pnt ) ;
                
                cb32pub = null ;
                ana = null ;
                
            }
            
            // キーコード長を取得.
            len = ConvertResourceParam.convertInt( pnt,telegram ) ;
            pnt += 4 ;
            
            // キーコードが設定されている場合.
            if( len > 0 ) {
                
                bin = new byte[ len ] ;
                Resource.arraycopy( telegram,pnt,bin,0,len ) ;
                
                out.setKeyCode( new String( bin,BaseDef.UTF8 ) ) ;
                bin = null ;
                pnt += len ;
                
            }
            
            // プロセスID長を取得.
            len = ConvertResourceParam.convertInt( pnt,telegram ) ;
            pnt += 4 ;
            
            // プロセスIDが設定されている場合.
            if( len > 0 ) {
                
                bin = new byte[ len ] ;
                Resource.arraycopy( telegram,pnt,bin,0,len ) ;
                
                out.setProcessId( new String( bin,BaseDef.UTF8 ) ) ;
                bin = null ;
                pnt += len ;
                
            }
            
            // プライオリティを設定.
            out.setPriority( ConvertResourceParam.convertInt( pnt,telegram ) ) ;
            pnt += 4 ;
            
            // Expireを設定.
            out.setExpire( ConvertResourceParam.convertInt( pnt,telegram ) ) ;
            pnt += 4 ;
            
            // 電文長を取得.
            len = ConvertResourceParam.convertInt( pnt,telegram ) ;
            pnt += 4 ;
            
            // 圧縮フラグを取得.
            zFlg = telegram.get( pnt ) ;
            pnt += 1 ;
            
            // 圧縮されている場合.
            if( ( zFlg & 0x00000001 ) == 0x00000001 ) {
                BinResource zDat = null ;
                
                zDat = Resource.createBinResource( resType,1 ) ;
                GZIPBinary.getInstance().convertGZIPByBinary( zDat,telegram,pnt,len ) ;
                
                out.setData( zDat ) ;
                zDat = null ;
                
                pnt += len ;
                
            }
            // 非圧縮の場合.
            else {
                
                data = Resource.createBinResource( resType,len ) ;
                Resource.arraycopy( telegram,pnt,data,0,len ) ;
                
                out.setData( data ) ;
                data = null ;
                
                pnt += len ;
                
            }
            
        } catch( AccessException ae ) {
            out.clear() ;
            throw ae ;
        } catch( Exception e ) {
            out.clear() ;
            throw new AccessException( e ) ;
        } finally {
            
            bin = null ;
            data = null ;
            
        }
        
    }
    
    /**
     * ハートビートの電文作成.
     * <BR><BR>
     * ハートビートの電文を作成します.
     * <BR>
     * @param in ハートビートプロトコルオブジェクトを設定します.
     * @return BinResource バイナリリソースが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BinResource getProtocolHeartBeatByTelegram( ProtocolHeartBeat in )
        throws InputException,AccessException {
        
        BinResource ret = null ;
        
        if( in == null ) {
            throw new InputException ( "引数は不正です" ) ;
        }
        
        try {
            ret = Resource.createBinResource(
                null,ChannelProtocol.createHeader( HEADER_HEARTBEAT,null,in,in.getReturnType() )
            ) ;
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * ハートビートの電文からProtocolHeartBeatオブジェクトを生成.
     * <BR><BR>
     * ハートビートの電文からProtocolHeartBeatオブジェクトを生成します.
     * <BR>
     * @param out ハートビートを受け取るProtocolHeartオブジェクトを設定します.
     * @param telegram 対象の電文を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void getTelegramByProtocolHeartBeat( ProtocolHeartBeat out,BinResource telegram )
        throws InputException,AccessException {
        
        if( out == null || telegram == null || telegram.size() <= 0 ) {
            
            throw new InputException( "引数は不正です" ) ;
            
        }
        else if( telegram.get( 0 ) != PROTOCOL_HEARTBEAT ) {
            
            throw new InputException(
                "指定電文はプロトコル正常戻り電文ではありません[src:" + telegram.get( 0 ) +
                " dest:" + PROTOCOL_HEARTBEAT + "]" ) ;
                
        }
        
        try {
            ChannelProtocol.convertHeader( out,telegram ) ;
        } catch( AccessException ae ) {
            out.clear() ;
            throw ae ;
        } catch( Exception e ) {
            out.clear() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * 正常戻り値の電文作成.
     * <BR><BR>
     * 正常戻り値の電文を作成します.
     * <BR>
     * @param in 正常戻り値プロトコルオブジェクトを設定します.
     * @return BinResource バイナリリソースが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BinResource getProtocolResultSuccessByTelegram( ProtocolResultSuccess in )
        throws InputException,AccessException {
        
        BinResource ret = null ;
        
        if( in == null ) {
            throw new InputException ( "引数は不正です" ) ;
        }
        
        try {
            ret = Resource.createBinResource(
                null,ChannelProtocol.createHeader( HEADER_SUCCESS,null,in,in.getReturnType() )
            ) ;
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 正常戻り値の電文からProtocolResultSuccessオブジェクトを生成.
     * <BR><BR>
     * 正常戻り値の電文からProtocolResultSuccessオブジェクトを生成します.
     * <BR>
     * @param out 正常戻り値を受け取るProtocolResultSuccessオブジェクトを設定します.
     * @param telegram 対象の電文を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void getTelegramByProtocolResultSuccess( ProtocolResultSuccess out,BinResource telegram )
        throws InputException,AccessException {
        
        if( out == null || telegram == null || telegram.size() <= 0 ) {
            
            if( out == null ) {
                throw new InputException( "out引数は不正です" ) ;
            }
            throw new InputException( "telegram引数は不正です" ) ;
            
        }
        else if( telegram.get( 0 ) != PROTOCOL_SUCCESS ) {
            
            throw new InputException(
                "指定電文はプロトコル正常戻り電文ではありません[src:" + telegram.get( 0 ) +
                " dest:" + PROTOCOL_SUCCESS + "]" ) ;
                
        }
        
        try {
            ChannelProtocol.convertHeader( out,telegram ) ;
        } catch( AccessException ae ) {
            out.clear() ;
            throw ae ;
        } catch( Exception e ) {
            out.clear() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * エラー戻り値の電文作成.
     * <BR><BR>
     * エラー戻り値の電文を作成します.
     * <BR>
     * @param in エラー戻り値プロトコルオブジェクトを設定します.
     * @return BinResource バイナリリソースが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BinResource getProtocolResultErrorByTelegram( ProtocolResultError in )
        throws InputException,AccessException {
        
        int pnt ;
        BinResource ret = null ;
        
        if( in == null ) {
            throw new InputException ( "引数は不正です" ) ;
        }
        
        try {
            
            ret = Resource.createBinResource(
                null,ChannelProtocol.createHeader( HEADER_ERROR,null,in,in.getReturnType() )
            ) ;
            
            pnt = ret.size() ;
            
            ConvertResourceParam.convertInt( ret,pnt,in.getErrorCode() ) ;
            pnt += 4 ;
            
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * エラー戻り値の電文からProtocolResultErrorオブジェクトを生成.
     * <BR><BR>
     * エラー戻り値の電文からProtocolResultErrorオブジェクトを生成します.
     * <BR>
     * @param out エラー戻り値を受け取るProtocolResultErrorオブジェクトを設定します.
     * @param telegram 対象の電文を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void getTelegramByProtocolResultError( ProtocolResultError out,BinResource telegram )
        throws InputException,AccessException {
        
        int pnt ;
        
        if( out == null || telegram == null || telegram.size() <= 0 ) {
            
            throw new InputException( "引数は不正です" ) ;
            
        }
        else if( telegram.get( 0 ) != PROTOCOL_ERROR ) {
            
            throw new InputException(
                "指定電文はプロトコルエラー戻り電文ではありません[src:" + telegram.get( 0 ) +
                " dest:" + PROTOCOL_ERROR + "]" ) ;
                
        }
        
        try {
            
            pnt = ChannelProtocol.convertHeader( out,telegram ) ;
            out.setErrorCode( ConvertResourceParam.convertInt( pnt,telegram ) ) ;
            pnt += 4 ;
            
        } catch( AccessException ae ) {
            out.clear() ;
            throw ae ;
        } catch( Exception e ) {
            out.clear() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    
    
    /**
     * ヘッダ内容を設定.
     */
    private static final byte[] createHeader( byte[] header,byte[] cb32,ProtocolObject o,int returnType )
        throws AccessException {
        
        int pnt ;
        
        byte[] qman = null ;
        byte[] srcQman = null ;
        byte[] name = null ;
        byte[] channel = null ;
        byte[] ret = null ;
        
        if(
            o.getQueueManagerName() == null || o.getQueueManagerName().length() <= 0 ||
            o.getName() == null || o.getName().length() <= 0 ||
            o.getChannelName() == null || o.getChannelName().length() <= 0
        ) {
            throw new AccessException(
                "必須条件[name,queueManagerName,channelName]が設定されていません" ) ;
        }
        
        // 処理名、キューマネージャ名をバイナリ変換.
        try {
            qman = o.getQueueManagerName().getBytes( BaseDef.UTF8 ) ;
            name = o.getName().getBytes( BaseDef.UTF8 ) ;
            channel = o.getChannelName().getBytes( BaseDef.UTF8 ) ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        // フル基本ヘッダ条件の場合.
        if( ( header[ 0 ] & 0x000000ff ) == PROTOCOL_DATA ) {
            
            try {
                srcQman = ( ( ProtocolData )o ).
                    getSrcQueueManagerName().getBytes( BaseDef.UTF8 ) ;
            } catch( Exception e ) {
                throw new AccessException( e ) ;
            }
            
            // ヘッダバイナリを生成.
            ret = new byte[
                2 +                                 // 基本ヘッダ.
                2 +                                 // 受信ポート番号.
                4 +                                 // チャネルID.
                NetDef.MAC_ADDRESS_LENGTH +         // マックアドレス.
                32 +                                // cb32暗号コード.
                
                4 +                                 // キューマネージャ名長.
                qman.length +                       // キューマネージャ名.
                4 +                                 // 送信元キューマネージャ名長.
                srcQman.length +                    // 送信元キューマネージャ名.
                4 +                                 // 処理名長.
                name.length +                       // 処理名.
                4 +                                 // チャネル名長.
                channel.length +                    // チャネル名.
                8                                   // 電文ID.
            ] ;
            
            // ヘッダ識別子をセット.
            pnt = 0 ;
            System.arraycopy( header,0,ret,pnt,2 ) ;
            pnt += 2 ;
            
            // 受信ポート番号をセット.
            ConvertParam.convertShort(
                ret,pnt,( short )( o.getReceivePort() & 0x0000ffff ) ) ;
            pnt += 2 ;
            
            // チャネルIDをセット.
            ConvertParam.convertInt(
                ret,pnt,o.getChannelId() ) ;
            pnt += 4 ;
            
            // MACアドレスをセット.
            byte[] macBin = ( ( ProtocolData )o ).getMacAddress() ;
            if( macBin != null && macBin.length == NetDef.MAC_ADDRESS_LENGTH ) {
                System.arraycopy( ( ( ProtocolData )o ).getMacAddress(),0,ret,pnt,6 ) ;
            }
            pnt += NetDef.MAC_ADDRESS_LENGTH ;
            
            // cb32暗号コードをセット.
            System.arraycopy( cb32,0,ret,pnt,32 ) ;
            pnt += 32 ;
            
        }
        // ミニ基本ヘッダ条件の場合.
        else {
            
            // ヘッダバイナリを生成.
            ret = new byte[
                2 +                                 // 基本ヘッダ.
                2 +                                 // 受信ポート番号.
                4 +                                 // チャネルID.
                4 +                                 // 処理結果タイプ.
                
                4 +                                 // キューマネージャ名長.
                qman.length +                       // キューマネージャ名.
                4 +                                 // 処理名長.
                name.length +                       // 処理名.
                4 +                                 // チャネル名長.
                channel.length +                    // チャネル名.
                8                                   // 電文ID.
            ] ;
            
            // ヘッダ識別子をセット.
            pnt = 0 ;
            System.arraycopy( header,0,ret,pnt,2 ) ;
            pnt += 2 ;
            
            // 受信ポート番号をセット.
            ConvertParam.convertShort(
                ret,pnt,( short )( o.getReceivePort() & 0x0000ffff ) ) ;
            pnt += 2 ;
            
            // チャネルIDをセット.
            ConvertParam.convertInt(
                ret,pnt,o.getChannelId() ) ;
            pnt += 4 ;
            
            // 処理結果タイプをセット.
            ConvertParam.convertInt( ret,pnt,returnType ) ;
            pnt += 4 ;
            
        }
        
        // キューマネージャ名長をセット.
        ConvertParam.convertInt( ret,pnt,qman.length ) ;
        pnt += 4 ;
        
        // キューマネージャ名をセット.
        System.arraycopy( qman,0,ret,pnt,qman.length ) ;
        pnt += qman.length ;
        
        // フル基本ヘッダ条件の場合.
        if( ( header[ 0 ] & 0x000000ff ) == PROTOCOL_DATA ) {
            
            // 送信元キューマネージャ名長をセット.
            ConvertParam.convertInt( ret,pnt,srcQman.length ) ;
            pnt += 4 ;
            
            // 送信元キューマネージャ名をセット.
            System.arraycopy( srcQman,0,ret,pnt,srcQman.length ) ;
            pnt += srcQman.length ;
            
        }
        
        // 処理名長をセット.
        ConvertParam.convertInt( ret,pnt,name.length ) ;
        pnt += 4 ;
        
        // 処理名をセット.
        System.arraycopy( name,0,ret,pnt,name.length ) ;
        pnt += name.length ;
        
        // チャネル名長をセット.
        ConvertParam.convertInt( ret,pnt,channel.length ) ;
        pnt += 4 ;
        
        // チャネル名をセット.
        System.arraycopy( channel,0,ret,pnt,channel.length ) ;
        pnt += channel.length ;
        
        // 電文IDをセット.
        ConvertParam.convertLong( ret,pnt,o.getId() ) ;
        pnt += 8 ;
        
        return ret ;
    }
    
    /**
     * ヘッダ情報をコンバート.
     */
    private static final int convertHeader( ProtocolObject out,BinResource telegram )
        throws AccessException {
        
        int pnt ;
        int len ;
        int ret ;
        
        byte[] bin = null ;
        
        try {
            
            out.clear() ;
            
            // フル基本ヘッダ条件の場合.
            if( telegram.get( 0 ) == PROTOCOL_DATA ) {
                // データプロトコル.
                
                pnt = 2 ;// ヘッダ.
                
                // 受信ポート番号を取得.
                out.setReceivePort(
                    ( int )( ConvertResourceParam.convertShort( pnt,telegram )
                        & 0x0000ffff ) ) ;
                pnt += 2 ;
                
                // チャネルIDを取得.
                out.setChannelId(
                    ConvertResourceParam.convertInt( pnt,telegram ) ) ;
                pnt += 4 ;
                
                // マックアドレスを取得.
                bin = new byte[ NetDef.MAC_ADDRESS_LENGTH ] ;
                Resource.arraycopy( telegram,pnt,bin,0,NetDef.MAC_ADDRESS_LENGTH  ) ;
                ( ( ProtocolData )out ).setMacAddress( bin ) ;
                pnt += NetDef.MAC_ADDRESS_LENGTH  + 32 ;
                
            }
            // ミニ基本ヘッダ条件の場合.
            else {
                // 戻り値系プロトコル.
                
                pnt = 2 ;// ヘッダシフト.
                
                // 受信ポート番号を取得.
                out.setReceivePort(
                    ( int )( ConvertResourceParam.convertShort( pnt,telegram ) & 0x0000ffff ) ) ;
                pnt += 2 ;
                
                // チャネルIDを取得.
                out.setChannelId(
                    ConvertResourceParam.convertInt( pnt,telegram ) ) ;
                pnt += 4 ;
                
                // 戻りタイプを取得.
                int retType = ConvertResourceParam.convertInt( pnt,telegram ) ;
                pnt += 4 ;
                
                // 戻り電文系の場合のみ、設定.
                if( out instanceof ProtocolResultError ) {
                    ( ( ProtocolResultError )out ).setReturnType( retType ) ;
                }
                else if( out instanceof ProtocolResultSuccess ) {
                    ( ( ProtocolResultSuccess )out ).setReturnType( retType ) ;
                }
                
            }
            
            // キューマネージャ名取得.
            len = ConvertResourceParam.convertInt( pnt,telegram ) ;
            pnt += 4 ;
            
            bin = new byte[ len ] ;
            Resource.arraycopy( telegram,pnt,bin,0,len ) ;
            
            out.setQueueManagerName( new String( bin,BaseDef.UTF8 ) ) ;
            bin = null ;
            pnt += len ;
            
            // フル基本ヘッダ条件の場合.
            if( telegram.get( 0 ) == PROTOCOL_DATA ) {
                
                // 送信元キューマネージャ名取得.
                len = ConvertResourceParam.convertInt( pnt,telegram ) ;
                pnt += 4 ;
                
                bin = new byte[ len ] ;
                Resource.arraycopy( telegram,pnt,bin,0,len ) ;
                
                ( ( ProtocolData )out ).setSrcQueueManagerName(
                    new String( bin,BaseDef.UTF8 ) ) ;
                bin = null ;
                pnt += len ;
                
            }
            
            // 処理名取得.
            len = ConvertResourceParam.convertInt( pnt,telegram ) ;
            pnt += 4 ;
            
            bin = new byte[ len ] ;
            Resource.arraycopy( telegram,pnt,bin,0,len ) ;
            
            out.setName( new String( bin,BaseDef.UTF8 ) ) ;
            bin = null ;
            pnt += len ;
            
            // チャネル名取得.
            len = ConvertResourceParam.convertInt( pnt,telegram ) ;
            pnt += 4 ;
            
            bin = new byte[ len ] ;
            Resource.arraycopy( telegram,pnt,bin,0,len ) ;
            
            out.setChannelName( new String( bin,BaseDef.UTF8 ) ) ;
            bin = null ;
            pnt += len ;
            
            // 電文Idを取得.
            out.setId( ConvertResourceParam.convertLong( pnt,telegram ) ) ;
            pnt += 8 ;
            
            ret = pnt ;
            
            if(
                out.getQueueManagerName() == null || out.getQueueManagerName().length() <= 0 ||
                out.getName() == null || out.getName().length() <= 0 ||
                out.getChannelName() == null || out.getChannelName().length() <= 0
            ) {
                throw new AccessException(
                    "必須条件[queueName,queueManagerName,channelName]が設定されていません" ) ;
            }
            
        } catch( AccessException ae ) {
            out.clear() ;
            throw ae ;
        } catch( Exception e ) {
            out.clear() ;
            throw new AccessException( e ) ;
        } finally {
            bin = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 非cb32時のstepコード生成.
     */
    private static final int getNoCb32StepCode( BinResource bin,int len )
        throws AccessException {
        
        int i ;
        int dat ;
        int ret ;
        
        BufferedBinResource buf = null ;
        
        try {
            
            buf = new BufferedBinResource( bin ) ;
            
            for( i = 0,ret = len ; i < len ; i ++ ) {
                dat = buf.get( i ) ;
                ret = ( ret + dat + ( dat & 0x00000001 ) ) & 0x000000ff ;
            }
            
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        } finally {
            if( buf != null ) {
                buf.clear() ;
            }
            buf = null ;
        }
        
        return ret ;
        
    }
    
}

