/*
 * @(#)ExecutionDataSwitch.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.receive ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.NumberTable;
import org.maachang.queue.access.status.ChannelStatus;
import org.maachang.queue.access.status.QueueManagerStatus;
import org.maachang.queue.access.status.QueueStatus;
import org.maachang.queue.main.channel.Channel;
import org.maachang.queue.main.channel.ChannelFactory;
import org.maachang.queue.main.channel.LastReceiveIdChild;
import org.maachang.queue.main.channel.LastReceiveIdManage;
import org.maachang.queue.main.channel.SendChannel;
import org.maachang.queue.main.channel.protocol.ChannelProtocol;
import org.maachang.queue.main.channel.protocol.ProtocolData;
import org.maachang.queue.main.channel.protocol.ProtocolObject;
import org.maachang.queue.main.channel.service.ChannelServiceError;
import org.maachang.queue.main.channel.service.ChannelServiceUtil;
import org.maachang.queue.main.channel.service.receive.core.CoreReceiveChild;
import org.maachang.queue.main.manager.BaseQueueManager;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.queue.MessageOverException;
import org.maachang.queue.main.queue.QueueDef;
import org.maachang.queue.main.queue.QueueFullException;
import org.maachang.queue.main.queue.base.BaseQueue;
import org.maachang.queue.main.queue.base.BaseQueueFactory;

/**
 * 受信電文[ProtocolData]処理実装.
 *
 * @version 2006/12/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ExecutionDataSwitch implements ExecutionSwitch {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ExecutionDataSwitch.class ) ;
    
    /**
     * データ電文プーリング.
     */
    private final NumberTable pool = new NumberTable() ;
    
    /**
     * プロトコルタイプを取得.
     * <BR><BR>
     * プロトコルタイプを取得します.
     * <BR>
     * @return int プロトコルタイプが返されます.<BR>
     *             [ProtocolObject.PROTOCOL_DATA]が返された場合、データオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_HEARTBEAT]が返された場合、ハートビートオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_SUCCESS]が返された場合、正常オブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_ERROR]が返された場合ｍエラーオブジェクトです.
     */
    public int getType() {
        return ProtocolObject.PROTOCOL_DATA ;
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * @param threadNum スレッド項番が設定されます.
     * @param child 処理対象の要素が設定されます.
     * @param sync 同期オブジェクト.
     */
    public void execution( int threadNum,CoreReceiveChild child,Synchronized sync ) {
        
        try {
            
            int error = -1 ;
            BaseQueue queue = null ;
            ResourceType resType = null ;
            
            // プロトコルをプールオブジェクトから取得.
            PoolProtocol poolData = ( PoolProtocol )pool.get( threadNum ) ;
            
            // 情報が存在しない場合は、生成して、プールに設定.
            if( poolData == null ) {
                poolData = new PoolProtocol( new ProtocolData() ) ;
                pool.add( threadNum,poolData ) ;
            }
            
            // データ情報を取得.
            ProtocolData dataProtocol = ( ProtocolData )poolData.getBaseProtocol() ;
            
            BinResource telegram = child.getBinResource() ;
            
            // キューマネージャ名を取得.
            String queueManagerName = ChannelProtocol.getQueueManager( telegram ) ;
            
            // 対象キューマネージャを取得.
            QueueManager man = QueueManagerFactory.get( queueManagerName ) ;
            
            // 対象キューマネージャが存在する場合は、キューマネージャの
            // リソースタイプを取得.
            if( man != null ) {
                resType = ( ( BaseQueueManager )man ).getResourceType() ;
            }
            
            // 受信電文バイナリから、データオブジェクトに変換.
            ChannelProtocol.getTelegramByProtocolData( dataProtocol,resType,telegram ) ;
            
            // 受信チャネルオブジェクトを取得.
            Channel channel = ChannelFactory.get(
                Channel.TYPE_RECEIVE,dataProtocol.getChannelName() ) ;
            
            // チャネルオブジェクトが存在しない場合.
            if( channel == null || channel.isChannel() == false ||
                ( channel instanceof SendChannel ) == true ) {
                error = ChannelServiceError.NOT_CHANNEL_NAME ;
            }
            // チャネルステータスが正常以外の場合.
            else if( channel.getState() != ChannelStatus.STATE_SUCCESS ) {
                error = ChannelServiceError.CHANNEL_STATE_BY_NOT_SUCCESS ;
            }
            // キューマネージャオブジェクトが存在しない場合.
            else if( man == null ) {
                error = ChannelServiceError.NOT_QUEUE_MANAGER_NAME ;
            }
            // キューマネージャステータスが「正常」以外の場合.
            else if( man.getState() != QueueManagerStatus.STATE_SUCCESS ) {
                error = ChannelServiceError.QUEUE_MANAGER_STATE_BY_NOT_SUCCESS ;
            }
            // キューオブジェクトが存在しない場合.
            else if( ( queue = BaseQueueFactory.get( QueueDef.TYPE_RECEIVE,
                queueManagerName,dataProtocol.getName() ) ) == null ) {
                error = ChannelServiceError.NOT_QUEUE_NAME ;
            }
            // キューステータスが「キュー満杯」を示す場合.
            else if( queue.getState().getState() == QueueStatus.STATE_FULL ) {
                error = ChannelServiceError.QUEUE_STATE_BY_FULL ;
            }
            // キューステータスが「正常」、「警告」以外の場合.
            else if( queue.getState().getState() != QueueStatus.STATE_SUCCESS  &&
                queue.getState().getState() != QueueStatus.STATE_WARNING ) {
                error = ChannelServiceError.QUEUE_STATE_BY_NOT_SUCCESS ;
            }
            // キューステータスが正常の場合.
            else {
                
                // 対象の受信IDを取得.
                long receiveId = dataProtocol.getId() ;
                
                // 受信キューID管理オブジェクトを取得.
                LastReceiveIdChild idChild  = null ;
                boolean childFlag = false ;
                
                // 受信可能であるかチェック.
                synchronized( channel ) {
                    
                    // チャネルに対する2度受信防止管理オブジェクトを取得.
                    LastReceiveIdManage idMan = channel.getLastReceiveIdManage() ;
                    
                    // 対象IDが受信可能である場合は、受信キューID管理オブジェクトを取得.
                    idChild = idMan.getReceiveIdChild(
                        queueManagerName,queue.getState().getName(),
                        dataProtocol.getMacAddressByLong(),receiveId ) ;
                    
                    // 対象受信IDが有効である場合.
                    if( idChild != null ) {
                        childFlag = true ;
                    }
                }
                
                // 受信ターゲットの場合.
                if( childFlag == true ) {
                    
                    // expire値を取得.
                    long exp = -1L ;
                    if( dataProtocol.getExpire() > 0 ) {
                        exp = ( long )( dataProtocol.getExpire() &
                            0x000000007fffffffL ) ;
                    }
                    
                    // 受信データをキューに設定.
                    try {
                        
                        // キューに設定するデータを確認.
                        queue.isPut( dataProtocol.getLength() ) ;
                        
                        // 先に正常結果電文を送信.
                        ChannelServiceUtil.sendReturnSuccess( child.getCb32Word(),
                            child.getAddress(),child.getPort(),
                            child.getConnectName(),dataProtocol,
                            poolData.getProtocolResultSuccess() ) ;
                        
                        // キュー情報にデータセット.
                        queue.putByAutoCommit( dataProtocol.getKeyCode(),
                            dataProtocol.getProcessId(),
                            dataProtocol.getData(),
                            dataProtocol.getPriority(),
                            exp ) ;
                        
                    } catch( MessageOverException mv ) {
                        
                        // 受信メッセージ長がキュー指定長より大きい場合.
                        error = ChannelServiceError.QUEUE_STATE_BY_MAX_LENGTH ;
                        
                    } catch( QueueFullException qf ) {
                        
                        // キュー満杯を検知.
                        error = ChannelServiceError.QUEUE_STATE_BY_FULL ;
                        
                    }
                    
                }
            }
            
            // キュー関連のエラーが存在する場合.
            if( error != -1 ) {
                
                // エラー電文を送信.
                ChannelServiceUtil.sendReturnError( child.getCb32Word(),
                    dataProtocol.getSrcQueueManagerName(),
                    child.getAddress(),child.getPort(),
                    child.getConnectName(),dataProtocol,
                    poolData.getProtocolResultError(),error ) ;
                
            }
            // 正常に終了した場合.
            else {
                
                // 正常結果を受信側に返信.
                ChannelServiceUtil.sendReturnSuccess( child.getCb32Word(),
                    dataProtocol.getSrcQueueManagerName(),
                    child.getAddress(),child.getPort(),
                    child.getConnectName(),dataProtocol,
                    poolData.getProtocolResultSuccess() ) ;
                
            }
            
        } catch( NullPointerException nul ) {
            throw nul ;
        } catch( OutOfMemoryError me ) {
            LOG.error( "[(Receive)Data]OutOfMemoryError",me ) ;
        } catch( Exception e ) {
            LOG.warn( "[(Receive)Data]受信処理時に例外",e ) ;
        }
        
    }
    
}

