/*
 * @(#)ExecutionHeartBeatSwitch.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.receive ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.NumberTable;
import org.maachang.queue.access.status.ChannelStatus;
import org.maachang.queue.access.status.QueueManagerStatus;
import org.maachang.queue.main.channel.Channel;
import org.maachang.queue.main.channel.ChannelFactory;
import org.maachang.queue.main.channel.SendChannel;
import org.maachang.queue.main.channel.protocol.ChannelProtocol;
import org.maachang.queue.main.channel.protocol.ProtocolHeartBeat;
import org.maachang.queue.main.channel.protocol.ProtocolObject;
import org.maachang.queue.main.channel.service.ChannelServiceError;
import org.maachang.queue.main.channel.service.ChannelServiceUtil;
import org.maachang.queue.main.channel.service.receive.core.CoreReceiveChild;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;

/**
 * 受信電文[ProtocolHeartBeat]処理実装.
 *
 * @version 2006/12/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ExecutionHeartBeatSwitch implements ExecutionSwitch {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ExecutionHeartBeatSwitch.class ) ;
    
    /**
     *  ハートビートプーリング.
     */
    private final NumberTable pool = new NumberTable() ;
    
    /**
     * コンストラクタ.
     */
    public ExecutionHeartBeatSwitch() {
    }
    
    /**
     * プロトコルタイプを取得.
     * <BR><BR>
     * プロトコルタイプを取得します.
     * <BR>
     * @return int プロトコルタイプが返されます.<BR>
     *             [ProtocolObject.PROTOCOL_DATA]が返された場合、データオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_HEARTBEAT]が返された場合、ハートビートオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_SUCCESS]が返された場合、正常オブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_ERROR]が返された場合ｍエラーオブジェクトです.
     */
    public int getType() {
        return ProtocolObject.PROTOCOL_HEARTBEAT ;
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * @param threadNum スレッド項番が設定されます.
     * @param child 処理対象の要素が設定されます.
     * @param sync 同期オブジェクト.
     */
    public void execution( int threadNum,CoreReceiveChild child,Synchronized sync ) {
        
        try {
            
            int error = -1 ;
            
            // プロトコルをプールオブジェクトから取得.
            PoolProtocol poolHeartBeat = ( PoolProtocol )pool.get( threadNum ) ;
            
            // 情報が存在しない場合は、生成して、プールに設定.
            if( poolHeartBeat == null ) {
                poolHeartBeat = new PoolProtocol( new ProtocolHeartBeat() ) ;
                pool.add( threadNum,poolHeartBeat ) ;
            }
            
            // ハートビート情報を取得.
            ProtocolHeartBeat heartBeat = ( ProtocolHeartBeat )poolHeartBeat.getBaseProtocol() ;
            
            // 受信電文バイナリから、ハートビートオブジェクトに変換.
            ChannelProtocol.getTelegramByProtocolHeartBeat( heartBeat,child.getBinResource() ) ;
            
            // 受信チャネルを取得.
            Channel channel = ChannelFactory.get( Channel.TYPE_RECEIVE,
                heartBeat.getChannelName() ) ;
            
            ///////////////////
            // エラーチェック.
            ///////////////////
            
            // 受信チャネルが存在しない場合.
            if( channel == null || channel.isChannel() == false ||
                ( channel instanceof SendChannel ) == true ) {
                
                // 受信チャネル非存在エラー.
                error = ChannelServiceError.NOT_CHANNEL_NAME ;
                
            }
            // チャネルステータスが「正常」「起動中」以外の場合.
            else if( channel.getState() != ChannelStatus.STATE_SUCCESS &&
                channel.getState() != ChannelStatus.STATE_STARTUP ) {
                
                // 受信チャネルステータスエラー.
                error = ChannelServiceError.CHANNEL_STATE_BY_NOT_SUCCESS ;
                
            }
            // キューマネージャ系チェック.
            else {
                
                // 対象キューマネージャを取得.
                QueueManager manager = QueueManagerFactory.get(
                    heartBeat.getQueueManagerName() ) ;
                
                // キューマネージャ名は存在しない場合.
                if( manager == null || manager.isQueueManager() == false ) {
                    
                    // キューマネージャは存在しない.
                    error = ChannelServiceError.NOT_QUEUE_MANAGER_NAME ;
                    
                }
                // キューマネージャステータスが「正常」「起動中」「エラー」以外の場合.
                else if( manager.getState() != QueueManagerStatus.STATE_SUCCESS &&
                    manager.getState() != QueueManagerStatus.STATE_STARTUP &&
                    manager.getState() != QueueManagerStatus.STATE_ERROR ) {
                    
                    // キューマネージャステータスエラー.
                    error = ChannelServiceError.QUEUE_MANAGER_STATE_BY_NOT_SUCCESS ;
                    
                }
                
            }
            
            // エラーが存在する場合.
            if( error != -1 ) {
                
                // エラー電文で処理.
                ChannelServiceUtil.sendReturnError( child.getCb32Word(),
                    child.getAddress(),child.getPort(),
                    child.getConnectName(),heartBeat,poolHeartBeat.getProtocolResultError(),error ) ;
                
            }
            // エラーが存在しない場合.
            else {
                
                // 正常電文で処理.
                ChannelServiceUtil.sendReturnSuccess( child.getCb32Word(),
                    child.getAddress(),child.getPort(),
                    child.getConnectName(),heartBeat,poolHeartBeat.getProtocolResultSuccess() ) ;
                
            }
            
            
        } catch( NullPointerException nul ) {
            throw nul ;
        } catch( OutOfMemoryError me ) {
            LOG.error( "[(Receive)HeartBeat]OutOfMemoryError",me ) ;
        } catch( Exception e ) {
            LOG.warn( "[(Receive)HearBeat]受信処理時に例外",e ) ;
        }
        
    }
    
}

