/*
 * @(#)ExecutionResultErrorSwitch.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.receive ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.NumberTable;
import org.maachang.queue.access.status.ChannelStatus;
import org.maachang.queue.main.channel.Channel;
import org.maachang.queue.main.channel.ChannelFactory;
import org.maachang.queue.main.channel.SendChannel;
import org.maachang.queue.main.channel.protocol.ChannelProtocol;
import org.maachang.queue.main.channel.protocol.ProtocolObject;
import org.maachang.queue.main.channel.protocol.ProtocolResultError;
import org.maachang.queue.main.channel.service.ChannelServiceError;
import org.maachang.queue.main.channel.service.receive.core.CoreReceiveChild;
import org.maachang.queue.main.channel.service.send.ChannelOption;
import org.maachang.queue.main.connect.Connect;
import org.maachang.queue.main.connect.ConnectFactory;

/**
 * 受信電文[ProtocolResultError]処理実装.
 *
 * @version 2006/12/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ExecutionResultErrorSwitch implements ExecutionSwitch {
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ExecutionResultErrorSwitch.class ) ;
    
    /**
     * 戻りエラープロトコルプーリング.
     */
    private final NumberTable pool = new NumberTable() ;
    
    /**
     * プロトコルタイプを取得.
     * <BR><BR>
     * プロトコルタイプを取得します.
     * <BR>
     * @return int プロトコルタイプが返されます.<BR>
     *             [ProtocolObject.PROTOCOL_DATA]が返された場合、データオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_HEARTBEAT]が返された場合、ハートビートオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_SUCCESS]が返された場合、正常オブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_ERROR]が返された場合ｍエラーオブジェクトです.
     */
    public int getType() {
        return ProtocolObject.PROTOCOL_ERROR ;
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * @param threadNum スレッド項番が設定されます.
     * @param child 処理対象の要素が設定されます.
     * @param sync 同期オブジェクト.
     */
    public void execution( int threadNum,CoreReceiveChild child,Synchronized sync ) {
        try {
            
            // プロトコルをプールオブジェクトから取得.
            ProtocolResultError resultError = ( ProtocolResultError )pool.get(
                threadNum ) ;
            
            // 情報が存在しない場合は、生成して、プールに設定.
            if( resultError == null ) {
                resultError = new ProtocolResultError() ;
                pool.add( threadNum,resultError ) ;
            }
            
            // 電文情報から、戻りエラー電文を生成.
            ChannelProtocol.getTelegramByProtocolResultError(
                resultError,child.getBinResource() ) ;
            
            // チャネルを取得.
            Channel channel = ChannelFactory.getToID(
                resultError.getChannelId() ) ;
            
            // 取得したチャネルが不正な場合.
            if( channel == null || channel.isChannel() == false ||
                ( channel instanceof SendChannel ) == false ) {
                // 処理しない.
                return ;
            }
            
            // 接続先のコネクションを取得.
            Connect con = ConnectFactory.get(
                ( ( SendChannel)channel ).getConnectName() ) ;
            
            // 対象チャネルのコネクションポートと、
            // 受信されたときのコネクションポートが一致しない場合.
            if( con == null || con.getPort() != resultError.getReceivePort() ) {
                // 処理しない.
                return ;
            }
            
            // 戻りエラー電文の処理元タイプで処理分離.
            switch( resultError.getReturnType() ) {
                case ProtocolObject.PROTOCOL_HEARTBEAT :
                    // ハートビート電文処理.
                    this.resultErrorByHeartBeat( child,channel,con,resultError,sync ) ;
                    break ;
            }
            
        } catch( NullPointerException nul ) {
            throw nul ;
        } catch( OutOfMemoryError me ) {
            LOG.error( "[(Receive)ResultError]OutOfMemoryError",me ) ;
        } catch( Exception e ) {
            LOG.warn( "[(Receive)ResultError]受信処理時に例外",e ) ;
        }
        
    }
    
    /**
     * ハートビート電文系エラー処理.
     */
    private final void resultErrorByHeartBeat(
        CoreReceiveChild child,Channel channel,Connect con,
        ProtocolResultError resultError,Synchronized sync ) {
        
        // チャネルオプションを取得.
        ChannelOption opt = ( ChannelOption )(
            ( SendChannel )channel ).getOption() ;
        
        // チャネルオプションが存在しない場合.
        if( opt == null ) {
            return ;
        }
        
        // 最新の応答電文であるかチェック.
        synchronized( opt ) {
            
            // 対象送信チャネルIDと、受信されたときのIDが一致しない場合.
            if( opt.getSendProtocol() == null ||
                resultError.getId() != opt.getSendProtocol().getId() ) {
                // 処理しない.
                return ;
            }
            
        }
        
        // 対象の送信チャネルステータスが、シャットダウン中か、
        // エラー以外の場合.
        if( channel.getState() != ChannelStatus.STATE_DOWN &&
            channel.getState() != ChannelStatus.STATE_SHUTDOWN &&
            channel.getState() != ChannelStatus.STATE_ERROR ) {
            
            // 受信ステータスに対して処理.
            switch( resultError.getErrorCode() ) {
                case ChannelServiceError.NOT_CHANNEL_NAME :
                    // 接続先のチャネル情報は存在しない.
                    channel.setState( ChannelStatus.STATE_NOT_CONNECT_CHANNEL ) ;
                    break ;
                case ChannelServiceError.CHANNEL_STATE_BY_NOT_SUCCESS :
                    // 接続先のチャネルは停止中.
                    channel.setState( ChannelStatus.STATE_NOT_HEART_BEAT ) ;
                    break ;
                case ChannelServiceError.NOT_QUEUE_MANAGER_NAME :
                    // 接続先のキューマネージャ名は存在しない.
                    channel.setState( ChannelStatus.STATE_NOT_QUEUE_MANAGER ) ;
                    break ;
                case ChannelServiceError.QUEUE_MANAGER_STATE_BY_NOT_SUCCESS :
                    // 接続先のキューマネージャは停止中.
                    // (チャネル停止と意味は同じ).
                    channel.setState( ChannelStatus.STATE_NOT_HEART_BEAT ) ;
                    break ;
            }
            
        }
        
    }
    
}

