/*
 * @(#)ReceiveSwitchImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.receive ;

import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.main.channel.service.receive.core.CoreReceiveChild;

/**
 * 受信電文処理用のスレッドオブジェクト.
 *
 * @version 2006/12/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
class ReceiveSwitchImple extends ExecutionThread implements ReceiveSwitch
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ReceiveSwitchImple.class ) ;
    
    
    
    /**
     * データ管理オブジェクト.
     */
    private Vector vector = null ;
    
    /**
     * スレッド項番.
     */
    private int threadNum = 0 ;
    
    /**
     * 実装インターフェイス.
     */
    private ExecutionSwitch executionSwitch = null ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    private ReceiveSwitchImple(){
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 受信電文処理用のスレッドオブジェクトを生成します.
     * <BR>
     * @param threadNum スレッド項番を設定します.
     * @param vector 対象のデータ管理オブジェクトを設定します.
     */
    public ReceiveSwitchImple( ExecutionSwitch executionSwitch,int threadNum,Vector vector ){
        
        sync.create() ;
        
        try{
            
            this.executionSwitch = executionSwitch ;
            this.vector = vector ;
            this.threadNum = threadNum ;
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        sync.clear() ;
        thread.clear() ;
        vector = null ;
        threadNum = 0 ;
        executionSwitch = null ;
    }
    
    /**
     * データセット.
     * <BR><BR>
     * データをセットします.
     * <BR>
     * @param coreReceiveChild 対象のデータを設定します.
     */
    public void setCoreReceiveChild( CoreReceiveChild coreReceiveChild ) {
        this.vector.add( coreReceiveChild ) ;
    }
    
    /**
     * スレッド項番を取得.
     * <BR><BR>
     * スレッド項番を取得します.
     * <BR>
     * @return int スレッド項番が返されます.
     */
    public int getThreadNum() {
        return this.threadNum ;
    }
    
    /**
     * プロトコルタイプを取得.
     * <BR><BR>
     * プロトコルタイプを取得します.
     * <BR>
     * @return int プロトコルタイプが返されます.<BR>
     *             [ProtocolObject.PROTOCOL_DATA]が返された場合、データオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_HEARTBEAT]が返された場合、ハートビートオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_SUCCESS]が返された場合、正常オブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_ERROR]が返された場合ｍエラーオブジェクトです.
     */
    public int getType() {
        return this.executionSwitch.getType() ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        String execName = null ;
        CoreReceiveChild child = null ;
        ExecutionSwitch exec = null ;
        Vector table = null ;
        int threadNo ;
        
        try{
            
            //UtilCom.idleTime() ;
            
            execName = executionSwitch.getClass().getName() ;
            
            // 対象条件を取得.
            try {
                synchronized( sync.get() ) {
                    threadNo = this.threadNum ;
                    exec = this.executionSwitch ;
                    table = this.vector ;
                }
                child = ( CoreReceiveChild )table.remove( 0 ) ;
            } catch( NullPointerException np ) {
                throw np ;
            } catch( Exception e ) {
                child = null ;
                threadNo = -1 ;
            }
            
            // 要素が存在する場合.
            if( child != null ) {
                // 処理実行.
                exec.execution( threadNo,child,sync ) ;
            }
            // バイナリが存在しない場合は、待機.
            else {
                UtilCom.idleTime() ;
            }
            
        } catch( OutOfMemoryError me ) {
            LOG.error( "OutOfMemoryError",me ) ;
        }catch( NullPointerException nul ){
            LOG.error( "スレッド停止しました[" + threadNum + "] - " + execName, nul ) ;
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました[" + threadNum + "] - " + execName, be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました[" + threadNum + "] - " + execName, e ) ;
        }finally{
            
        }
        
    }
    
}

