/*
 * @(#)CoreReceiveThread.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.receive.core ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.main.connect.Connect;
import org.maachang.queue.main.connect.ConnectFactory;

/**
 * コア受信処理.
 *
 * @version 2006/12/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class CoreReceiveThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( CoreReceiveThread.class ) ;
    
    /**
     * コネクションデータ設定待機回数.
     */
    private static final int MAX_RETRY = 64 ;
    
    
    
    /**
     * スレッド項番.
     */
    private int threadNum = 0 ;
    
    /**
     * コネクションアドレス受け取り用.
     */
    private final ConnectAddress readAddress = new ConnectAddress() ;
    
    /**
     * コア受信テーブル.
     */
    private CoreReceiveTable coreReceiveTable = null ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    /**
     * 暗号ワード格納オブジェクト.
     */
    private final String[] cb32Word = new String[ 1 ] ;
    
    
    /**
     * コンストラクタ.
     */
    private CoreReceiveThread(){
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 受信用コア処理を行うスレッドを生成します.
     * <BR>
     * @param threadNum スレッド項番を設定します.
     * @param coreReceiveTable テーブルオブジェクトを設定します.
     */
    public CoreReceiveThread( int threadNum,CoreReceiveTable coreReceiveTable ){
        
        sync.create() ;
        
        try{
            
            this.coreReceiveTable = coreReceiveTable ;
            this.threadNum = threadNum ;
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        sync.clear() ;
        thread.clear() ;
        readAddress.clear() ;
        threadNum = 0 ;
        coreReceiveTable = null ;
    }
    
    /**
     * テーブルオブジェクトを取得.
     * <BR><BR>
     * テーブルオブジェクトを取得します.
     * <BR>
     * @return CoreReceiveTable テーブルオブジェクトが返されます.
     */
    public final CoreReceiveTable getCoreReceiveTable() {
        return coreReceiveTable ;
    }
    
    /**
     * スレッド項番を取得.
     * <BR><BR>
     * スレッド項番を取得します.
     * <BR>
     * @return int スレッド項番が返されます.
     */
    public final int getThreadNum() {
        return this.threadNum ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int i,j ;
        int len ;
        boolean flag = false ;
        String[] names = null ;
        
        try{
            
            //UtilCom.idleTime() ;
            
            // コネクション名群を取得.
            names = ConnectFactory.getNames() ;
            
            // コネクション名が存在する場合.
            if( names != null && ( len = names.length ) > 0 ) {
                
                // 受信データ管理テーブルを取得.
                CoreReceiveTable coreReceiveTable = null ;
                ConnectAddress readAddr = null ;
                
                synchronized( sync.get() ) {
                    coreReceiveTable = this.coreReceiveTable ;
                    readAddr = this.readAddress ;
                }
                
                for( i = 0 ; i < len ; i ++ ) {
                    
                    // コネクションオブジェクトを取得.
                    Connect conn = ConnectFactory.get( names[ i ] ) ;
                    
                    // 存在しない場合は、処理しない.
                    if( conn == null ) {
                        UtilCom.idleTime() ;
                        continue ;
                    }
                    
                    // 送信された情報を取得.
                    try {
                        
                        // 情報が存在しなくなるまで取得.
                        for( ;; ) {
                            
                            // 受信処理.
                            BinResource bin = conn.recvByResource( readAddr,cb32Word ) ;
                            
                            // 受信データが存在しない場合は処理しない.
                            if( bin == null || bin.size() <= 0 ) {
                                
                                // 処理しない.
                                UtilCom.idleTime() ;
                                break ;
                                
                            }
                            
                            // 情報を設定.
                            if( coreReceiveTable.put(
                                conn.getName(),bin,readAddr.getAddress(),
                                readAddr.getPort(),cb32Word[ 0 ] ) == false ) {
                                
                                // 情報を設定できなかった場合は、設定できるまで待機.
                                for( flag = false ;; ) {
                                    
                                    // ログ出力待機時間.
                                    for( j = 0 ; j < MAX_RETRY ; j ++ ) {
                                        
                                        // 設定できるまで処理を待つ.
                                        if( ( flag = coreReceiveTable.put(
                                            conn.getName(),bin,readAddr.getAddress(),
                                            readAddr.getPort(),cb32Word[ 0 ] )
                                            ) == false ) {
                                            
                                            // ロール満杯により設定できない場合は、
                                            // 追加できるまで待つ.
                                            UtilCom.idleTime() ;
                                            continue ;
                                            
                                        }
                                        
                                        break ;
                                        
                                    }
                                    // 情報が設定できなかった場合.
                                    if( flag == true ) {
                                        break ;
                                    }
                                    
                                    // ロール満杯を示すログを出力.
                                    LOG.warn( "## CoreReceiveTable[" + threadNum + "] - maxRoll!!" ) ;
                                    
                                }
                                
                            }
                            
                            bin = null ;
                            
                        }
                        
                    } catch( ExecutionException ee ) {
                    }
                    
                    names[ i ] = null ;
                    
                }
                
            }
            // コネクションが存在しない場合.
            else {
                // 待機.
                UtilCom.idleTime() ;
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました[" + this.threadNum + "]", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました[" + this.threadNum + "]", e ) ;
        }finally{
            
        }
        
    }
    
}

