/*
 * @(#)SendQueueManage.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.send ;

import org.maachang.commons.util.CharTable;
import org.maachang.queue.main.queue.SendMqOption;
import org.maachang.queue.main.queue.base.BaseQueue;
import org.maachang.queue.main.queue.base.core.QArrayChild;
import org.maachang.queue.main.queue.base.core.QSendSeq;

/**
 * コミット済み送信キューデータ送信用管理.
 *
 * @version 2006/12/22
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class SendQueueManage {
    
    /**
     * 区切り文字.
     */
    private static final String CUT_CODE = "@" ;
    
    /**
     * 送信中止タイムアウト.
     */
    private static final long STOP_SEND_TIMEOUT = 120000L ;
    
    /**
     * 再送タイミング.
     */
    private static final long RESEND_TIMING = 3500L ;
    
    /**
     * 送信キューロック.
     */
    private final LockSendQueue lock = new LockSendQueue() ;
    
    /**
     * 待ちデータ管理テーブル.
     */
    private final CharTable table = new CharTable() ;
    
    /**
     * コンストラクタ.
     */
    public SendQueueManage() {
        
    }
    
    /**
     * 送信終了、送信中止確認.
     * <BR><BR>
     * 送信完了であるか、送信中止であるかを確認します.
     * <BR>
     * @param queue 対象のキューオブジェクトを設定します.
     * @param qch 対象の送信条件を設定します.
     * @return boolean 処理中止であるか返されます.
     */
    public synchronized boolean checkExitOrStop( BaseQueue queue,QArrayChild qch ) {
        
        if( qch != null && qch.getOption() != null ) {
            
            SendQueueParam param = ( SendQueueParam )qch.getOption() ;
            
            // 処理中の場合.
            if( param.isExecution() == true ) {
                return true ;
            }
            
            // 送信完了の場合.
            this.isSendSuccess( queue,qch ) ;
            
            // 送信中止であるか確認.
            if( System.currentTimeMillis() >
                param.getCreateTime() + SendQueueManage.STOP_SEND_TIMEOUT ) {
                this.removeTable( queue,param.getNowSendId() ) ;
                param.create() ;
                return true ;
            }
            
        }
        
        return false ;
        
    }
    
    /**
     * 指定送信IDに対するチャネルIDを送信完了に設定.
     * <BR><BR>
     * 指定送信IDに対するチャネルIDを送信完了に設定します.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param id 対象の送信IDを設定します.
     * @param channelId 対象のチャネルIDを設定します.
     */
    public synchronized void receiveChannelId(
        String manager,String queue,long id,int channelId ) {
        if( id < 0 || channelId == -1 ) {
            return ;
        }
        
        QArrayChild qch = ( QArrayChild )table.get(
            SendQueueManage.getIndex( manager,queue,
            	QSendSeq.getNowNextIdByNowId( id ) ) ) ;
        
        if( qch != null ) {
            int[] channels = ( ( SendQueueParam )qch.getOption() ).getSendChannelId() ;
            int len = channels.length ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( channels[ i ] == channelId ) {
                    channels[ i ] = -1 ;
                    break ;
                }
            }
        }
    }
    
    /**
     * 再送タイミングチェック.
     * <BR><BR>
     * 再送タイミングをチェックします.
     * <BR>
     * @param queue 送信キューを設定します.
     * @param qch 対象の送信条件を設定します.
     * @return boolean 再送タイミング結果が返されます.
     */
    public synchronized boolean isResend( BaseQueue queue,QArrayChild qch ) {
        
        if( queue != null && queue.getOption() != null && qch != null ) {
        	
        	SendQueueParam param = null ;
        	
        	///////////////////////
        	// 新しい送信対象の場合.
        	///////////////////////
            if( qch.getOption() == null ) {
                this.newSend( queue,qch ) ;
                
                param = ( SendQueueParam )qch.getOption() ;
                
                // 処理中に設定.
                param.setExecution( true ) ;
                return true ;
                
            }
            // データにパラメータが存在するが、管理テーブルに
            // そのパラメータが存在しない場合.
            else if( qch.getSendId() > 0L &&
            	this.isTable(
            		queue,QSendSeq.getNowNextIdByNowId( qch.getSendId() )
            	) == false ) {
            	
            	this.resetSend( queue,qch ) ;
            	
            }
            
            // 送信メッセージパラメータを取得.
            param = ( SendQueueParam )qch.getOption() ;
            
            ///////////////////////////
            // 既に１度送信している場合.
            ///////////////////////////
            
            // 現在処理中の場合.
            if( param.isExecution() == true ) {
                // 処理しない.
                return false ;
            }
            
            // 送信実行処理タイミングの場合.
            if( System.currentTimeMillis() >=
                param.getUpdateTime() + SendQueueManage.RESEND_TIMING ) {
            	
                param = ( SendQueueParam )qch.getOption() ;
                
                // 更新日付を最新に設定.
                param.updateTime() ;
                
                // 処理中に設定.
                param.setExecution( true ) ;
                
                return true ;
                
            }
            
        }
        
        return false ;
    }
    
    /**
     * 処理中の状態を解除.
     * <BR><BR>
     * 処理中の状態を解除します.
     * <BR>
     * @param qch 対象の送信条件を設定します.
     */
    public synchronized void releaseExecution( QArrayChild qch ) {
        if( qch != null && qch.getOption() != null ) {
            ( ( SendQueueParam )qch.getOption() ).setExecution( false ) ;
        }
    }
    
    /**
     * 送信キューロックオブジェクトを取得.
     * <BR><BR>
     * 送信キューロックオブジェクトを取得します.
     * <BR>
     * @return LockSendQueue 送信キューロックオブジェクトが返されます.
     */
    public LockSendQueue getLock() {
        return lock ;
    }
    
    /**
     * 送信完了チェック.
     * <BR><BR>
     * 送信が完了しているかチェックします.
     * <BR>
     * @param queue 送信キューを設定します.
     * @param qch 対象の送信条件を設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、削除されました.
     */
    public synchronized boolean isSendSuccess( BaseQueue queue,QArrayChild qch ) {
        
    	// 情報が既に破棄されている場合.
    	if( qch == null || qch.isUse() == false ) {
    		return true ;
    	}
    	
        // 送信終了であるか確認.
        SendQueueParam param = ( SendQueueParam )qch.getOption() ;
        int[] channels = param.getSendChannelId() ;
        int len = channels.length ;
        boolean flag = false ;
        
        // 送信終了かチェック.
        for( int i = 0 ; i < len ; i ++ ) {
        	
        	// 対象チャネルが送信終了でない場合.
            if( channels[ i ] != -1 ) {
                flag = true ;
                break ;
            }
            
        }
        
        // 送信終了の場合.
        if( flag == false ) {
        	
        	// 管理・キューメッセージを削除.
            this.removeTable( queue,param.getNowSendId() ) ;
            queue.removeQArrayChild( qch ) ;
            qch.clear() ;
            
            return true ;
            
        }
        
        return false ;
    }
    
    /**
     * 新しく送信する条件を追加.
     * <BR><BR>
     * 新しく送信する条件を追加します.
     * <BR>
     * @param queue 送信キューを設定します.
     * @param qch 新しく送信する条件を設定します.
     */
    private void newSend( BaseQueue queue,QArrayChild qch ) {
        
        if( queue != null && queue.getOption() != null &&
            qch != null && qch.getOption() == null ) {
            
        	// 新しいパラメータを生成して、条件をセット.
        	SendQueueParam param = new SendQueueParam() ;
            
            // 送信先チャネル群を設定.
            param.setSendChannelId(
                ( ( SendMqOption )queue.getOption() ).getIds() ) ;
            
            // 新しい送信IDを生成.
            param.setSendId( queue.getSendSequence().sequenceNowIdAll() ) ;
            
            // 送信データにオプションとしてセット.
            qch.setOption( param ) ;
            
            // 管理テーブルに追加.
            this.addTable( queue,param.getNowSendId(),qch ) ;
            
        }
        
    }
    
    /**
     * 送信条件を再セットする条件を追加.
     * <BR><BR>
     * 送信条件を再セットする条件を追加します.
     * <BR>
     * @param queue 送信キューを設定します.
     * @param qch 新しく送信する条件を設定します.
     */
    private void resetSend( BaseQueue queue,QArrayChild qch ) {
        
        if( queue != null && queue.getOption() != null &&
            qch != null && qch.getOption() != null ) {
            
        	// 新しいパラメータを生成して、条件をセット.
        	SendQueueParam param = ( SendQueueParam )qch.getOption() ;
        	
            // 管理テーブルに追加.
            this.addTable( queue,param.getNowSendId(),qch ) ;
            
        }
        
    }
    
    
    /**
     * テーブルに設定.
     */
    private void addTable( BaseQueue queue,int id,QArrayChild qch ) {
        table.add( SendQueueManage.getIndex(
            queue.getState().getParentManagerName(),
            queue.getState().getName(),id ),qch ) ;
    }
    
    /**
     * テーブルから削除.
     */
    private void removeTable( BaseQueue queue,int id ) {
        table.remove( SendQueueManage.getIndex(
            queue.getState().getParentManagerName(),
            queue.getState().getName(),id ) ) ;
    }
    
    /**
     * テーブルに存在するかチェック.
     */
    private boolean isTable( BaseQueue queue,int id ) {
        return table.isData( SendQueueManage.getIndex(
            queue.getState().getParentManagerName(),
            queue.getState().getName(),id ) ) ;
    }
    
    /**
     * インデックス名を生成.
     */
    private static final String getIndex(
        String manager,String queue,int id ) {
        
        return new StringBuffer().
            append( manager ).
            append( CUT_CODE ).
            append( queue ).
            append( CUT_CODE ).
            append( id ).
            toString() ;
    }
}

