/*
 * @(#)SendService.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.send ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.sys.NamingManager;

/**
 * 送信サービス.
 * 
 * @version 2006/12/22
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class SendService {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( SendService.class ) ;
    
    /**
     * ネーミングマネージャ登録名.
     */
    public static final String NAMING_MANAGER = "SERVICE@maachangq.main.channel.service.send" ;
    
    
    /**
     * スレッドデフォルト値.
     */
    private static final int DEF_THREAD = 3 ;
    
    /**
     * スレッド最小数.
     */
    private static final int MIN_THREAD = 1 ;
    
    /**
     * スレッド最大数.
     */
    private static final int MAX_THREAD = 6 ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public SendService() {
        
    }
    
    /**
     * 送信サービス初期化処理.
     * <BR><BR>
     * @param sendDataThread 送信スレッド数を設定します.
     * @param time ハートビート処理間隔を設定します.
     * @param closeCount コネクション不能までのカウント値を設定します.
     */
    public final void init( int sendDataThread,long time,int closeCount ) {
        
        int i ;
        
        if( sendDataThread == -1 ) {
            sendDataThread = DEF_THREAD ;
        }
        
        sendDataThread = ( sendDataThread <= MIN_THREAD ) ? MIN_THREAD : sendDataThread ;
        sendDataThread = ( sendDataThread >= MAX_THREAD ) ? MAX_THREAD : sendDataThread ;
        
        LOG.info( "#### 送信サービス生成処理 ####" ) ;
        LOG.info( "# 送信スレッド数:" + sendDataThread ) ;
        LOG.info( "# ハートビート処理間隔:" + time ) ;
        LOG.info( "# ハートビートリトライクローズカウント:" + closeCount ) ;
        
        // 送信管理テーブルを生成.
        SendQueueManage sendQueueManage = new SendQueueManage() ;
        
        // 送信スレッドを生成.
        SendQueueDataThread[] sendQueueDataThread = new SendQueueDataThread[ sendDataThread ] ;
        for( i = 0 ; i < sendDataThread ; i ++ ) {
            sendQueueDataThread[ i ] = new SendQueueDataThread( i,sendQueueManage ) ;
        }
        
        // ハートビートスレッドを生成.
        SendHeartBeatThread sendHeartBeatThread = new SendHeartBeatThread( time,closeCount ) ;
        
        // 生成オブジェクトを１つのオブジェクトに集約.
        SendServiceBean swObjs = null ;
        swObjs = new SendServiceBean() ;
        swObjs.setSendHeartBeatThread( sendHeartBeatThread ) ;
        swObjs.setSendQueueDataThread( sendQueueDataThread ) ;
        swObjs.setSendQueueManage( sendQueueManage ) ;
        
        this.destroy( false ) ;
        
        // ネーミングマネージャに登録.
        synchronized( SYNC ) {
            
            NamingManager.add( NAMING_MANAGER,swObjs ) ;
            
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy() {
        this.destroy( true ) ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @param mode ログ表示モードを設定します.
     */
    public final void destroy( boolean mode ) {
        
        if( mode == true ) {
            LOG.info( "#### 送信サービス破棄処理 ####" ) ;
        }
        
        synchronized( SYNC ) {
            
            NamingManager.remove( NAMING_MANAGER ) ;
            
        }
    }
    
    /**
     * 送信管理テーブルを取得.
     * <BR><BR>
     * 送信管理テーブルを取得します.
     * <BR>
     * @return SendQueueManage 送信管理テーブルが返されます.
     */
    public static final SendQueueManage getSendQueueManage() {
        
        synchronized( SYNC ) {
            
            SendServiceBean sw = SendService.getSendServiceBean() ;
            if( sw != null ) {
                return sw.getSendQueueManage() ;
            }
            
        }
        
        return null ;
    }
    
    /**
     * ハートビートスレッドを取得.
     * <BR><BR>
     * ハートビートスレッドを取得します.
     * <BR>
     * @return SendHeartBeatThread ハートビートスレッドが返されます.
     */
    public static final SendHeartBeatThread getSendHeartBeatThread() {
        
        synchronized( SYNC ) {
            
            SendServiceBean sw = SendService.getSendServiceBean() ;
            if( sw != null ) {
                return sw.getSendHeartBeatThread() ;
            }
            
        }
        
        return null ;
    }
    
    /**
     * 送信データスレッドを取得.
     * <BR><BR>
     * 送信データスレッドを取得します.
     * <BR>
     * @return SendQueueDataThread 送信データスレッドが返されます.
     */
    public static final SendQueueDataThread getSendQueueDataThread() {
        
        synchronized( SYNC ) {
            
            SendServiceBean sw = SendService.getSendServiceBean() ;
            if( sw != null ) {
                return sw.getSendQueueDataThread()[ sw.getSendQueueDataPosition() ] ;
            }
            
        }
        
        return null ;
        
    }
    
    /**
     * オブジェクトが有効であるかチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オブジェクトは有効です.<BR>
     *                 [false]が返された場合、オブジェクトは無効です.
     */
    public static final boolean isUse() {
        synchronized( SYNC ) {
            
            SendServiceBean sw = SendService.getSendServiceBean() ;
            if( sw != null ) {
                
                int i ;
                int len ;
                
                // 送信データスレッドチェック.
                if( sw.getSendQueueDataThread() == null ||
                    ( len = sw.getSendQueueDataThread().length ) <= 0 ) {
                    return false ;
                }
                for( i = 0 ; i < len ; i ++ ) {
                    if( sw.getSendQueueDataThread()[ i ] == null ||
                        sw.getSendQueueDataThread()[ i ].isThread() == false ) {
                        return false ;
                    }
                }
                
                // ハートビートスレッドチェック.
                if( sw.getSendHeartBeatThread() == null ||
                    sw.getSendHeartBeatThread().isThread() == false ) {
                    return false ;
                }
                
                // 全て動作確認OK.
                return true ;
                
            }
            
        }
        
        return false ;
    }
    
    /**
     * このオブジェクトが破棄されたかチェック.
     * <BR><BR>
     * このオブジェクトが破棄されたかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.
     */
    public static final boolean isDestroy() {
        synchronized( SYNC ) {
            return ( NamingManager.get( NAMING_MANAGER ) == null ) ? false : true ;
        }
    }
    
    /**
     * SendServiceBeanオブジェクトを取得.
     */
    private static final SendServiceBean getSendServiceBean() {
        Object o = NamingManager.get( NAMING_MANAGER ) ;
        if( o != null && o instanceof SendServiceBean ) {
            return ( SendServiceBean )o ;
        }
        return null ;
    }
    
}

/**
 * 送信サービスBeanオブジェクト.
 */
class SendServiceBean {
    
    /**
     * 送信データ管理テーブル.
     */
    private SendQueueManage sendQueueManage = null ;
    
    /**
     * 送信データスレッド.
     */
    private SendQueueDataThread[] sendQueueDataThread = null ;
    
    /**
     * ハートビートスレッド.
     */
    private SendHeartBeatThread sendHeartBeatThread = null ;
    
    /**
     * 送信データオブジェクトポジション.
     */
    private int sendQueueDataPosition = 0 ;
    
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクトクリア.
     * <BR><BR>
     * オブジェクトをクリアします.
     */
    public void clear() {
        sendQueueManage = null ;
        sendQueueDataThread = null ;
        sendHeartBeatThread = null ;
        sendQueueDataPosition = 0 ;
    }

    /**
     * sendHeartBeatThreadを取得.
     * <BR><BR>
     * @return sendHeartBeatThread が返されます。
     */
    public SendHeartBeatThread getSendHeartBeatThread() {
        return sendHeartBeatThread;
    }

    /**
     * sendHeartBeatThreadを設定.
     * <BR><BR>
     * @param sendHeartBeatThread 対象のsendHeartBeatThreadを設定.
     */
    public void setSendHeartBeatThread(SendHeartBeatThread sendHeartBeatThread) {
        this.sendHeartBeatThread = sendHeartBeatThread;
    }

    /**
     * sendQueueDataThreadを取得.
     * <BR><BR>
     * @return sendQueueDataThread が返されます。
     */
    public SendQueueDataThread[] getSendQueueDataThread() {
        return sendQueueDataThread;
    }

    /**
     * sendQueueDataThreadを設定.
     * <BR><BR>
     * @param sendQueueDataThread 対象のsendQueueDataThreadを設定.
     */
    public void setSendQueueDataThread(SendQueueDataThread[] sendQueueDataThread) {
        this.sendQueueDataThread = sendQueueDataThread;
    }

    /**
     * sendQueueManageを取得.
     * <BR><BR>
     * @return sendQueueManage が返されます。
     */
    public SendQueueManage getSendQueueManage() {
        return sendQueueManage;
    }

    /**
     * sendQueueManageを設定.
     * <BR><BR>
     * @param sendQueueManage 対象のsendQueueManageを設定.
     */
    public void setSendQueueManage(SendQueueManage sendQueueManage) {
        this.sendQueueManage = sendQueueManage;
    }

    /**
     * sendQueueDataPositionを取得.
     * <BR><BR>
     * @return sendQueueDataPosition が返されます。
     */
    public int getSendQueueDataPosition() {
        sendQueueDataPosition ++ ;
        if( sendQueueDataThread != null &&
            sendQueueDataThread.length <= sendQueueDataPosition ) {
            sendQueueDataPosition = 0 ;
        }
        return sendQueueDataPosition;
    }
}

