/*
 * @(#)Connect.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.connect ;

import java.net.InetAddress;

import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.net.ConnectTimeoutException;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.ResourceType;
import org.maachang.connect.MaachangConnect;
import org.maachang.queue.config.MqDefine;

/**
 * コネクションインターフェイス.
 *  
 * @version 2006/08/30
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public interface Connect
{
    
    /**
     * デフォルトコネクション送信ポート.
     */
    public static final int DEF_PORT = MqDefine.DEFAULT_CHANNEL_PORT ;
    
    
    /**
     * 送信処理.
     * <BR><BR>
     * 指定アドレスに対して電文を送信します.<BR>
     * また、送信対象の内容はバイナリです.
     * <BR>
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param addr 送信先のアドレスを設定します.
     * @param binary 送信対象の電文を設定します.
     */
    public void sendByBinary( String cb32Word,InetAddress addr,byte[] binary ) ;
    
    /**
     * 送信処理.
     * <BR><BR>
     * 指定アドレスに対して電文を送信します.<BR>
     * また、送信対象の内容はバイナリです.
     * <BR>
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param addr 送信先のアドレスを設定します.
     * @param port 送信先の受信アドレスを設定します.
     * @param binary 送信対象の電文を設定します.
     */
    public void sendByBinary( String cb32Word,InetAddress addr,int port,byte[] binary ) ;
    
    /**
     * 送信処理.
     * <BR><BR>
     * 指定アドレスに対して電文を送信します.<BR>
     * また、送信対象の内容はバイナリリソースです.
     * <BR>
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param addr 送信先のアドレスを設定します.
     * @param binary 送信対象の電文を設定します.
     */
    public void sendByResource( String cb32Word,InetAddress addr,BinResource binary ) ;
    
    /**
     * 送信処理.
     * <BR><BR>
     * 指定アドレスに対して電文を送信します.<BR>
     * また、送信対象の内容はバイナリリソースです.
     * <BR>
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param addr 送信先のアドレスを設定します.
     * @param port 送信先の受信アドレスを設定します.
     * @param binary 送信対象の電文を設定します.
     */
    public void sendByResource( String cb32Word,InetAddress addr,int port,BinResource binary ) ;
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受け取った電文情報を受信します.<BR>
     * また受信対象はバイナリです.
     * <BR>
     * @param out 受信元もアドレス情報が格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param cb32Word 対象の暗号ワードが格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @return byte[] 受信されたバイナリ情報が返されます.
     * @exception ExecutionException 実行例外.
     */
    public byte[] recvByBinary( ConnectAddress out,String[] cb32Word )
        throws ExecutionException ;
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受け取った電文情報を受信します.<BR>
     * また受信対象はバイナリリソースです.
     * <BR>
     * @param out 受信元もアドレス情報が格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param cb32Word 対象の暗号ワードが格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @return BinResource 受信されたバイナリ情報が返されます.
     * @exception ExecutionException 実行例外.
     */
    public BinResource recvByResource( ConnectAddress out,String[] cb32Word )
        throws ExecutionException ;
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受け取った電文情報を受信します.<BR>
     * また受信対象はバイナリです.
     * <BR>
     * @param out 受信元もアドレス情報が格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param cb32Word 対象の暗号ワードが格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param timeout タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限に待ちます.
     * @return byte[] 受信されたバイナリ情報が返されます.
     * @exception ConnectTimeoutException 受信タイムアウト例外.
     * @exception ExecutionException 実行例外.
     */
    public byte[] recvByBinary( ConnectAddress out,String[] cb32Word,int timeout )
        throws ConnectTimeoutException,ExecutionException ;
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受け取った電文情報を受信します.<BR>
     * また受信対象はバイナリリソースです.
     * <BR>
     * @param out 受信元もアドレス情報が格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param cb32Word 対象の暗号ワードが格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param timeout タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限に待ちます.
     * @return BinResource 受信されたバイナリ情報が返されます.
     * @exception ConnectTimeoutException 受信タイムアウト例外.
     * @exception ExecutionException 実行例外.
     */
    public BinResource recvByResource( ConnectAddress out,String[] cb32Word,int timeout )
        throws ConnectTimeoutException,ExecutionException ;
    
    /**
     * 通信オブジェクト名を取得.
     * <BR><BR>
     * 通信オブジェクト名を取得します.
     * <BR>
     * @return String 通信オブジェクト名が返されます.
     */
    public String getName() ;
    
    /**
     * リソースタイプを取得.
     * <BR><BR>
     * リソースタイプを取得します.
     * <BR>
     * @return ResourceType リソースタイプが返されます.
     */
    public ResourceType getResourceType() ;
    
    /**
     * キャッシュ名を取得.
     * <BR><BR>
     * キャッシュ名を取得します.
     * <BR>
     * @return String キャッシュ名が返されます.
     */
    public String getCacheName() ;
    
    /**
     * 通信バッファ長を取得.
     * <BR><BR>
     * 通信バッファ長が返されます.
     * <BR>
     * @return int 通信バッファ長が返されます.
     */
    public int getBuffer() ;
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインドアドレスを取得します.
     * <BR>
     * @return InetAddress バインドアドレスが返されます.
     */
    public InetAddress getBindAddress() ;
    
    /**
     * バインドポート番号を取得.
     * <BR><BR>
     * バインドポート番号を取得します.
     * <BR>
     * @return int 送信ポート番号が返されます.
     */
    public int getPort() ;
    
    /**
     * 暗号ワードを追加.
     * <BR><BR>
     * 暗号ワードを追加します.
     * <BR>
     * @param word 対象の暗号ワードを設定します.
     */
    public void addCb32Word( String word ) ;
    
    /**
     * 暗号ワードを削除.
     * <BR><BR>
     * 暗号ワードを削除します.
     * <BR>
     * @param word 削除対象の暗号ワードを設定します.
     */
    public void removeCb32Word( String word ) ;
    
    /**
     * 暗号ワード群を取得.
     * <BR><BR>
     * 暗号ワード群を取得します.
     * <BR>
     * @return String[] 暗号ワードが返されます.
     */
    public String[] getCb32Word() ;
    
    /**
     * 暗号ワード数を取得.
     * <BR><BR>
     * 暗号ワード数を取得します.
     * <BR>
     * @return int 暗号ワード数が返されます.
     */
    public int getCb32WordSize() ;
    
    /**
     * コネクションBeanを取得.
     * <BR><BR>
     * コネクションBeanを取得します.
     * <BR>
     * @return ConnectBean コネクションBeanが返されます.
     */
    public ConnectBean getConnectBean() ;
    
    /**
     * 暗号モードを設定.
     * <BR><BR>
     * 暗号モードを設定します.
     * <BR>
     * @param mode 暗号モードを設定します.
     */
    public void setCb32( boolean mode ) ;
    
    /**
     * バインド先のMACアドレスを取得.
     * <BR><BR>
     * バインド先のMACアドレスを取得します.
     * <BR>
     * @return byte[] MACアドレスが返されます.
     */
    public byte[] getMacAddress() ;
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインド時に設定されたInetAddressが返されます.
     * <BR>
     * @return InetAddress バインドInetAddressが返されます.
     */
    public InetAddress getBindInetAddress() ;
    
    /**
     * 暗号モードを取得.
     * <BR><BR>
     * 暗号モードを取得します.
     * <BR>
     * @return boolean 暗号モードが返されます.<BR>
     *                 [true]が返された場合、暗号モードはONです.<BR>
     *                 [false]が返された場合、暗号モードはOFFです.
     */
    public boolean isCb32() ;
    
    /**
     * オブジェクトが利用可能かチェック.
     * <BR><BR>
     * オブジェクトが利用可能かチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オブジェクトは利用可能です.<BR>
     *                 [false]が返された場合、オブジェクトは利用不可能です.
     */
    public boolean isUseObject() ;
    
    /**
     * 通信元オブジェクトを取得.
     * <BR><BR>
     * 通信元オブジェクトを取得します.
     * <BR>
     * @return MaachangConnect 通信元オブジェクトが返されます.
     */
    public MaachangConnect getNativeConnect() ;
    
}

