/*
 * @(#)ConnectFactory.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.connect ;

import java.net.InetAddress;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.sys.NamingManager;
import org.maachang.commons.util.SearchPortion;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;

/**
 * コネクションファクトリ.
 *  
 * @version 2006/08/30
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class ConnectFactory
{
    
    /**
     * ネーミングマネージャ登録拡張子.
     */
    public static final String NAMING_PLUS = "@maachangq.channel.connection.factory" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public ConnectFactory() {
        
    }
    
    /**
     * コネクション情報を全て破棄.
     * <BR><BR>
     * コネクション情報を全て破棄します.
     */
    public final void destroy() {
        
        synchronized( SYNC ) {
            
            int len ;
            String[] names = ConnectFactory.getNames() ;
            if( names != null && ( len = names.length ) > 0 ) {
                for( int i = 0 ; i < len ; i ++ ) {
                    try {
                        ConnectFactory.remove( names[ i ] ) ;
                    } catch( Exception e ) {
                    }
                }
            }
            
        }
    }
    
    /**
     * コネクションネットワーク生成.
     * <BR><BR>
     * 条件を設定してコネクションネットワークを生成します.
     * <BR>
     * @param bean 対象のコネクションBeanを設定します.
     * @exception Exception 例外.
     */
    public static final void create( ConnectBean bean )
        throws Exception {
        
        if( bean == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        InetAddress addr = null ;
        if( bean.getBindAddress() != null ) {
            try {
                addr = InetAddress.getByName( bean.getBindAddress() ) ;
            } catch( Exception e ) {
            }
        }
        
        ConnectFactory.create( bean.getConnectObjectName(),
            bean.getMqCacheName(),bean.getPort(),addr,bean.getBufferLength(),
            bean.isCb32(),bean.getCb32Word() ) ;
    }
    
    /**
     * コネクションネットワーク生成.
     * <BR><BR>
     * 条件を設定してコネクションネットワークを生成します.
     * <BR>
     * @param naming ネーミングサービスに登録するための名前を設定します.
     * @param mqCacheName MQキャッシュ名を設定します.
     * @param port バインド先ポートを設定します.
     * @param bindAddr バインド先アドレスを設定します.
     * @param buffer 送受信バッファ長を設定します.
     * @param cb32 対象の暗号フラグを設定します.
     * @param cb32Word 対象の暗号ワード群を設定します.
     * @exception Exception 例外.
     */
    public static final void create( String naming,String mqCacheName,int port,
        InetAddress bindAddr,int buffer,boolean cb32,String[] cb32Word )
        throws Exception {
        
        String key = null ;
        ConnectImple imple = null ;
        
        if(
            naming == null || ( naming = naming.trim().toLowerCase() ).length() <= 0 ||
            ( key = ConnectFactory.createNaming( naming ) ) == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        else if( NamingManager.get( key ) != null ) {
            throw new InputException( "対象のコネクションネットワーク名[" + naming +
                "]は既に登録されています" ) ;
        }
        
        ConnectFactory.checkBind( port,bindAddr ) ;
        
        synchronized( SYNC ) {
            
            imple = new ConnectImple( naming,mqCacheName,port,
                bindAddr,buffer,cb32,cb32Word ) ;
            NamingManager.add( key,imple ) ;
            
        }
        
    }
    
    /**
     * コネクションネットワーク削除.
     * <BR><BR>
     * コネクションネットワークオブジェクトを削除します.
     * <BR>
     * @param naming 削除対象のネーミングを設定します.
     */
    public static final void remove( String naming ) {
        
        Object o = null ;
        
        if( ( naming = ConnectFactory.createNaming( naming ) ) == null ) {
            return ;
        }
        
        synchronized( SYNC ) {
            
            o = NamingManager.get( naming ) ;
            
            if( o != null && o instanceof ConnectImple ) {
                
                NamingManager.remove( naming ) ;
                
            }
            
        }
        
    }
    
    /**
     * コネクションネットワーク取得.
     * <BR><BR>
     * コネクションネットワークオブジェクトを取得します.
     * <BR>
     * @param naming 取得対象のネーミングを設定します.
     * @return Connect コネクションネットワーク情報が返されます.
     */
    public static final Connect get( String naming ) {
        
        Object o = null ;
        Connect ret = null ;
        
        if( ( naming = ConnectFactory.createNaming( naming ) ) == null ) {
            return null ;
        }
        
        synchronized( SYNC ) {
            
            o = NamingManager.get( naming ) ;
            if( o != null && o instanceof Connect ) {
                ret = ( Connect )o ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 登録されているコネクションネットワーク名一覧を取得.
     * <BR><BR>
     * 登録されているコネクションネットワーク名一覧を取得します.
     * <BR>
     * @return String[] コネクションネットワーク名一覧が返されます.
     */
    public static final String[] getNames() {
        
        int i ;
        int len ;
        int namingLen ;
        
        String[] ret = null ;
        
        synchronized( SYNC ) {
            
            ret = SearchPortion.searchString(
                NamingManager.getNames(),
                new StringBuffer().append( "*" ).append( NAMING_PLUS ).toString() ) ;
            
            if( ret != null && ( len = ret.length ) > 0 ) {
                namingLen = NAMING_PLUS.length() ;
                for( i = 0 ; i < len ; i ++ ) {
                    
                    ret[ i ] = ret[ i ].substring( 0,ret[ i ].length() - namingLen ) ;
                    
                }
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 登録されているコネクションネットワーク数を取得.
     * <BR><BR>
     * 登録されているコネクションネットワーク数を取得します.
     * <BR>
     * @return int コネクションネットワーク数が返されます.
     */
    public static final int size() {
        
        int ret = 0 ;
        
        synchronized( SYNC ) {
            
            String[] names = SearchPortion.searchString(
                NamingManager.getNames(),
                new StringBuffer().append( "*" ).append( NAMING_PLUS ).toString() ) ;
            
            if( names != null ) {
                ret = names.length ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Object 同期オブジェクトが返されます.
     */
    public static final Object getSync() {
        return SYNC ;
    }
    
    
    
    /**
     * コネクションネットワークファクトリ名生成.
     */
    private static final String createNaming( String name ) {
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return null ;
        }
        
        return new StringBuffer().append( name ).append( NAMING_PLUS ).toString() ;
        
    }
    
    /**
     * 既に同一のバインド条件があるかチェック.
     */
    private static final void checkBind( int port,InetAddress addr )
        throws Exception {
        synchronized( SYNC ) {
            String[] names = ConnectFactory.getNames() ;
            if( names != null && names.length > 0 ) {
                int len = names.length ;
                
                if( addr == null ) {
                    
                    for( int i = 0 ; i < len ; i ++ ) {
                        
                        Connect conn = ConnectFactory.get( names[ i ] ) ;
                        if( conn.getBindInetAddress() == null &&
                            conn.getPort() == port ) {
                            throw new MaachangQException(
                                "指定バインド条件は既に存在します",
                                MaachangQErrorCode.ERROR_USE_BIND_CONNECT ) ;
                        }
                        
                    }
                    
                }
                else {
                    
                    for( int i = 0 ; i < len ; i ++ ) {
                        
                        Connect conn = ConnectFactory.get( names[ i ] ) ;
                        if( addr.equals( conn.getBindInetAddress() ) == true &&
                            conn.getPort() == port ) {
                            throw new MaachangQException(
                                "指定バインド条件は既に存在します",
                                MaachangQErrorCode.ERROR_USE_BIND_CONNECT ) ;
                        }
                        
                    }
                    
                }
            }
        }
    }
}

