/*
 * @(#)BaseQueueManagerImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.manager ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.array.ObjectArray;
import org.maachang.queue.access.status.QueueManagerStatus;
import org.maachang.queue.config.MqDefine;
import org.maachang.queue.main.MqUtil;
import org.maachang.queue.main.cache.MqCache;
import org.maachang.queue.main.cache.MqCacheFactory;
import org.maachang.queue.main.queue.Queue;
import org.maachang.queue.main.queue.QueueBean;
import org.maachang.queue.main.queue.QueueKey;
import org.maachang.queue.main.queue.base.BaseQueue;
import org.maachang.queue.main.queue.base.BaseQueueBean;
import org.maachang.queue.main.queue.base.BaseQueueFactory;

/**
 * キューマネージャオブジェクト.
 * <BR><BR>
 * １つのキューマネージャを表すオブジェクトです.
 * 
 * @version 2006/09/03
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class BaseQueueManagerImple implements BaseQueueManager {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( BaseQueueManagerImple.class ) ;
    
    /**
     * キューマネージャ名.
     */
    private String name = null ;
    
    /**
     * キューマネージャキャッシュ名.
     */
    private String cacheName = null ;
    
    /**
     * キューステータス.
     */
    private int state = QueueManagerStatus.STATE_DOWN ;
    
    /**
     * キュー管理テーブル.
     */
    private ObjectArray table = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    private BaseQueueManagerImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * キューマネージャを生成します.
     * <BR>
     * @param name 対象のキューマネージャ名を設定します.
     * @param size キャッシュセクタ数を設定します.<BR>
     *             設定可能な最小値は[MqDefine.MIN_CACHE_SECTOR]です.<BR>
     *             設定可能な最大値は[MqDefine.MAX_CACHE_SECTOR]です.<BR>
     *             [-1]を設定した場合、キャッシュファイルは生成されません.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public BaseQueueManagerImple( String name,int size )
        throws InputException,AccessException {
        
        String cacheName = null ;
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( size < -1 ) {
            size = -1 ;
        }
        else if( size <= MqDefine.MIN_CACHE_SECTOR ) {
            size = MqDefine.MIN_CACHE_SECTOR ;
        }
        else if( size >= MqDefine.MAX_CACHE_SECTOR ) {
            size = MqDefine.MAX_CACHE_SECTOR ;
        }
        
        sync.create() ;
        
        try {
            
            synchronized( sync.get() ) {
                
                if( size >= 0 ) {
                    cacheName = MqUtil.replaceString(
                        MqDefine.SEND_QUEUE_CACHE,
                        MqDefine.REPLACE_NAME_QUEUE_MANAGER,
                        name
                    ) ;
                    
                    MqCache cache = MqCacheFactory.get( cacheName ) ;
                    
                    if( cache == null ) {
                        MqCacheFactory.create( cacheName,size ) ;
                    }
                    else if( cache.getCacheSize() != size ) {
                        MqCacheFactory.remove( true,cacheName ) ;
                        MqCacheFactory.create( cacheName,size ) ;
                    }
                }
                
                this.name = name ;
                this.cacheName = cacheName ;
                this.state = QueueManagerStatus.STATE_STARTUP ;
                this.table = new ObjectArray() ;
                
            }
            
        } catch( AccessException ae ) {
            this.destroy() ;
            throw ae ;
        } catch( Exception e ) {
            this.destroy() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try {
            this.destroy() ;
        } catch( Exception t ) {
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * 対象のオブジェクトを破棄します.
     */
    public void destroy() {
        
        int i ;
        int len ;
        
        QueueKey em = null ;
        
        try {
            synchronized( sync.get() ) {
                
                if( table != null && ( len = table.size() ) > 0 ) {
                    
                    for( i = 0 ; i < len ; i ++ ) {
                        if( ( em = ( QueueKey )table.get( i ) ) != null ) {
                            BaseQueueFactory.delete(
                                false,em.getQueueType(),name,em.getName() ) ;
                        }
                    }
                    
                    table.clear() ;
                    
                }
                
                if( MqCacheFactory.get( cacheName ) != null ) {
                    MqCacheFactory.remove( cacheName ) ;
                }
                
                table = null ;
                name = null ;
                cacheName = null ;
                state = QueueManagerStatus.STATE_DOWN ;
                
            }
            
        } catch( Exception e ) {
        }
        
        sync.clear() ;
        
        table = null ;
        name = null ;
        cacheName = null ;
        state = QueueManagerStatus.STATE_DOWN ;
        
    }
    
    /**
     * キャッシュ情報を完全削除してオブジェクト破棄.
     * <BR><BR>
     * キャッシュ情報を完全破棄してオブジェクトを破棄します.
     */
    protected void deleteCacheToDestroy() {
        
        try { 
            synchronized( sync.get() ) {
                MqCacheFactory.remove(
                    true,cacheName
                ) ;
                this.destroy() ;
            }
            
        } catch( Exception e ) {
        }
        
        this.destroy() ;
    }
    
    
    /**
     * 新しいQueueBeanオブジェクトを生成.
     * <BR><BR>
     * 新しいQueueBeanオブジェクトを生成します.
     * <BR>
     * @param queueType 対象のキュータイプを設定します.
     * @param queueName 対象のキュー名を設定します.
     * @return AdminQueueBean QueueBeanオブジェクトが返されます.
     * @exception InputException 入力例外.
     */
    public QueueBean createQueueBean( int queueType,String queueName )
        throws InputException {
        
        BaseQueueBean ret = null ;
        
        if(
            queueName == null ||
            ( queueName = queueName.trim().toLowerCase() ).length() <= 0 ) {
            
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            synchronized( sync.get() ) {
                
                if( BaseQueueFactory.get( queueType,name,queueName ) != null ) {
                    throw new InputException(
                        "既にキュー名[queueType:" + queueType +
                        " queueManagerName:" + name +
                        " queueName:" + queueName +
                        "]は存在します" ) ;
                }
                
                ret = new BaseQueueBean() ;
                ret.setType( queueType ) ;
                ret.setQueueName( queueName ) ;
                ret.setQueueManagerName( name ) ;
                
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キューマネージャ名を取得.
     * <BR><BR>
     * キューマネージャ名を取得します.
     * <BR>
     * @return String キューマネージャ名が返されます.
     */
    public String getName() {
        
        String ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.name ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュ名を取得.
     * <BR><BR>
     * キャッシュ名を取得します.
     * <BR>
     * @return String キャッシュ名が返されます.<BR>
     *                [null]が返された場合、キャッシュは利用しません.
     */
    public String getCacheName() {
        
        String ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.cacheName ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * リソースタイプを取得.
     * <BR><BR>
     * キューマネージャのリソースタイプが返されます.
     * <BR>
     * @return ResourceType リソースタイプが返されます.<BR>
     *                      [null]が返された場合、キャッシュは利用しません.
     */
    public ResourceType getResourceType() {
        
        ResourceType ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = MqCacheFactory.
                    get( cacheName ).
                    getResourceType() ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    
    /**
     * キャッシュオブジェクトを取得.
     * <BR><BR>
     * このキューマネージャに割り当てられているキャッシュ情報を取得します.
     * <BR>
     * @return MqCache キャッシュオブジェクトが返されます.<BR>
     *                 [null]が返された場合、キャッシュは利用しません.
     */
    public MqCache getMqCache() {
        
        MqCache ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = MqCacheFactory.get( cacheName ) ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キューマネージャステータスを取得.
     * <BR><BR>
     * 対象のキューマネージャステータスを取得します.
     * <BR>
     * @param state 対象のキューマネージャステータス情報を設定します.
     */
    public void setState( int state ) {
        
        if(
            state != QueueManagerStatus.STATE_SUCCESS &&
            state != QueueManagerStatus.STATE_DOWN &&
            state != QueueManagerStatus.STATE_STARTUP &&
            state != QueueManagerStatus.STATE_SHUTDOWN &&
            state != QueueManagerStatus.STATE_ERROR &&
            state != QueueManagerStatus.STATE_PLANNED_STOP
        ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                if( this.state != state ) {
                    LOG.info(
                        new StringBuffer().
                        append( "キューマネージャ(" ).
                        append( name ).
                        append( ")のステータスは " ).
                        append( QueueManagerStatus.getStateByString( this.state ) ).
                        append( " から " ).
                        append( QueueManagerStatus.getStateByString( state ) ).
                        append( " に変更されました" ).
                        toString() ) ;
                    this.state = state ;
                }
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * キューマネージャステータスを取得.
     * <BR><BR>
     * 対象のキューマネージャステータスを取得します.
     * <BR>
     * @return int 対象のキューマネージャステータス情報が返されます.<BR>
     *             [STATE_SUCCESS]が返された場合、「正常」です.<BR>
     *             [STATE_DOWN]が返された場合、「停止」です.<BR>
     *             [STATE_STARTUP]が返された場合、「起動中」です.<BR>
     *             [STATE_SHUTDOWN]が返された場合、「停止」です.<BR>
     *             [STATE_ERROR]が返された場合、「異常」です.<BR>
     *             [STATE_PLANNED_STOP]が返された場合、「計画停止」です.
     */
    public int getState() {
        
        int ret ;
        
        try {
            synchronized( sync.get() ) {
                ret = state ;
            }
        } catch( Exception e ) {
            ret = QueueManagerStatus.STATE_DOWN ;
        }
        
        return ret ;
        
    }
    
    /**
     * キュー情報を追加.
     * <BR><BR>
     * キュー情報を、このマネージャに追加します.
     * <BR>
     * @param queue 追加対象のキューオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public void addQueue( Queue queue )
        throws InputException {
        
        QueueKey key = null ;
        
        if(
            queue == null || ( key = queue.getKey() ) == null ||
            ( queue instanceof BaseQueue ) == false
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            synchronized( sync.get() ) {
                
                table.add( key ) ;
                
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * キュー情報を削除.
     * <BR><BR>
     * キュー情報を、このマネージャから削除します.
     * <BR>
     * @param key 削除対象のKey要素情報を設定します.
     */
    public void removeQueue( QueueKey key ) {
        
        if( key == null ) {
            return ;
        }
        
        try {
            
            synchronized( sync.get() ) {
                
                this.removeQueue(
                    true,
                    key.getQueueType(),
                    key.getName()
                ) ;
                
            }
            
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * キュー情報を削除.
     * <BR><BR>
     * キュー情報を、このマネージャから削除します.
     * <BR>
     * @param queueType 削除対象のキュータイプを設定します.
     * @param name 削除対象のキュー名を設定します.
     */
    public void removeQueue( int queueType,String name ) {
        
        try {
            
            synchronized( sync.get() ) {
                
                this.removeQueue(
                    true,
                    queueType,
                    name
                ) ;
                
            }
            
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * キュー情報を削除.
     * <BR><BR>
     * キュー情報を、このマネージャから削除します.
     * <BR>
     * @param mode 削除時にキュー情報も削除するか設定します.<BR>
     *             [true]を設定した場合、キュー情報も削除します.<BR>
     *             [false]を設定した場合、キュー情報は削除しません.
     * @param key 削除対象のKey要素情報を設定します.
     */
    public void removeQueue( boolean mode,QueueKey key ) {
        
        if( key == null ) {
            return ;
        }
        
        try {
            
            synchronized( sync.get() ) {
                
                this.removeQueue(
                    mode,
                    key.getQueueType(),
                    key.getName()
                ) ;
                
            }
            
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * キュー情報を削除.
     * <BR><BR>
     * キュー情報を、このマネージャから削除します.
     * <BR>
     * @param mode 削除時にキュー情報も削除するか設定します.<BR>
     *             [true]を設定した場合、キュー情報も削除します.<BR>
     *             [false]を設定した場合、キュー情報は削除しません.
     * @param queueType 削除対象のキュータイプを設定します.
     * @param name 削除対象のキュー名を設定します.
     */
    public void removeQueue( boolean mode,int queueType,String name ) {
        
        int i ;
        int len ;
        
        QueueKey em = null ;
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                
                if( table != null && ( len = table.size() ) > 0 ) {
                    for( i = 0 ; i < len ; i ++ ) {
                        
                        if(
                            ( em = ( QueueKey )table.get( i ) ) != null &&
                            em.getQueueType() == queueType &&
                            name.equals( em.getName() )
                        ) {
                            
                            table.remove( i ) ;
                            
                            if( mode == true ) {
                                BaseQueueFactory.delete(
                                    false,queueType,this.name,name ) ;
                            }
                            
                            break ;
                            
                        }
                        
                    }
                    
                }
                    
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * キュー情報を取得.
     * <BR><BR>
     * キュー情報を、このマネージャから取得します.
     * <BR>
     * @param key 対象のKey要素情報を設定します.
     */
    public Queue getQueue( QueueKey key ) {
        
        Queue ret = null ;
        
        if( key == null ) {
            return null ;
        }
        
        try {
            
            synchronized( sync.get() ) {
                
                ret = this.getQueue(
                    key.getQueueType(),
                    key.getName()
                ) ;
                
            }
            
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キュー情報を取得.
     * <BR><BR>
     * キュー情報を、このマネージャから取得します.
     * <BR>
     * @param queueType 対象のキュータイプを設定します.
     * @param name 対象のキュー名を設定します.
     */
    public Queue getQueue( int queueType,String name ) {
        
        Queue ret = null ;
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return null ;
        }
        
        try {
            synchronized( sync.get() ) {
                
                ret = BaseQueueFactory.get(
                    queueType,this.name,name ) ;
                    
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キュー数を取得.
     * <BR><BR>
     * このマネージャで管理しているキュー数を取得.
     * <BR>
     * @return int キュー数が返されます.
     */
    public int getQueueSize() {
        
        int ret ;
        
        try {
            synchronized( sync.get() ) {
                ret = table.size() ;
            }
        } catch( Exception e ) {
            ret = 0 ;
        }
        
        return ret ;
        
    }
    
    /**
     * キュー数を取得.
     * <BR><BR>
     * このマネージャで管理しているキュー数を取得.
     * <BR>
     * @param queueType 対象のキュータイプを設定します.
     * @return int キュー数が返されます.
     */
    public int getQueueSize( int queueType ) {
        
        int i ;
        int len ;
        int ret = 0 ;
        
        QueueKey em = null ;
        
        try {
            synchronized( sync.get() ) {
                
                len = table.size() ;
                for( i = 0,ret = 0 ; i < len ; i ++ ) {
                    if(
                        ( em = ( QueueKey )table.get( i ) ) != null &&
                        em.getQueueType() == queueType
                    ) {
                        ret ++ ;
                    }
                }
                
            }
            
        } catch( Exception e ) {
            ret = 0 ;
        }
        
        return ret ;
        
    }
    
    /**
     * Key要素情報一覧を取得.
     * <BR><BR>
     * このマネージャで管理しているKey要素情報一覧を取得.
     * <BR>
     * @return QueueKey[] Key要素一覧が返されます.
     */
    public QueueKey[] getKeyElements() {
        
        int len ;
        
        QueueKey[] ret = null ;
        
        try {
            synchronized( sync.get() ) {
                
                if( table != null && ( len = table.size() ) > 0 ) {
                    ret = new QueueKey[ len ] ;
                    System.arraycopy( table.getObjects(),0,ret,0,len ) ;
                }
                
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * Key要素情報一覧を取得.
     * <BR><BR>
     * このマネージャで管理しているKey要素情報一覧を取得.
     * <BR>
     * @param queueType 対象のキュータイプを設定します.
     * @return QueueKey[] Key要素一覧が返されます.
     */
    public QueueKey[] getKeyElements( int queueType ) {
        
        int i ;
        int len ;
        int cnt ;
        int sz ;
        
        QueueKey em = null ;
        QueueKey[] ret = null ;
        
        try {
            synchronized( sync.get() ) {
                
                if(
                    table != null && ( len = table.size() ) > 0 &&
                    ( sz = this.getQueueSize( queueType ) ) > 0
                ) {
                    ret = new QueueKey[ sz ] ;
                    for( i = 0,cnt = 0 ; i < len ; i ++ ) {
                        if(
                            ( em = ( QueueKey )table.get( i ) ) != null &&
                            em.getQueueType() == queueType
                        ) {
                            ret[ cnt ] = em ;
                            cnt ++ ;
                        }
                    }
                    
                }
                
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * QueueManagerBeanを取得.
     * <BR><BR>
     * このマネージャのQueueManagerBean情報を取得します.
     * <BR>
     * @return AdminQueueManagerBean QueueManagerBean情報が返されます.
     */
    public QueueManagerBean getQueueManagerBean() {
        
        int i ;
        int len ;
        
        String[] names = null ;
        MqCache cache = null ;
        QueueManagerBean ret = null ;
        
        try {
            synchronized( sync.get() ) {
                
                ret = new QueueManagerBean() ;
                ret.setQueueManagerName( this.name ) ;
                
                cache = MqCacheFactory.get( this.name ) ;
                
                if( cache == null ) {
                    ret.setCacheSize( -1 ) ;
                }
                else {
                    ret.setCacheSize( cache.getCacheSize() ) ;
                }
                
                if( table != null && ( len = table.size() ) > 0 ) {
                    names = new String[ len ] ;
                    for( i = 0 ; i < len ; i ++ ) {
                        names[ i ] = ( String )this.table.get( i ) ;
                    }
                }
                
                ret.setQueueNames( names ) ;
                
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キューマネージャオブジェクトが有効であるか取得.
     * <BR><BR>
     * キューマネージャオブジェクトが有効であるか取得します.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public boolean isQueueManager() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                ret = ( name != null && name.length() > 0 ) ?
                    true : false ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
}

