/*
 * @(#)AbstractReceiveMq.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue ;

import org.maachang.queue.access.MaachangQException;


/**
 * 受信MQを表すオブジェクト.
 * <BR><BR>
 * 1つの受信MQを表すオブジェクトです.
 *  
 * @version 2006/12/21
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
abstract class AbstractReceiveMq extends AbstractSendMq
    implements ReceiveMqInterface,SendMqInterface {
    
    /**
     * デフォルトタイムアウト値.
     */
    private static final int DEF_TIMEOUT = 50 ;
    
    /**
     * メッセージ情報を取得.
     * <BR><BR>
     * 対象のメッセージ情報を取得します.<BR>
     * また取得できた場合、取得された情報は削除されます.
     * <BR>
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param transactionId 対象のトランザクションIDを設定します.
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @return boolean 取得結果が返されます.<BR>
     *                  [true]が返された場合、情報取得に成功しました.<BR>
     *                  [false]が返された場合、情報取得に失敗しました.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean get( QueueValue out,int[] transactionId,String key )
        throws MaachangQException {
        
        int connId = -1 ;
        if( transactionId != null ) {
            connId = transactionId[ 0 ] ;
            this.checkTransaction( connId ) ;
        }
        super.checkState() ;
        
        int resultId = super.queue.get( connId,out,key,DEF_TIMEOUT ) ;
        
        // 情報が取得できない場合.
        if( resultId < 0 ) {
            return false ;
        }
        // 情報取得成功後、オートコミットでなく、現在の接続処理の、
        // トランザクションIDが存在しない場合.
        else if( transactionId != null  &&
            connId == Queue.TRANSACTION_BY_ALREADY_EXISTENCE ) {
            // 新しいトランザクションIDを設定.
            transactionId[ 0 ] = resultId ;
        }
        
        return true ;
        
    }
    
    /**
     * メッセージ情報取得後、正常に情報を取得できた場合の処理.
     * <BR><BR>
     * メッセージ情報取得後、正常に情報を取得できた場合の処理します.
     * <BR>
     * @param transactionId 対象のトランザクションIDを設定します.
     * @param value 対象の１つのキューデータ情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void successGet( int[] transactionId,QueueValue value )
        throws MaachangQException {
        
        if( value == null ) {
            return ;
        }
        
        int connId = -1 ;
        if( transactionId != null ) {
            connId = transactionId[ 0 ] ;
            this.checkTransaction( connId ) ;
        }
        super.checkState() ;
        
        super.queue.successGet( connId,value ) ;
        
    }
    
    /**
     * get処理ができるかチェック.
     * <BR><BR>
     * get処理ができるかチェックします.
     * <BR>
     * @param transactionId 対象のトランザクションIDを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    private void checkTransaction( int transactionId )
        throws MaachangQException {
        if( super.isAutoCommit() == false ) {
            
            if( super.getTransactionId() == Queue.TRANSACTION_BY_ALREADY_EXISTENCE ) {
                // 現在条件に対してトランザクションIDが存在しない場合.
                return ;
            }
            else if( super.getTransactionId() != transactionId ) {
                throw new AlreadyExistenceException(
                    "他のコネクションがトランザクションを保持しています" ) ;
            }
        }
    }
    
}

