/*
 * @(#)Mq.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue ;

import org.maachang.commons.thread.Synchronized;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.main.manager.QueueManager;

/**
 * MQを表すインターフェイス.
 * <BR><BR>
 * 1つのMQを表すインターフェイスです.
 *  
 * @version 2006/08/29
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public interface Mq {
    
    /**
     * トランザクションを開始.
     * <BR><BR>
     * トランザクションを開始します.
     * <BR>
     * @return int トランザクションIDが返されます.<BR>
     *             [Queue.TRANSACTION_BY_AUTOCOMMIT]が返された場合、
     *             このキューはオートコミットです.
     * @exception MaachangQException MaachangQ例外.
     */
    public int transaction()
        throws MaachangQException ;
    
    /**
     * コミット処理.
     * <BR><BR>
     * トランザクションに対する、コミット処理を実施します.
     * <BR>
     * @param transactionId 対象のトランザクションIDを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void commit( int transactionId )
        throws MaachangQException ;
    
    /**
     * ロールバック処理.
     * <BR><BR>
     * トランザクションに対する、ロールバック処理を実施します.
     * <BR>
     * @param transactionId 対象のトランザクションIDを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void rollback( int transactionId )
        throws MaachangQException ;
    
    /**
     * キューマネージャオブジェクトを取得.
     * <BR><BR>
     * キューマネージャオブジェクトを取得します.
     * <BR>
     * @return QueueManager キューマネージャオブジェクトが返されます.
     */
    public QueueManager getQueueManager() ;
    
    /**
     * 対象QueueBean情報を設定.
     * <BR><BR>
     * 対象QueueBean情報を設定します.
     * <BR>
     * @param bean 設定対象のQueueBeanを設定します.
     */
    public void setQueueBean( QueueBean bean ) ;
    
    /**
     * QueueBeanオブジェクトを取得.
     * <BR><BR>
     * QueueBeanオブジェクトを取得します.
     * <BR>
     * @return AdminQueueBean QueueBeanオブジェクトが返されます.
     */
    public QueueBean getQueueBean() ;
    
    /**
     * QueueKeyオブジェクトを取得.
     * <BR><BR>
     * QueueKeyオブジェクトを取得します.
     * <BR>
     * @return QueueKey QueueKeyオブジェクトが返されます.
     */
    public QueueKey getQueueKey() ;
    
    /**
     * 現在のMQステータスを取得.
     * <BR><BR>
     * 現在のMQステータスを取得します.
     * <BR>
     * @return int MQステータスが返されます.<BR>
     *             [STATE_SUCCESS]が返された場合、「正常」です.<BR>
     *             [STATE_DOWN]が返された場合、「停止」です.<BR>
     *             [STATE_STARTUP]が返された場合、「起動中」です.<BR>
     *             [STATE_SHUTDOWN]が返された場合、「停止」です.<BR>
     *             [STATE_ERROR]が返された場合、「異常」です.<BR>
     *             [STATE_WARNING]が返された場合、「キュー格納数指定量による警告」です.<BR>
     *             [STATE_FULL]が返された場合、「キュー満杯」です.
     */
    public int getState() ;
    
    /**
     * 現在の格納電文数を取得.
     * <BR><BR>
     * 現在の格納電文数を取得します.
     * <BR>
     * @return int 格納されている電文数が返されます.
     */
    public int size() ;
    
    /**
     * 現在の最大キュー数を取得.
     * <BR>BR>
     * 現在の最大キュー数を取得します.
     * <BR>
     * @return int 現在の最大キュー数が返されます.
     */
    public int getMaxSize() ;
    
    /**
     * 現在のトランザクションIDを取得.
     * <BR><BR>
     * 現在のトランザクションIDを取得します.
     * <BR>
     * @return int 現在のトランザクションIDが返されます.
     */
    public int getTransactionId() ;
    
    /**
     * キュー同期オブジェクトを取得.
     * <BR><BR>
     * キュー同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized キュー同期オブジェクトが返されます.
     */
    public Synchronized getSynchronized() ;
    
    /**
     * このキューがオートコミットであるかチェック.
     * <BR><BR>
     * このキューがオートコミットであるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オートコミットです.<BR>
     *                 [false]が返された場合、オートコミットではありません.
     */
    public boolean isAutoCommit() ;
    
    /**
     * MQが有効であるかチェック.
     * <BR><BR>
     * MQが有効であるかチェックします.<BR>
     * またこのオブジェクトの有効性を見るだけでない、
     * Queueが登録されていない場合も、無効扱いとなります.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、MQは有効です.<BR>
     *                 [false]が返された場合、MQは無効です.
     */
    public boolean isUseMq() ;
    
    /**
     * 対象キューに設定可能かチェック.
     * <BR><BR>
     * 対象キューに設定可能かチェックします.
     * <BR>
     * @param len 対象の電文長を設定します.
     * @exception MessageOverException メッセージオーバー例外.
     * @exception QueueFullException キューフル例外.
     */
    public void isPut( int len )
        throws MessageOverException,QueueFullException ;
    
}

