/*
 * @(#)MqManager.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.sys.NamingManager;
import org.maachang.commons.util.CharTable;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.main.manager.BaseQueueManager;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.queue.base.BaseQueueFactory;

/**
 * MQ管理オブジェクト.
 *  
 * @version 2006/12/21
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class MqManager {
    
    /**
     * ネーミングマネージャ登録拡張子.
     */
    public static final String NAMING_MANAGER = "MANAGER@maachangq.queue" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public MqManager() {
        
    }
    
    /**
     * Mqプーリング初期化処理.
     */
    public final void init() {
        
        MqManagerTable table = new MqManagerTable() ;
        
        // ネーミングマネージャに登録.
        synchronized( SYNC ) {
            
            NamingManager.add( NAMING_MANAGER,table ) ;
            
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy() {
        
        synchronized( SYNC ) {
            
            MqManagerTable table =
                ( MqManagerTable )NamingManager.get( NAMING_MANAGER ) ;
            if( table != null ) {
                table.destroy() ;
            }
            
            NamingManager.remove( NAMING_MANAGER ) ;
            
        }
    }
    
    /**
     * キュー情報を生成.
     * <BR><BR>
     * 対象キュー情報を生成します.
     * <BR>
     * @param bean 生成対象のQueueBeanを設定します.
     * @return Mq 対象のキュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static Mq create( QueueBean bean )
        throws MaachangQException {
        synchronized( SYNC ) {
            MqManagerTable tbl = MqManager.getTable() ;
            if( tbl != null ) {
                return tbl.create( bean ) ;
            }
        }
        return null ;
    }
    
    /**
     * キュー情報を削除.
     * <BR><BR>
     * キュー情報を削除します.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static void delete( String manager,String queue,int queueType )
        throws MaachangQException {
        synchronized( SYNC ) {
            MqManagerTable tbl = MqManager.getTable() ;
            if( tbl != null ) {
                tbl.delete( manager,queue,queueType ) ;
            }
        }
    }
    
    /**
     * キュー情報を取得.
     * <BR><BR>
     * キュー情報を取得します.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @return Mq 対象のキュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static Mq get( String manager,String queue,int queueType )
        throws MaachangQException {
        return MqManager.get( false,manager,queue,queueType ) ;
    }
    
    /**
     * キュー情報を取得.
     * <BR><BR>
     * キュー情報を取得します.
     * <BR>
     * @param mode 状態無視状態で情報を取得するモード.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @return Mq 対象のキュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static Mq get( boolean mode,String manager,String queue,int queueType )
        throws MaachangQException {
        synchronized( SYNC ) {
            MqManagerTable tbl = MqManager.getTable() ;
            if( tbl != null ) {
                return tbl.get( mode,manager,queue,queueType ) ;
            }
        }
        return null ;
    }
    
    /**
     * 存在MQ情報一覧を取得.
     * <BR><BR>
     * 存在するMQ情報一覧を取得します.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @return QueueKey[] 対象のQueueKeyが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static QueueKey[] getQueues( String manager )
        throws MaachangQException {
        return MqManager.getQueues( false,manager ) ;
    }
    
    /**
     * 存在MQ情報一覧を取得.
     * <BR><BR>
     * 存在するMQ情報一覧を取得します.
     * <BR>
     * @param mode 状態無視状態で情報を取得するモード.
     * @param manager 対象のキューマネージャ名を設定します.
     * @return QueueKey[] 対象のQueueKeyが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static QueueKey[] getQueues( boolean mode,String manager )
        throws MaachangQException {
        synchronized( SYNC ) {
            MqManagerTable tbl = MqManager.getTable() ;
            if( tbl != null ) {
                return tbl.getQueues( mode,manager ) ;
            }
        }
        return null ;
    }
    
    /**
     * キューサイズを取得.
     * <BR><BR>
     * 存在するキューサイズを取得します.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @return int 対象のキュー数が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static int size( String manager )
        throws MaachangQException {
        synchronized( SYNC ) {
            MqManagerTable tbl = MqManager.getTable() ;
            if( tbl != null ) {
                return tbl.size( manager ) ;
            }
        }
        return 0 ;
    }
    
    /**
     * 対象MQが存在するかチェック.
     * <BR><BR>
     * 対象MQが存在するかチェックします.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー情報を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @return boolean チェック結果が返されます.
     */
    public static boolean isMq( String manager,String queue,int queueType ) {
        synchronized( SYNC ) {
            MqManagerTable tbl = MqManager.getTable() ;
            if( tbl != null ) {
                return tbl.isMq( manager,queue,queueType ) ;
            }
        }
        return false ;
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Object 同期オブジェクトが返されます.
     */
    public static final Object getSync() {
        return SYNC ;
    }
    
    /**
     * テーブルオブジェクトを取得.
     */
    private static final MqManagerTable getTable() {
        return ( MqManagerTable )NamingManager.get( NAMING_MANAGER ) ;
    }
}

/**
 * プーリングデータ管理テーブル.
 */
class MqManagerTable {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( MqManagerTable.class ) ;
    
    /**
     * 区切り名.
     */
    private static final String CUT_NAME = "@" ;
    
    /**
     * プーリング管理インデックステーブル.
     */
    private CharTable table = null ;
    
    
    
    /**
     * コンストラクタ.
     */
    public MqManagerTable() {
        this.table = new CharTable() ;
        
        LOG.info( "#### Mqマネージャ生成処理 ####" ) ;
        
    }
    
    /**
     * オブジェクト終了時の処理.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public void destroy() {
        
        LOG.info( "#### Mqマネージャ破棄処理 ####" ) ;
        this.table = null  ;
        
    }
    
    /**
     * キュー情報を生成.
     * <BR><BR>
     * 対象キュー情報を生成します.
     * <BR>
     * @param bean 生成対象のQueueBeanを設定します.
     * @return Mq 対象のキュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public Mq create( QueueBean bean )
        throws MaachangQException {
        
        if( bean == null ) {
            return null ;
        }
        
        this.checkQueue(
            true,bean.getType(),
            bean.getQueueManagerName(),
            bean.getQueueName() ) ;
        
        Mq mq = null ;
        
        switch( bean.getType() ) {
            case QueueDef.TYPE_SEND :
                mq = new SendMqImple( bean ) ;
                break ;
            case QueueDef.TYPE_RECEIVE :
                mq = new ReceiveMqImple( bean ) ;
                break ;
        }
        
        String key = this.getIndexKey(
            bean.getType(),bean.getQueueManagerName(),
            bean.getQueueName() ) ;
        
        table.add( key,mq ) ;
        
        return mq ;
    }
    
    /**
     * キュー情報を削除.
     * <BR><BR>
     * キュー情報を削除します.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void delete( String manager,String queue,int queueType )
        throws MaachangQException {
        
        this.checkQueue( false,queueType,manager,queue ) ;
        
        String key = this.getIndexKey( queueType,manager,queue ) ;
        
        BaseQueueFactory.delete( queueType,manager,queue ) ;
        Mq mq = ( Mq )table.remove( key ) ;
        if( mq != null ) {
            ( ( AbstractMq )mq ).destroy() ;
        }
        
        QueueManager man = QueueManagerFactory.get( manager ) ;
        if( man != null ) {
            ( ( BaseQueueManager )man ).removeQueue(
                queueType,queue ) ;
        }
        
    }
    
    /**
     * キュー情報を取得.
     * <BR><BR>
     * キュー情報を取得します.
     * <BR>
     * @param mode 状態無視状態で情報を取得するモード.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @return Mq 対象のキュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public Mq get( boolean mode,String manager,String queue,int queueType )
        throws MaachangQException {
        
        if( mode == false ) {
            this.checkQueue( false,queueType,manager,queue ) ;
        }
        
        String key = this.getIndexKey( queueType,manager,queue ) ;
        
        Mq mq = ( Mq )table.get( key ) ;
        if( mq == null ) {
            QueueManager man = QueueManagerFactory.get( manager ) ;
            switch( queueType ) {
                case QueueDef.TYPE_SEND :
                    mq = new SendMqImple( man,queue ) ;
                    break ;
                case QueueDef.TYPE_RECEIVE :
                    mq = new ReceiveMqImple( man,queue ) ;
                    break ;
            }
            
            table.add( key,mq ) ;
        }
        
        if( mode == false ) {
            ( ( AbstractMq )mq ).checkState() ;
        }
        
        return mq ;
    }
    
    /**
     * 存在MQ情報一覧を取得.
     * <BR><BR>
     * 存在するMQ情報一覧を取得します.
     * <BR>
     * @param mode 状態無視状態で情報を取得するモード.
     * @param manager 対象のキューマネージャ名を設定します.
     * @return QueueKey[] 対象のQueueKeyが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public QueueKey[] getQueues( boolean mode,String manager )
        throws MaachangQException {
        
        if( mode == false ) {
            this.checkManager( manager ) ;
        }
        
        QueueManager man = QueueManagerFactory.get( manager ) ;
        return man.getKeyElements() ;
        
    }
    
    /**
     * キューサイズを取得.
     * <BR><BR>
     * 存在するキューサイズを取得します.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @return int 対象のキュー数が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public int size( String manager )
        throws MaachangQException {
        
        this.checkManager( manager ) ;
        
        QueueManager man = QueueManagerFactory.get( manager ) ;
        return man.getQueueSize() ;
        
    }
    
    /**
     * 対象MQが存在するかチェック.
     * <BR><BR>
     * 対象MQが存在するかチェックします.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー情報を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @return boolean チェック結果が返されます.
     */
    public boolean isMq( String manager,String queue,int queueType ) {
        return BaseQueueFactory.isQueue( queueType,manager,queue ) ;
    }
    
    /**
     * 対象キューマネージャが存在するかチェック.
     */
    private boolean isManager( String manager ) {
        return ( QueueManagerFactory.get( manager ) == null ) ?
            false : true ;
    }
    
    /**
     * インデックスKey名を取得.
     */
    private String getIndexKey( int type,String manager,String name ) {
        
        return new StringBuffer().
            append( type ).
            append( CUT_NAME ).
            append( CUT_NAME ).
            append( manager ).
            append( CUT_NAME ).
            append( name ).
            toString() ;
            
    }
    
    /**
     * キュー存在/非存在チェック.
     */
    private final void checkQueue( boolean mode,int type,String manager,String name )
        throws MaachangQException {
        
        this.checkManager( manager ) ;
        
        if( this.isMq( manager,name,type ) == mode ) {
            
            String result = ( mode == true ) ?
                "既に存在します" : "存在しません" ;
            String sendReceive = ( type == QueueDef.TYPE_SEND ) ?
                "送信" : "受信" ;
            
            int error = ( mode == true ) ?
                MaachangQErrorCode.ERROR_USE_QUEUE :
                MaachangQErrorCode.ERROR_NOT_QUEUE ;
            
            throw new MaachangQException(
                "対象の" + sendReceive + "キュー(manager:" + manager +
                " queue:" + name +
                ")は" + result,error ) ;
        }
        
    }
    
    /**
     * キューマネージャ存在チェック.
     */
    private final void checkManager( String manager ) 
        throws MaachangQException {
        
        if( this.isManager( manager ) == false ) {
            throw new MaachangQException(
                "対象キューマネージャ名(" + manager +
                ")は存在しません",
                MaachangQErrorCode.ERROR_NOT_QUEUE_MANAGER ) ;
        }
        
    }
}
