/*
 * @(#)Queue.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue ;

import org.maachang.commons.exception.InputException;
import org.maachang.queue.main.queue.base.BaseQueueKey;

/**
 * １つのキューを表すインターフェイス.
 * <BR><BR>
 * 1つのキュー情報を表すインターフェイスです.
 *  
 * @version 2005/12/29
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public interface Queue
{
    
    /**
     * トランザクション条件 : オートコミット.
     */
    public static final int TRANSACTION_BY_AUTOCOMMIT = 0 ;
    
    /**
     * トランザクション条件 : 既に他のトランザクションが存在.
     */
    public static final int TRANSACTION_BY_ALREADY_EXISTENCE = -1 ;
    
    /**
     * トランザクション開始.
     * <BR><BR>
     * トランザクションを開始します.
     * <BR>
     * @return int トランザクションIDが返されます.
     *             [0]が返された場合、このキューはオートコミットです.<BR>
     *             [-1]が返された場合、既に他条件でトランザクションが開始されています.
     */
    public int transaction() ;
    
    /**
     * コミット処理.
     * <BR><BR>
     * コミット処理を行います.
     * <BR>
     * @param id トランザクションIDを設定します.
     */
    public void commit( int id ) ;
    
    /**
     * ロールバック処理.
     * <BR><BR>
     * ロールバック処理を行います.
     * <BR>
     * @param id トランザクションIDを設定します.
     */
    public void rollback( int id ) ;
    
    /**
     * メッセージ情報を設定.
     * <BR><BR>
     * 対象のメッセージ情報を設定します.
     * <BR>
     * @param id トランザクションIDを設定します.<BR>
     *           トランザクション開始前の場合[-1]を設定します.<BR>
     *           トランザクション無効で処理する場合は[Integer.MAX_VALUE]を設定します.
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @param processId 対象のプロセスIDを設定します.<BR>
     *            [null]や[""]を設定した場合、プロセスID条件を無視します.
     * @param message 対象のメッセージ情報を設定します.
     * @param priority 対象の優先順位を設定します.<BR>
     *                 設定できる最小値は[0]です.<BR>
     *                 設定できる最大値は[9500]です.
     * @param expire 対象のExpire値を設定します.<BR>
     *               [0]以下を設定した場合、未設定となります.
     * @return int トランザクションIDが返されます.<BR>
     *             [-1]が返された場合、トランザクションが存在します.
     * @exception InputException 入力例外.
     * @exception MessageOverException メッセージ上限越え例外.
     * @exception QueueFullException キュー満杯例外.
     */
    public int put( int id,String key,String processId,Object message,int priority,long expire )
        throws InputException,MessageOverException,QueueFullException ;
    
    /**
     * メッセージ情報取得後、正常に情報を取得できた場合の処理.
     * <BR><BR>
     * メッセージ情報取得後、正常に情報を取得できた場合の処理します.
     * <BR>
     * @param id トランザクションIDを設定します.<BR>
     *           トランザクション無効で処理する場合は[0]か[Integer.MAX_VALUE]を設定します.
     * @param value 対象の１つのキューデータ情報を設定します.
     */
    public void successGet( int id,QueueValue value ) ;
    
    /**
     * メッセージ情報を取得.
     * <BR><BR>
     * 対象のメッセージ情報を取得します.<BR>
     * また取得できた場合、取得された情報は削除されます.
     * <BR>
     * @param id トランザクションIDを設定します.<BR>
     *           トランザクション開始前の場合[-1]を設定します.<BR>
     *           トランザクション無効で処理する場合は[Integer.MAX_VALUE]を設定します.
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]や[0以下]を設定した場合、情報が
     *                受信されるまで、待ちます.
     * @return int トランザクションIDが返されます.
     *             [-1]が返された場合、情報は存在しません.
     */
    public int get( int id,QueueValue out,String key,int timeout ) ;
    
    /**
     * 対象Bean情報を設定.
     * <BR><BR>
     * 対象Bean情報を設定します.
     * <BR>
     * @param bean 設定対象のBeanを設定します.
     */
    public void setBean( QueueBean bean ) ;
    
    /**
     * 現在の格納電文長を取得.
     * <BR><BR>
     * 現在の格納電文長を取得します.
     * <BR>
     * @return int 格納されている電文長が返されます.
     */
    public int size() ;
    
    /**
     * キューKeyオブジェクトを取得.
     * <BR><BR>
     * キューKeyオブジェクトを取得します.
     * <BR>
     * @return BaseQueueKey キューKeyオブジェクトが返されます.
     */
    public BaseQueueKey getKey() ;
    
    /**
     * キュー定義情報を取得.
     * <BR><BR>
     * キュー定義情報を取得します.
     * <BR>
     * @return AdminQueueBean キュー定義情報が返されます.
     */
    public QueueBean getQueueBean() ;
    
    /**
     * キューステータス情報を取得.
     * <BR><BR>
     * 対象のキューステータス情報を取得します.
     * <BR>
     * @return QueueState 対象のキューステータス情報が返されます.
     */
    public QueueState getState() ;
    
    /**
     * 現在のトランザクションIDを取得.
     * <BR><BR>
     * 現在のトランザクションIDを取得します.
     * <BR>
     * @return int 現在のトランザクションIDが返されます.
     */
    public int getTransactionId() ;
    
    /**
     * このキューがオートコミットであるかチェック.
     * <BR><BR>
     * このキューがオートコミットであるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オートコミットです.<BR>
     *                 [false]が返された場合、オートコミットではありません.
     */
    public boolean isAuthCommit() ;
    
    /**
     * このキューが有効であるかチェック.
     * <BR><BR>
     * このキュー情報が有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、このキューは有効です.
     *                 [false]が返された場合、このキューは無効です.
     */
    public boolean isQueue() ;
}

