/*
 * @(#)QueueState.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue ;

import org.maachang.commons.resource.ResourceType;

/**
 * キューステータス情報.
 * <BR><BR>
 * キューステータス情報を表すインターフェイスです.
 *  
 * @version 2005/12/29
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public interface QueueState
{
    
    /**
     * キューリソースタイプを取得.
     * <BR><BR>
     * 対象のキューリソースタイプを取得します.
     * <BR>
     * @return ResourceType 対象のリソースタイプが返されます.
     */
    public ResourceType getResourceType() ;
    
    /**
     * キュータイプを取得.
     * <BR><BR>
     * 対象のキュータイプを取得します.
     * <BR>
     * @return int キュータイプが返されます.
     */
    public int getQueueType() ;
    
    /**
     * キュー名を取得.
     * <BR><BR>
     * 対象のキュー名を取得します.
     * <BR>
     * @return String 対象のキュー名が返されます.
     */
    public String getName() ;
    
    /**
     * 管理元キューマネージャ名を取得.
     * <BR><BR>
     * 管理元のキューマネージャ名を取得します.
     * <BR>
     * @return String 管理元のキューマネージャ名が返されます.
     */
    public String getParentManagerName() ;
    
    /**
     * 現在キューに格納されているメッセージ数を取得.
     * <BR><BR>
     * 現在キューに格納されているメッセージ数を取得します.
     * <BR>
     * @return int 現在キューに格納されているメッセージ数が返されます.
     */
    public int size() ;
    
    /**
     * キュー情報管理サイズを取得.
     * <BR><BR>
     * 対象のキュー情報管理サイズを取得します.
     * <BR>
     * @return int 対象のキュー情報管理サイズが返されます.
     */
    public int getMaxQueue() ;
    
    /**
     * 警告キューサイズを取得.
     * <BR><BR>
     * 警告キューサイズを取得します.
     * <BR>
     * @returun int 警告キューサイズが返されます.
     */
    public int getWarningQueue() ;
    
    /**
     * データ最大長を取得.
     * <BR><BR>
     * キューに格納するデータの最大長を取得します.
     * <BR>
     * @return int データ最大長が返されます.<BR>
     *             [-1]が返された場合、制限はありません.
     */
    public int getMaxDataLength() ;
    
    /**
     * 警告数域値を取得.
     * <BR><BR>
     * キュー格納数に対する警告数域を取得します.
     * <BR>
     * @return double 警告数域が返されます.
     */
    public double getWarningPersend() ;
    
    /**
     * 圧縮条件取得.
     * <BR><BR>
     * メッセージ情報を圧縮(GZIP)する条件を取得します.
     * <BR>
     * @return boolean 圧縮条件が返されます.<BR>
     *                 [true]が返された場合、メッセージ情報は圧縮されます.<BR>
     *                 [false]が返された場合、メッセージ情報は圧縮されません.
     */
    public boolean isGzipFlag() ;
    
    /**
     * キャッシュ名を取得.
     * <BR><BR>
     * キャッシュ名を取得します.
     * <BR>
     * @return String 対象のキャッシュ名が返されます.
     */
    public String getCacheName() ;
    
    /**
     * キューステータスを前回ステータスに戻す.
     * <BR><BR>
     * キューステータスを前回ステータスに戻します.
     */
    public void stateToOldState() ;
    
    /**
     * キューステータスを設定.
     * <BR><BR>
     * 対象のキューステータスを設定します.
     * <BR>
     * @param state 対象のキューステータスを設定します.
     */
    public void setState( int state ) ;
    
    /**
     * キューステータスを取得.
     * <BR><BR>
     * 対象のキューステータスを取得します.
     * <BR>
     * @return int 対象のキューステータス情報が返されます.<BR>
     *             [STATE_SUCCESS]が返された場合、「正常」です.<BR>
     *             [STATE_DOWN]が返された場合、「停止」です.<BR>
     *             [STATE_STARTUP]が返された場合、「起動中」です.<BR>
     *             [STATE_SHUTDOWN]が返された場合、「停止」です.<BR>
     *             [STATE_ERROR]が返された場合、「異常」です.<BR>
     *             [STATE_WARNING]が返された場合、「キュー格納数指定量による警告」です.<BR>
     *             [STATE_FULL]が返された場合、「キュー満杯」です.
     */
    public int getState() ;
    
    /**
     * オートコミット情報を取得.
     * <BR><BR>
     * このオブジェクトがオートコミットであるか取得します.
     * <BR>
     * @return boolean オートコミットであるか取得します.<BR>
     *                 [true]が返された場合、オートコミットです.<BR>
     *                 [false]が返された場合、オートコミットではありません.
     */
    public boolean isAutoCommit() ;
    
}

