/*
 * @(#)QueueValue.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue ;

import org.maachang.commons.exception.InputException;
import org.maachang.queue.main.manager.BaseQueueManager;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.queue.base.BaseQueue;
import org.maachang.queue.main.queue.base.BaseQueueBean;
import org.maachang.queue.main.queue.base.BaseQueueFactory;

/**
 * 受信キュー.
 * <BR><BR>
 * 1つの受信キューを表すオブジェクトです.
 *  
 * @version 2006/09/04
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class ReceiveMqImple extends AbstractReceiveMq implements ReceiveMq {
    
    /**
     * コンストラクタ.
     */
    private ReceiveMqImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * QueueBeanオブジェクトを設定して新しい受信MQを生成します.
     * <BR>
     * @param bean 対象のQueueBeanを設定します.
     * @exception InputException 入力例外.
     */
    public ReceiveMqImple( QueueBean bean ) throws InputException {
        
        BaseQueue bq = null ;
        BaseQueueManager bman = null ;
        String queueName = null ;
        String queueManager = null ;
        
        if(
            bean == null ||
            bean.getType() != QueueDef.TYPE_RECEIVE ||
            ( queueName = bean.getQueueName() ) == null ||
            queueName.length() <= 0 ||
            ( queueManager = bean.getQueueManagerName() ) == null ||
            queueManager.length() < 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            
            // 対象QueueBeanからキューマネージャを取得.
            bman = ( BaseQueueManager )QueueManagerFactory.get( queueManager ) ;
            if( bman == null || bman.isQueueManager() == false ) {
                // キューマネージャが存在しない場合はエラー.
                throw new InputException(
                    "対象キューマネージャ[" + queueManager +
                    "]は存在しないか不正です" ) ;
            }
            
            // 対象QueueBeanからキューを取得.
            bq = ( BaseQueue )BaseQueueFactory.get(
                QueueDef.TYPE_RECEIVE,queueManager,queueName ) ;
            if( bq == null || bq.isQueue() == false ) {
                // 新しくMQを生成.
                bq = super.createMq( ( BaseQueueBean )bean ) ;
            }
            // 存在する場合は、既に存在するQueue情報で取得.
            else {
                bean = bq.getQueueBean() ;
                if( bean.getOption() != null ) {
                    ( ( BaseQueueBean )bean ).setOption( null ) ;
                    bq.setOption( null ) ;
                }
            }
            
            // オプションが存在する場合は、削除.
            if( bean.getOption() != null ) {
                ( ( BaseQueueBean )bean ).setOption( null ) ;
            }
            
            // メンバー変数セット.
            this.queueManager = bman ;
            this.queue = bq ;
            
        } catch( InputException in ) {
            this.destroy() ;
            throw in ;
        } catch( Exception e ) {
            this.destroy() ;
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定して受信MQを取得します.
     * <BR>
     * @param manager 対象のキューマネージャオブジェクトを設定します.
     * @param name 対象のキュー名を設定します.
     * @exception InputException 入力例外.
     * @exception NotMqException 非存在MQ例外.
     */
    public ReceiveMqImple( QueueManager manager,String name )
        throws InputException,NotMqException {
        
        BaseQueue bq = null ;
        BaseQueueManager bman = null ;
        String queueManager = null ;
        
        if(
            manager == null || manager.isQueueManager() == false ||
            name == null || name.length() <= 0 ) {
            
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            
            // キューマネージャ情報を取得.
            bman = ( BaseQueueManager )manager ;
            queueManager = bman.getName() ;
            
            // 対象条件のキューを取得.
            bq = ( BaseQueue )BaseQueueFactory.get(
                QueueDef.TYPE_RECEIVE,queueManager,name ) ;
            
            // キューが存在しない場合.
            if( bq == null || bq.isQueue() == false ) {
                throw new NotMqException(
                    "対象の(" + manager.getName() +
                    ")マネージャに対して指定MQ(" + name +
                    ")は存在しません" ) ;
            }
            
            // メンバー変数セット.
            this.queueManager = bman ;
            this.queue = bq ;
            
            
        } catch( NotMqException nq ) {
            this.destroy() ;
            throw nq ;
        } catch( Exception e ) {
            this.destroy() ;
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            super.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
}

