/*
 * @(#)QueueValue.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue ;

import java.io.Serializable;
import java.util.Arrays;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.array.ObjectArray;
import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.main.channel.Channel;
import org.maachang.queue.main.channel.ChannelFactory;
import org.maachang.queue.main.manager.BaseQueueManager;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.queue.base.BaseQueue;
import org.maachang.queue.main.queue.base.BaseQueueBean;
import org.maachang.queue.main.queue.base.BaseQueueFactory;
import org.maachang.queue.main.queue.base.QueueEx;

/**
 * 送信キュー.
 * <BR><BR>
 * 1つの送信キューを表すオブジェクトです.
 *  
 * @version 2006/09/04
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class SendMqImple extends AbstractSendMq implements SendMq {
    
    /**
     * コンストラクタ.
     */
    private SendMqImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * QueueBeanオブジェクトを設定して新しい送信MQを生成します.
     * <BR>
     * @param bean 対象のQueueBeanを設定します.
     * @exception InputException 入力例外.
     */
    public SendMqImple( QueueBean bean ) throws InputException {
        
        BaseQueue bq = null ;
        BaseQueueManager bman = null ;
        String queueName = null ;
        String queueManager = null ;
        
        if(
            bean == null ||
            bean.getType() != QueueDef.TYPE_SEND ||
            ( queueName = bean.getQueueName() ) == null ||
            queueName.length() <= 0 ||
            ( queueManager = bean.getQueueManagerName() ) == null ||
            queueManager.length() < 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        // チャネルが存在しない場合はエラー.
        //if( ChannelFactory.size() <= 0 ) {
        //    throw new InputException( "利用可能なチャネルが存在しません" ) ;
        //}
        
        // オプション未定義.
        if(
            bean.getOption() == null ||
            ( bean.getOption() instanceof SendMqOption ) == false ) {
            // 空のオプションを定義.
            ( ( BaseQueueBean )bean ).setOption( new SendMqOption() ) ;
        }
        
        try {
            
            // 対象QueueBeanからキューマネージャを取得.
            bman = ( BaseQueueManager )QueueManagerFactory.get( queueManager ) ;
            if( bman == null || bman.isQueueManager() == false ) {
                // キューマネージャが存在しない場合はエラー.
                throw new InputException(
                    "対象キューマネージャ[" + queueManager +
                    "]は存在しないか不正です" ) ;
            }
            
            // 対象QueueBeanからキューを取得.
            bq = ( BaseQueue )BaseQueueFactory.get(
                QueueDef.TYPE_SEND,queueManager,queueName ) ;
            if( bq == null || bq.isQueue() == false ) {
                
                // 新しくMQを生成.
                bq = super.createMq( ( BaseQueueBean )bean ) ;
                
            }
            // 存在する場合は、既に存在するQueue情報で取得.
            else {
                
                // 現在存在するQueueのBeanを採用.
                bean = bq.getQueueBean() ;
                
                // オプション未定義.
                if(
                    bean.getOption() == null ||
                    ( bean.getOption() instanceof SendMqOption ) == false ) {
                    // 空のオプションを定義.
                    Serializable opt = new SendMqOption() ;
                    bq.setOption( opt ) ;
                    ( ( BaseQueueBean )bean ).setOption( opt ) ;
                }
                
            }
            
            // メンバー変数セット.
            this.queueManager = bman ;
            this.queue = bq ;
            
        } catch( InputException in ) {
            this.destroy() ;
            throw in ;
        } catch( Exception e ) {
            this.destroy() ;
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定して送信MQを取得します.
     * <BR>
     * @param manager 対象のキューマネージャオブジェクトを設定します.
     * @param name 対象のキュー名を設定します.
     * @exception InputException 入力例外.
     * @exception NotMqException 非存在MQ例外.
     */
    public SendMqImple( QueueManager manager,String name )
        throws InputException,NotMqException {
        
        BaseQueue bq = null ;
        BaseQueueManager bman = null ;
        QueueBean bn = null ;
        String queueManager = null ;
        
        if(
            manager == null || manager.isQueueManager() == false ||
            name == null || name.length() <= 0 ) {
            
            throw new InputException( "引数は不正です" ) ;
        }
        
        // チャネルが存在しない場合はエラー.
        //if( ChannelFactory.size() <= 0 ) {
        //    throw new InputException( "利用可能なチャネルが存在しません" ) ;
        //}
        
        try {
            
            // キューマネージャ情報を取得.
            bman = ( BaseQueueManager )manager ;
            queueManager = bman.getName() ;
            
            // 対象条件のキューを取得.
            bq = ( BaseQueue )BaseQueueFactory.get(
                QueueDef.TYPE_SEND,queueManager,name ) ;
            
            // キューが存在しない場合.
            if( bq == null || bq.isQueue() == false ) {
                throw new NotMqException(
                    "対象の(" + manager.getName() +
                    ")マネージャに対して指定MQ(" + name +
                    ")は存在しません" ) ;
            }
            // キューが存在する場合.
            else {
                
                // キューからQueueBeanを取得.
                bn = bq.getQueueBean() ;
                
                // オプションセット.
                ( ( BaseQueueBean )bn ).setOption( new SendMqOption() ) ;
                
                // オプション未定義.
                if(
                    bn.getOption() == null ||
                    ( bn.getOption() instanceof SendMqOption ) == false ) {
                    // 空のオプションを定義.
                    Serializable opt = new SendMqOption() ;
                    bq.setOption( opt ) ;
                    bq.setOption( opt ) ;
                }
                
            }
            
            // メンバー変数セット.
            this.queueManager = bman ;
            this.queue = bq ;
            
        } catch( NotMqException nq ) {
            this.destroy() ;
            throw nq ;
        } catch( Exception e ) {
            this.destroy() ;
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 送信チャネルを追加.
     * <BR><BR>
     * この送信キューに対して送信チャネルを追加します.
     * <BR>
     * @param name 追加対象のチャネル名を設定します.
     * @exception InputException 入力例外.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized void addChannel( String name )
        throws InputException,MaachangQException {
        
        SendMqOption opt = null ;
        
        if( name == null || name.length() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            synchronized( this.queue.getSynchronized().get() ) {
                if(
                    this.queueManager != null &&
                    this.queueManager.isQueueManager() == true &&
                    this.queue != null &&
                    this.queue.isQueue() == true
                 ) {
                    
                    if( ( opt = ( SendMqOption )this.queue.getOption() ) == null ) {
                        throw new InputException( "オプション情報が不正です" ) ;
                    }
                    
                    if( ChannelFactory.get( MaachangQAccessDefine.TYPE_SEND,name ) == null ) {
                        throw new MaachangQException(
                            "送信チャネル名(" + name + ")は存在しません",
                            MaachangQErrorCode.ERROR_NOT_CHANNEL ) ;
                    }
                    
                    opt.add( name ) ;
                }
            }
        } catch( MaachangQException fa ) {
            throw fa ;
        }
        
    }
    
    /**
     * 送信チャネルを削除.
     * <BR><BR>
     * この送信キューに設定されている送信チャネルを削除します.
     * <BR>
     * @param name 削除対象のチャネル名を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void removeChannel( String name )
        throws InputException {
        
        SendMqOption opt = null ;
        
        if( name == null || name.length() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            synchronized( this.queue.getSynchronized().get() ) {
                if(
                    this.queueManager != null &&
                    this.queueManager.isQueueManager() == true &&
                    this.queue != null &&
                    this.queue.isQueue() == true
                 ) {
                    if( ( opt = ( SendMqOption )this.queue.getOption() ) == null ) {
                        throw new InputException( "オプション情報が不正です" ) ;
                    }
                    
                    opt.remove( name ) ;
                }
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * チャネル名を取得.
     * <BR><BR>
     * この送信キューに設定されている送信チャネル名を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return String 対象のチャネル名が返されます.
     */
    public synchronized String getChannelName( int no ) {
        
        SendMqOption opt = null ;
        
        try {
            synchronized( this.queue.getSynchronized().get() ) {
                if( ( opt = this.getOption() ) == null ) {
                    return null ;
                }
                
                return opt.get( no ) ;
            }
        } catch( Exception e ) {
        }
        
        return null ;
    }
    
    /**
     * チャネルIDを取得.
     * <BR><BR>
     * この送信キューに設定されている送信チャネルIDを取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return int 対象のチャネルIDが返されます.
     */
    public synchronized int getChannelID( int no ) {
        
        SendMqOption opt = null ;
        
        try {
            synchronized( this.queue.getSynchronized().get() ) {
                if( ( opt = this.getOption() ) == null ) {
                    return -1 ;
                }
                
                return opt.getID( no ) ;
            }
        } catch( Exception e ) {
        }
        
        return -1 ;
    }
    
    /**
     * チャネルオブジェクトを取得.
     * <BR><BR>
     * この送信キューに設定されている送信チャネルを取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return Channel 対象のチャネルオブジェクトが返されます.
     */
    public synchronized Channel getChannel( int no ) {
        
        SendMqOption opt = null ;
        
        try {
            synchronized( this.queue.getSynchronized().get() ) {
                if( ( opt = this.getOption() ) == null ) {
                    return null ;
                }
                
                return opt.getChannel( no ) ;
            }
        } catch( Exception e ) {
        }
        
        return null ;
    }
    
    /**
     * チャネル名一覧を取得.
     * <BR><BR>
     * この送信キューに設定されている送信チャネル名一覧が返されます.
     * <BR>
     * @return String[] 送信チャネル名一覧が返されます.
     */
    public synchronized String[] getChannelNames() {
        
        SendMqOption opt = null ;
        
        try {
            synchronized( this.queue.getSynchronized().get() ) {
                if( ( opt = this.getOption() ) == null ) {
                    return null ;
                }
                
                return opt.getNames() ;
            }
        } catch( Exception e ) {
        }
        
        return null ;
    }
    
    /**
     * チャネルID一覧を取得.
     * <BR><BR>
     * この送信キューに設定されている送信チャネルID一覧が返されます.
     * <BR>
     * @return int[] 送信チャネルID一覧が返されます.
     */
    public synchronized int[] getChannelDs() {
        
        SendMqOption opt = null ;
        
        try {
            synchronized( this.queue.getSynchronized().get() ) {
                if( ( opt = this.getOption() ) == null ) {
                    return null ;
                }
                
                return opt.getIds() ;
            }
        } catch( Exception e ) {
        }
        
        return null ;
    }
    
    /**
     * 設定チャネル数を取得.
     * <BR><BR>
     * 設定されているチャネル数を取得します.
     * <BR>
     * @return int 設定されているチャネル数が返されます.
     */
    public synchronized int getChannelSize() {
        
        SendMqOption opt = null ;
        
        try {
            synchronized( this.queue.getSynchronized().get() ) {
                if( ( opt = this.getOption() ) == null ) {
                    return -1 ;
                }
                
                return opt.size() ;
            }
        } catch( Exception e ) {
        }
        
        return -1 ;
    }
    
    /**
     * 送信キューからチャネルID一覧を取得.
     * <BR><BR>
     * 送信キューからチャネルID一覧を取得します.
     * <BR>
     * @param queue 対象のキューを設定します.
     * @return int[] チャネルID一覧が返されます.
     */
    public static final int[] getChannelID( Queue queue ) {
        
        int i ;
        int len ;
        int cnt ;
        
        ObjectArray ary = null ;
        Channel channel = null ;
        int[] ret = null ;
        
        try {
            synchronized( ( ( BaseQueue)queue ).getSynchronized().get() ) {
                if(
                    queue == null || queue.isQueue() == false ||
                    queue.getState().getQueueType() != QueueDef.TYPE_SEND
                ) {
                    return null ;
                }
                
                if(
                    ( ary = ( ObjectArray )( ( QueueEx )queue ).getOption() ) != null &&
                    ( len = ary.size() ) > 0
                ) {
                    
                    ret = new int[ len ] ;
                    Arrays.fill( ret,-1 ) ;
                    
                    for( i = 0,cnt = 0 ; i < len ; i ++ ) {
                        
                        channel = ChannelFactory.get(
                            Channel.TYPE_SEND,( String )ary.get( i ) ) ;
                        
                        if( channel != null ) {
                            
                            ret[ cnt ] = channel.getId() ;
                            cnt ++ ;
                            
                        }
                        
                    }
                    
                    if( cnt <= 0 ) {
                        ret = null ;
                    }
                    
                }
                
                return ret ;
            }
        } catch( Exception e ) {
        }
        
        return null ;
    }
    
    /**
     * 送信キューオプションを取得.
     */
    private SendMqOption getOption() {
        
        Serializable opt = null ;
        if( this.queue == null || this.queue.isQueue() == false ) {
            return null ;
        }
        opt = this.queue.getOption() ;
        if( opt == null || ( opt instanceof SendMqOption ) == false ) {
            opt = new SendMqOption() ;
            this.queue.setOption( opt ) ;
        }
        return ( SendMqOption )opt ;
    }
    
}

