/*
 * @(#)BaseQueueImple.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue.base ;

import java.io.Serializable;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.SequenceID;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.access.status.QueueStatus;
import org.maachang.queue.main.queue.MessageOverException;
import org.maachang.queue.main.queue.QueueBean;
import org.maachang.queue.main.queue.QueueDef;
import org.maachang.queue.main.queue.QueueFullException;
import org.maachang.queue.main.queue.QueueMessage;
import org.maachang.queue.main.queue.QueueState;
import org.maachang.queue.main.queue.QueueUtil;
import org.maachang.queue.main.queue.QueueValue;
import org.maachang.queue.main.queue.base.core.QArrayChild;
import org.maachang.queue.main.queue.base.core.QSendSeq;
import org.maachang.queue.main.queue.base.core.QTable;

/**
 * １つの基本キューを表すオブジェクト.
 * <BR><BR>
 * 1つの基本キュー情報を表すオブジェクトです.
 *  
 * @version 2005/12/13
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class BaseQueueImple implements BaseQueue {
    
    /**
     * 基本最大プライオリティ.
     */
    private static final int MAX_BASE_PRIORITY = QueueDef.MAX_BASE_PRIORITY ;
    
    /**
     * キュー管理オブジェクト.
     */
    private BaseQueueStateImple qstate = null ;
    
    /**
     * キューオプション.
     */
    private Serializable option = null ;
    
    /**
     * トランザクションID生成オブジェクト.
     */
    private SequenceID tranID = null ;
    
    /**
     * 現在のトランザクションID.
     */
    private int nowTranID = -1 ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    private BaseQueueImple()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを生成します.
     * <BR>
     * @param state 対象のステータスオブジェクトを設定します.
     * @param option キューオプションを設定します.
     * @exception InputException 入力例外.
     */
    public BaseQueueImple( QueueState state,Serializable option )
        throws InputException
    {
        if( state == null || ( state instanceof BaseQueueStateImple ) == false ){
            throw new InputException( "指定ステータスは不正なオブジェクトです" ) ;
        }
        
        this.qstate = ( BaseQueueStateImple )state ;
        this.option = option ;
        this.tranID = new SequenceID( 1,Integer.MAX_VALUE-1 ) ;
        this.sync.create() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * 対象のオブジェクトを破棄します.
     */
    public void destroy()
    {
        try {
            synchronized( sync.get() ) {
                if( qstate != null ) {
                    qstate.clear() ;
                }
                qstate = null ;
                option = null ;
                tranID = null ;
            }
        } catch( Exception e ) {
        }
        
        qstate = null ;
        option = null ;
        tranID = null ;
        nowTranID = -1 ;
        sync.clear() ;
        
    }
    
    /**
     * トランザクション開始.
     * <BR><BR>
     * トランザクションを開始します.
     * <BR>
     * @return int トランザクションIDが返されます.
     *             [0]が返された場合、このキューはオートコミットです.<BR>
     *             [-1]が返された場合、既に他条件でトランザクションが開始されています.
     */
    public int transaction() {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                if( qstate.isAutoCommit() == true ) {
                    ret = 0 ;
                }
                else if( nowTranID == -1 ) {
                    nowTranID = tranID.getID() ;
                    ret = nowTranID ;
                }
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * コミット処理.
     * <BR><BR>
     * コミット処理を行います.
     * <BR>
     * @param id トランザクションIDを設定します.
     */
    public void commit( int id )
    {
        try {
            synchronized( sync.get() ) {
                if( nowTranID != -1 && nowTranID == id ) {
                    qstate.getTable().commit() ;
                    nowTranID = -1 ;
                    
                    // 現在のステータスが、「正常」、「警告」「満杯」の場合.
                    if( qstate.getState() == QueueStatus.STATE_SUCCESS ||
                        qstate.getState() == QueueStatus.STATE_WARNING ||
                        qstate.getState() == QueueStatus.STATE_FULL ) {
                        
                        // データサイズに対してステータスを更新する.
                        this.checkState( false ) ;
                    }
                }
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * ロールバック処理.
     * <BR><BR>
     * ロールバック処理を行います.
     * <BR>
     * @param id トランザクションIDを設定します.
     */
    public void rollback( int id )
    {
        try {
            synchronized( sync.get() ) {
                if( nowTranID != -1 && nowTranID == id ) {
                    qstate.getTable().rollback() ;
                    nowTranID = -1 ;
                    
                    // 現在のステータスが、「正常」、「警告」「満杯」の場合.
                    if( qstate.getState() == QueueStatus.STATE_SUCCESS ||
                        qstate.getState() == QueueStatus.STATE_WARNING ||
                        qstate.getState() == QueueStatus.STATE_FULL ) {
                        
                        // データサイズに対してステータスを更新する.
                        this.checkState( false ) ;
                    }
                }
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * メッセージ情報を設定.
     * <BR><BR>
     * 対象のメッセージ情報を設定します.
     * <BR>
     * @param id トランザクションIDを設定します.<BR>
     *           トランザクション開始前の場合[-1]を設定します.<BR>
     *           トランザクション無効で処理する場合は[0]か[Integer.MAX_VALUE]を設定します.
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @param processId 対象のプロセスIDを設定します.<BR>
     *            [null]や[""]を設定した場合、プロセスID条件を無視します.
     * @param message 対象のメッセージ情報を設定します.
     * @param priority 対象の優先順位を設定します.<BR>
     *                 設定できる最小値は[0]です.<BR>
     *                 設定できる最大値は[9500]です.
     * @param expire 対象のExpire値を設定します.<BR>
     *               [0]以下を設定した場合、未設定となります.
     * @return int トランザクションIDが返されます.<BR>
     *             [-1]が返された場合、トランザクションが存在します.
     * @exception InputException 入力例外.
     * @exception MessageOverException メッセージ上限越え例外.
     * @exception QueueFullException キュー満杯例外.
     */
    public int put( int id,String key,String processId,Object message,int priority,long expire )
        throws InputException,MessageOverException,QueueFullException
    {
        int len ;
        int ret = -1 ;
        
        QueueMessage qmsg = null ;
        
        if( message == null || ( len = QueueUtil.getMessageLength( message ) ) <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( key == null || key.length() <= 0 ) {
            
            key = null ;
            
        }
        
        if( processId == null || processId.length() <= 0 ) {
            
            processId = null ;
            
        }
        
        if( priority >= MAX_BASE_PRIORITY ) {
            priority = MAX_BASE_PRIORITY ;
        }
        
        // 強制コミットで情報を設定する場合.
        if( id == 0 || id == Integer.MAX_VALUE ) {
            ret = this.putByAutoCommit( key,processId,message,priority,expire ) ;
        }
        // トランザクションで処理する場合.
        else {
            
            // 未トランザクションの場合.
            if( id <= -1 ) {
                id = -1 ;
            }
            
            try {
                
                synchronized( sync.get() ) {
                    
                    // オートコミットの場合か、トランザクションIDが一致する場合.
                    if(
                        qstate.isAutoCommit() == true ||
                        ( qstate.isAutoCommit() == false && nowTranID == id )
                    ) {
                        
                        // 追加チェック処理.
                        this.putByState( len ) ;
                        
                        // メッセージを処理する.
                        qmsg = QueueUtil.convertMessageByQueueMessage(
                            qstate.isGzipFlag(),
                            qstate.getResourceType(),
                            message,
                            len
                        ) ;
                        
                        qstate.getTable().add( key,qmsg,processId,priority,expire ) ;
                        
                        // オートコミットでなく、トランザクション開始前の場合.
                        if( qstate.isAutoCommit() == false && nowTranID == -1 ) {
                            // 新しいトランザクションIDを発行.
                            nowTranID = tranID.getID() ;
                        }
                        
                        // 戻り値に現在のトランザクションIDをセット.
                        ret = nowTranID ;
                        
                        // オートコミットの場合は、トランザクションIDを0として返す.
                        if( qstate.isAutoCommit() == true ) {
                            ret = 0 ;
                        }
                        
                        // ステータス更新.
                        this.checkState() ;
                        
                    }
                    // トランザクションIDが一致しない場合.
                    else {
                        ret = -1 ;
                    }
                    
                }
                
            } catch( InputException in ) {
                throw in ;
            } catch( QueueFullException ful ) {
                throw ful ;
            } catch( MessageOverException mo ) {
                throw mo ;
            } catch( Exception e ) {
                ret = -1 ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * メッセージ情報を強制コミットで設定.
     * <BR><BR>
     * 対象のメッセージ情報を強制コミット条件で設定します.
     * <BR>
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @param processId 対象のキー条件を設定します.<BR>
     *                  [null]や[""]を設定した場合、プロセスID条件を無視します.
     * @param message 対象のメッセージ情報を設定します.
     * @param priority 実際に設定する優先順位を設定します.<BR>
     *                 設定できる最小値は[0]です.<BR>
     *                 設定できる最大値は[9500]です.
     * @param expire 対象のExpire値を設定します.<BR>
     *               [0]以下を設定した場合、未設定となります.
     * @return int トランザクションIDが返されます.<BR>
     *             [-1]が返された場合、トランザクションが存在します.
     * @exception InputException 入力例外.
     * @exception MessageOverException メッセージ上限越え例外.
     * @exception QueueFullException キュー満杯例外.
     */
    public int putByAutoCommit( String key,String processId,Object message,int priority,long expire )
        throws InputException,MessageOverException,QueueFullException
    {
        int len ;
        int ret = -1 ;
        
        QArrayChild ch = null ;
        QueueMessage qmsg = null ;
        
        if( message == null || ( len = QueueUtil.getMessageLength( message ) ) <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( key == null || key.length() <= 0 ) {
            
            key = null ;
            
        }
        
        if( processId == null || processId.length() <= 0 ) {
            
            processId = null ;
            
        }
        
        try {
            
            synchronized( sync.get() ) {
                
                // チェック処理.
                this.putByState( len ) ;
                
                // メッセージを処理する.
                qmsg = QueueUtil.convertMessageByQueueMessage(
                    qstate.isGzipFlag(),
                    qstate.getResourceType(),
                    message,
                    len
                ) ;
                
                ch = qstate.getTable().add( key,qmsg,processId,priority,expire ) ;
                
                // 追加データが有効な場合.
                if( ch != null && ch.isUse() == true ) {
                    
                    // コミット処理.
                    qstate.getTable().commit( ch ) ;
                    
                    // ステータス更新.
                    this.checkState() ;
                    
                }
                
            }
            
            ret = 0 ;
            
        } catch( InputException in ) {
            throw in ;
        } catch( QueueFullException ful ) {
            throw ful ;
        } catch( MessageOverException mo ) {
            throw mo ;
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * メッセージ情報取得後、正常に情報を取得できた場合の処理.
     * <BR><BR>
     * メッセージ情報取得後、正常に情報を取得できた場合の処理します.
     * <BR>
     * @param id トランザクションIDを設定します.<BR>
     *           トランザクション無効で処理する場合は[0]か[Integer.MAX_VALUE]を設定します.
     * @param value 対象の１つのキューデータ情報を設定します.
     */
    public void successGet( int id,QueueValue value ) {
        QArrayChild ch = null ;
        if( value == null || ( ch = value.getChild() ) == null ) {
            return ;
        }
        if( id == 0 || id == Integer.MAX_VALUE ) {
            // 取得情報を削除.
            qstate.getTable().remove( ch ) ;
            
            // コミット処理.
            qstate.getTable().commit( ch ) ;
        }
        else {
            // 取得情報を削除.
            qstate.getTable().remove( ch ) ;
        }
        
        // ステータス更新.
        this.checkState() ;
    }
    
    /**
     * メッセージ情報を取得.
     * <BR><BR>
     * 対象のメッセージ情報を取得します.<BR>
     * また取得できた場合、取得された情報は削除されます.
     * <BR>
     * @param id トランザクションIDを設定します.<BR>
     *           トランザクション開始前の場合[-1]を設定します.<BR>
     *           トランザクション無効で処理する場合は[0]か[Integer.MAX_VALUE]を設定します.
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]や[0以下]を設定した場合、情報が
     *                受信されるまで、待ちます.
     * @return int トランザクションIDが返されます.
     *             [-1]が返された場合、情報は存在しません.
     */
    public int get( int id,QueueValue out,String key,int timeout )
    {
        long tm ;
        long telegramID ;
        long exp ;
        long srcExp ;
        int priority ;
        boolean flg ;
        boolean fstTmFlg ;
        int ret = -1 ;
        
        Object o = null ;
        String srcKey = null ;
        QArrayChild ch = null ;
        
        if( out == null ) {
            return -1 ;
        }
        
        if( key == null || key.length() <= 0 ) {
            key = null ;
        }
        
        // 強制コミットで情報を設定する場合.
        if( id == 0 || id == Integer.MAX_VALUE ) {
            ret = this.getByAutoCommit( false,out,key,timeout ) ;
        }
        // トランザクションで処理する場合.
        else {
            
            // 未トランザクションの場合.
            if( id <= -1 ) {
                id = -1 ;
            }
            
            // 初期処理.
            out.clear() ;
            tm = System.currentTimeMillis() + ( long )timeout ;
            
            // トランザクションIDが一致しない場合は処理中断.
            try {
                synchronized( sync.get() ) {
                    
                    if( qstate.isAutoCommit() == false && nowTranID != id ) {
                        return -1 ;
                    }
                    
                }
            } catch( Exception e ) {
                return -1 ;
            }
            
            for( flg = false,fstTmFlg = false ;; ) {
                
                // 処理終了フラグがセットされている場合.
                if( flg == true ) {
                    //ret = -1 ;
                    break ;
                }
                
                try {
                    
                    // タイムアウト値が有効な場合は、タイムアウト待ち.
                    if( timeout > 0 ) {
                        
                        if( tm <= System.currentTimeMillis() ) {
                            
                            // 最初のループだけは無視する.
                            if( fstTmFlg == false ) {
                                fstTmFlg = true ;
                            }
                            else {
                                flg = true ;
                                out.clear() ;
                                continue ;
                            }
                            
                        }
                        
                    }
                    
                    synchronized( sync.get() ) {
                        
                        // トランザクションIDが一致しなくなった場合は、処理待機.
                        if( qstate.isAutoCommit() == false && nowTranID != id ) {
                            UtilCom.cpuCreate() ;
                            continue ;
                        }
                        
                        // 情報を取得.
                        if( key == null ) {
                            ch = qstate.getTable().getQArrayChild( 0 ) ;
                        }
                        else {
                            ch = qstate.getTable().getQArrayChild( key,0 ) ;
                        }
                        
                        // 情報が存在する場合.
                        if( ch != null && ch.isUse() == true ) {
                            
                            // 対象情報を取得.
                            o = ch.getValue() ;
                            if( o != null && ( o instanceof QueueMessage ) == true ) {
                                o = QueueUtil.convertQueueMessageByMessage(
                                    qstate.getResourceType(),( QueueMessage )o ) ;
                            }
                            
                            // 電文IDを取得.
                            telegramID = ch.getID() ;
                            
                            // 対象キー情報を取得.
                            srcKey = ch.getKey() ;
                            srcKey = ( srcKey != null && srcKey.length() > 0 ) ? srcKey : null ;
                            
                            // 元のExpire値を取得.
                            srcExp = ch.getExpire() ;
                            
                            // プライオリティ値を取得.
                            priority = ch.getPriority() ;
                            
                            // Expire情報を取得.
                            exp = qstate.getTable().getExpire( ch ) ;
                            
                            // 2007.01.17-修正.
                            // 取得データの削除は、メソッド[successGet]
                            // で行うように修正.
                            
                            // 取得情報を削除.
                            //qstate.getTable().remove( ch ) ;
                            
                            // 取得されたExpire値が有効で、既に古い情報と認識された場合.
                            if( exp != -1L && exp <= System.currentTimeMillis() ) {
                                
                                // 対象条件は破棄してコミット.
                                qstate.getTable().remove( ch ) ;
                                qstate.getTable().commit( ch ) ;
                                ch = null ;
                                
                                // タイムアウト待ちを１度だけ無視.
                                fstTmFlg = false ;
                                
                                // 情報は取得しない.
                                out.clear() ;
                                continue ;
                                
                            }
                            
                            // この処理でトランザクションが開始された場合.
                            if( qstate.isAutoCommit() == false && nowTranID == -1 ) {
                                
                                // 新しいトランザクションIDを発行.
                                nowTranID = tranID.getID() ;
                                
                            }
                            
                            // 戻り値にトランザクションIDをセット.
                            ret = nowTranID ;
                            
                            // オートコミットの場合は、トランザクションIDを0として返す.
                            if( qstate.isAutoCommit() == true ) {
                                ret = 0 ;
                            }
                            
                            // 情報の戻り値として、内容をセット.
                            out.setId( ret ) ;
                            out.setTelegramID( telegramID ) ;
                            out.setExpire( srcExp ) ;
                            out.setKey( srcKey ) ;
                            out.setPriority( priority ) ;
                            out.setMessage( o ) ;
                            out.setChild( ch ) ;
                            
                            // 処理終了フラグを立てる.
                            flg = true ;
                            
                        }
                        // 情報が存在しない場合.
                        else {
                            // 一定時間待機.
                            UtilCom.cpuCreate() ;
                        }
                        
                        // ステータス更新.
                        this.checkState() ;
                        
                    }
                    
                } catch( Exception e ) {
                    out.clear() ;
                    flg = true ;
                }finally {
                    o = null ;
                    ch = null ;
                }
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * メッセージ情報を強制コミットで取得.
     * <BR><BR>
     * 対象のメッセージ情報を強制コミットで取得します.
     * <BR>
     * @param qmsg キューメッセージで取得するか設定します.<BR>
     *             [true]を設定した場合、キューメッセージで取得します.
     *             [false]を設定した場合、キューメッセージで取得しません.
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]や[0以下]を設定した場合、情報が
     *                受信されるまで、待ちます.
     * @return int トランザクションIDが返されます.
     *             [-1]が返された場合、情報は存在しません.
     */
    public int getByAutoCommit( boolean qmsg,QueueValue out,int timeout )
    {
        return this.getByAutoCommit( false,qmsg,out,null,timeout ) ;
    }
    
    /**
     * メッセージ情報を強制コミットで取得.
     * <BR><BR>
     * 対象のメッセージ情報を強制コミットで取得します.
     * <BR>
     * @param qmsg キューメッセージで取得するか設定します.<BR>
     *             [true]を設定した場合、キューメッセージで取得します.
     *             [false]を設定した場合、キューメッセージで取得しません.
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]や[0以下]を設定した場合、情報が
     *                受信されるまで、待ちます.
     * @return int トランザクションIDが返されます.
     *             [-1]が返された場合、情報は存在しません.
     */
    public int getByAutoCommit( boolean qmsg,QueueValue out,String key,int timeout )
    {
        return this.getByAutoCommit( false,qmsg,out,key,timeout ) ;
    }
    
    /**
     * メッセージ情報を強制コミットで取得.
     * <BR><BR>
     * 対象のメッセージ情報を強制コミットで取得します.
     * <BR>
     * @param commit コミットデータのみを取得する条件を設定します.<BR>
     *               [true]を設定した場合、コミットデータのみを取得します.<BR>
     *               [false]を設定した場合、条件を指定せず取得します.
     * @param qmsg キューメッセージで取得するか設定します.<BR>
     *             [true]を設定した場合、キューメッセージで取得します.
     *             [false]を設定した場合、キューメッセージで取得しません.
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]や[0以下]を設定した場合、情報が
     *                受信されるまで、待ちます.
     * @return int トランザクションIDが返されます.
     *             [-1]が返された場合、情報は存在しません.
     */
    public int getByAutoCommit( boolean commit,boolean qmsg,QueueValue out,int timeout )
    {
        return this.getByAutoCommit( commit,qmsg,out,null,timeout ) ;
    }
    
    /**
     * メッセージ情報を強制コミットで取得.
     * <BR><BR>
     * 対象のメッセージ情報を強制コミットで取得します.
     * <BR>
     * @param commit コミットデータのみを取得する条件を設定します.<BR>
     *               [true]を設定した場合、コミットデータのみを取得します.<BR>
     *               [false]を設定した場合、条件を指定せず取得します.
     * @param qmsg キューメッセージで取得するか設定します.<BR>
     *             [true]を設定した場合、キューメッセージで取得します.
     *             [false]を設定した場合、キューメッセージで取得しません.
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]や[0以下]を設定した場合、情報が
     *                受信されるまで、待ちます.
     * @return int トランザクションIDが返されます.
     *             [-1]が返された場合、情報は存在しません.
     */
    public int getByAutoCommit( boolean commit,boolean qmsg,QueueValue out,String key,int timeout )
    {
        int i ;
        long tm ;
        long telegramID ;
        long exp ;
        long srcExp ;
        int priority ;
        boolean flg ;
        boolean fstTmFlg ;
        int ret = -1 ;
        
        Object o = null ;
        String srcKey = null ;
        String processId = null ;
        QTable table = null ;
        QArrayChild ch = null ;
        
        if( out == null ) {
            return -1 ;
        }
        
        if( key == null || key.length() <= 0 ) {
            key = null ;
        }
        
        try {
            synchronized( sync.get() ) {
                table = qstate.getTable() ;
            }
        } catch( Exception e ) {
            return -1 ;
        }
        
        // 初期処理.
        out.clear() ;
        tm = System.currentTimeMillis() + ( long )timeout ;
        
        for( flg = false,fstTmFlg = false ;; ) {
            
            // 処理終了フラグがセットされている場合.
            if( flg == true ) {
                break ;
            }
            
            try {
                
                // タイムアウト値が有効な場合は、タイムアウト待ち.
                if( timeout > 0 ) {
                    
                    if( tm <= System.currentTimeMillis() ) {
                        
                        // 最初のループだけは無視する.
                        if( fstTmFlg == false ) {
                            fstTmFlg = true ;
                        }
                        else {
                            flg = true ;
                            out.clear() ;
                            continue ;
                        }
                        
                    }
                    
                }
                
                synchronized( sync.get() ) {
                    
                    // コミット電文のみを取得.
                    if( commit == true && qstate.isAutoCommit() == false ) {
                        
                        for( i = 0 ;; i ++ ) {
                            
                            if( key == null ) {
                                ch = table.getQArrayChild( i ) ;
                            }
                            else{
                                ch = table.getQArrayChild( key,i ) ;
                            }
                            
                            if( ch == null || ch.isUse() == false ) {
                                ch = null ;
                                break ;
                            }
                            else if(
                                ch != null && ch.isUse() == true &&
                                ch.getState() == QArrayChild.STATE_TO_COMMIT
                            )
                            {
                                break ;
                            }
                        }
                        
                        // キー条件なしコミットデータが存在しない場合.
                        if( key == null && ch == null ) {
                            
                            // コミットデータ存在参照値をクリア.
                            qstate.getTable().resetUseCommitDataFlag() ;
                            
                        }
                        
                    }
                    // 全ての条件で取得.
                    else {
                        
                        // 情報を取得.
                        if( key == null ) {
                            ch = table.getQArrayChild( 0 ) ;
                        }
                        else {
                            ch = table.getQArrayChild( key,0 ) ;
                        }
                        
                    }
                    
                    // 情報が存在する場合.
                    if( ch != null  && ch.isUse() == true ) {
                        
                        // 対象情報を取得.
                        o = ch.getValue() ;
                        if(
                            qmsg == false &&
                            o != null &&
                            ( o instanceof QueueMessage ) == true ) {
                            
                            o = QueueUtil.convertQueueMessageByMessage(
                                qstate.getResourceType(),( QueueMessage )o ) ;
                            
                        }
                        
                        // 電文IDを取得.
                        telegramID = ch.getID() ;
                        
                        // 対象キー情報を取得.
                        srcKey = ch.getKey() ;
                        srcKey = ( srcKey != null && srcKey.length() > 0 ) ? srcKey : null ;
                        
                        // プロセスIDを取得.
                        processId = ch.getProcessId() ;
                        
                        // 元のExpire値を取得.
                        srcExp = ch.getExpire() ;
                        
                        // プライオリティ値を取得.
                        priority = ch.getPriority() ;
                        
                        // Expire情報を取得.
                        exp = table.getExpire( ch ) ;
                        
                        // 2007.01.17-修正.
                        // 取得データの削除は、メソッド[successGet]
                        // で行うように修正.
                        
                        // 取得情報を削除.
                        //table.remove( ch ) ;
                        
                        // コミット処理.
                        //table.commit( ch ) ;
                        
                        // 取得されたExpire値が有効で、既に古い情報と認識された場合.
                        if( exp != -1L && exp <= System.currentTimeMillis() ) {
                            
                            // 取得情報を削除.
                            table.remove( ch ) ;
                            
                            // コミット処理.
                            table.commit( ch ) ;
                            ch = null ;
                            
                            // タイムアウト待ちを１度だけ無視.
                            fstTmFlg = false ;
                            
                            // 情報は取得しない.
                            out.clear() ;
                            continue ;
                            
                        }
                        
                        // 戻り値に強制オートコミット用トランザクションIDをセット.
                        ret = 0 ;
                        
                        // 情報の戻り値として、内容をセット.
                        out.setId( ret ) ;
                        out.setTelegramID( telegramID ) ;
                        out.setExpire( srcExp ) ;
                        out.setKey( srcKey ) ;
                        out.setProcessId( processId ) ;
                        out.setPriority( priority ) ;
                        out.setMessage( o ) ;
                        out.setChild( ch ) ;
                        
                        // 処理終了フラグを立てる.
                        flg = true ;
                        
                    }
                    // 情報が存在しない場合.
                    else {
                        // 一定時間待機.
                        UtilCom.cpuCreate() ;
                    }
                    
                    // ステータス更新.
                    this.checkState() ;
                }
                
            } catch( Exception e ) {
                out.clear() ;
                flg = true ;
            }finally {
                o = null ;
                table = null ;
                ch = null ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 対象キュー配列要素を取得.
     * <BR><BR>
     * 対象のキュー配列要素を取得します.
     * <BR>
     * @param commit コミットデータのみを取得する条件を設定します.<BR>
     *               [true]を設定した場合、コミットデータのみを取得します.<BR>
     *               [false]を設定した場合、条件を指定せず取得します.
     * @param no 取得対象の情報項番を設定します.
     * @return QArrayChild 対象のキュー配列要素が返されます.<BR>
     *                     [null]が返された場合、情報は存在しません.
     */
    public QArrayChild getQArrayChild( boolean commit,int no ) {
        
        return this.getQArrayChild( commit,null,no ) ;
        
    }
    
    /**
     * 対象キュー配列要素を取得.
     * <BR><BR>
     * 対象のキュー配列要素を取得します.
     * <BR>
     * @param commit コミットデータのみを取得する条件を設定します.<BR>
     *               [true]を設定した場合、コミットデータのみを取得します.<BR>
     *               [false]を設定した場合、条件を指定せず取得します.
     * @param key 取得対象のキー名を設定します.
     * @param no 取得対象の情報項番を設定します.
     * @return QArrayChild 対象のキュー配列要素が返されます.<BR>
     *                     [null]が返された場合、情報は存在しません.
     */
    public QArrayChild getQArrayChild( boolean commit,String key,int no ) {
        
        int i ;
        int cnt = 0 ;
        
        QTable table = null ;
        QArrayChild ch = null ;
        QArrayChild ret = null ;
        
        if( key == null || key.length() <= 0 ) {
            key = null ;
        }
        if( no <= 0 ) {
            no = 0 ;
        }
        
        try {
            
            synchronized( sync.get() ) {
                
                table = qstate.getTable() ;
                
                if( no >= table.size() ) {
                    return null ;
                }
                
                // コミットのみを取得.
                if( commit == true && qstate.isAutoCommit() == false ) {
                    
                    for( i = 0 ;; i ++ ) {
                        
                        if( key == null ) {
                            ch = table.getQArrayChild( i ) ;
                        }
                        else{
                            ch = table.getQArrayChild( key,i ) ;
                        }
                        
                        if( ch == null || ch.isUse() == false ) {
                            ch = null ;
                            break ;
                        }
                        
                        if( ch.getState() == QArrayChild.STATE_TO_COMMIT ) {
                            if( cnt >= no ) {
                                ret = ch ;
                                break ;
                            }
                            cnt ++ ;
                        }
                    }
                    
                    // キー条件なしコミットデータが存在しない場合.
                    if( key == null && ch == null ) {
                        
                        // コミットデータ存在参照値をクリア.
                        qstate.getTable().resetUseCommitDataFlag() ;
                        
                    }
                    
                }
                // 全ての条件で取得.
                else {
                    
                    for( i = 0 ; ; i ++ ) {
                        if( key == null ) {
                            ch = table.getQArrayChild( i ) ;
                        }
                        else {
                            ch = table.getQArrayChild( key,i ) ;
                        }
                        
                        if( ch == null || ch.isUse() == false ) {
                            break ;
                        }
                        else if( cnt >= no ) {
                            ret = ch ;
                            break ;
                        }
                        
                        cnt ++ ;
                        
                    }
                    
                }
                
                // データ取得が可能な場合.
                if( ret != null && ret.getSendId() <= 0L ) {
                    ret.setSendIdToLong( -1L ) ;
                }
                
                // サイズに対してステータスを更新.
                this.checkState() ;
                
            }
            
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象のキュー配列要素を削除.
     * <BR><BR>
     * 対象のキュー配列要素を削除します.
     * <BR>
     * @param ch 削除対象のキュー配列要素を指定します.
     */
    public void removeQArrayChild( QArrayChild ch ) {
        
        if( ch != null && ch.isUse() == true ) {
            
            // 取得情報を削除してコミット.
            qstate.getTable().remove( ch ) ;
            qstate.getTable().commit( ch ) ;
            
            // サイズに対してステータスを更新.
            this.checkState() ;
        }
        
    }
    
    /**
     * 対象Bean情報を設定.
     * <BR><BR>
     * 対象Bean情報を設定します.
     * <BR>
     * @param bean 設定対象のBeanを設定します.
     */
    public void setBean( QueueBean bean ) {
        
        if( bean == null ) {
            return ;
        }
        
        BaseQueueStateImple state = qstate ;
        
        if( state.getQueueType() == QueueDef.TYPE_SEND &&
            bean.getOption() != null ) {
            this.option = bean.getOption() ;
        }
        
        if( bean.getMaxMessageLength() != Integer.MAX_VALUE ) {
            state.setMaxDataLength( bean.getMaxMessageLength() ) ;
        }
        if( bean.getMaxQueue() != Integer.MAX_VALUE ) {
            state.setMaxQueue( bean.getMaxQueue() ) ;
        }
        if( bean.getWarning() != Integer.MAX_VALUE ) {
            state.setWarningPersend( bean.getWarning() ) ;
        }
        
    }
    
    /**
     * 現在の格納電文長を取得.
     * <BR><BR>
     * 現在の格納電文長を取得します.
     * <BR>
     * @return int 格納されている電文長が返されます.
     */
    public int size() {
        
        return this.size( false ) ;
        
    }
    
    /**
     * 現在の格納電文長を取得.
     * <BR><BR>
     * 現在の格納電文長を取得します.
     * <BR>
     * @param commit コミット条件のみを取得するか設定します.
     *               [true]を設定した場合、コミットデータのみを取得します.<BR>
     *               [false]を設定した場合、条件を指定せず取得します.
     * @return int 格納されている電文長が返されます.
     */
    public int size( boolean commit ) {
        
        int i ;
        int ret ;
        
        if( commit == false ) {
            
            try {
                synchronized( sync.get() ) {
                    ret = qstate.getTable().size() ;
                }
            } catch( Exception e ) {
                ret = 0 ;
            }
            
        }
        else {
            
            try {
                
                synchronized( sync.get() ) {
                    
                    QTable table = qstate.getTable() ;
                    QArrayChild ch = null ;
                    
                    for( i = 0,ret = 0 ;; i ++ ) {
                        
                        ch = table.getQArrayChild( i ) ;
                        
                        if( ch == null || ch.isUse() == false ) {
                            break ;
                        }
                        else if( ch.getState() == QArrayChild.STATE_TO_COMMIT ) {
                            ret ++ ;
                        }
                    }
                    
                }
                
            } catch( Exception e ) {
                ret = 0 ;
            }
        }
        
        return ret ;
    }
    
    /**
     * キューKeyオブジェクトを取得.
     * <BR><BR>
     * キューKeyオブジェクトを取得します.
     * <BR>
     * @return BaseQueueKey キューKeyオブジェクトが返されます.
     */
    public BaseQueueKey getKey() {
        
        BaseQueueKey ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = new BaseQueueKey(
                    qstate.getQueueType(),
                    qstate.getParentManagerName(),
                    qstate.getName()
                ) ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * キュー定義情報を取得.
     * <BR><BR>
     * キュー定義情報を取得します.
     * <BR>
     * @return AdminQueueBean キュー定義情報が返されます.
     */
    public QueueBean getQueueBean() {
        
        BaseQueueBean ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = new BaseQueueBean() ;
                ret.setType( qstate.getQueueType() ) ;
                ret.setGzip( qstate.isGzipFlag() ) ;
                ret.setQueueManagerName( qstate.getParentManagerName() ) ;
                ret.setQueueName( qstate.getName() ) ;
                ret.setOption(
                    ( option == null || ( option instanceof Serializable ) == false ) ?
                        null : ( Serializable )option ) ;
                ret.setWarning( qstate.getWarningPersend() ) ;
                ret.setMaxMessageLength( qstate.getMaxDataLength() ) ;
                ret.setMaxQueue( qstate.getMaxQueue() ) ;
                ret.setAutoCommit( qstate.isAutoCommit() ) ;
                ret.setCacheName( qstate.getCacheName() ) ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キューステータス情報を取得.
     * <BR><BR>
     * 対象のキューステータス情報を取得します.
     * <BR>
     * @return QueueState 対象のキューステータス情報が返されます.
     */
    public QueueState getState()
    {
        QueueState ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = qstate ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * オプションを設定.
     * <BR><BR>
     * 対象のオプションを設定します.
     * <BR>
     * @param option 対象のオプションを設定します.
     */
    public void setOption( Serializable option ) {
        
        try {
            synchronized( sync.get() ) {
                this.option = option ;
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * オプションを取得.
     * <BR><BR>
     * 対象のオプションを取得します.
     * <BR>
     * @return Serializable 対象のオプションが返されます.
     */
    public Serializable getOption() {
        
        Serializable ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = option ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在のトランザクションIDを取得.
     * <BR><BR>
     * 現在のトランザクションIDを取得します.
     * <BR>
     * @return int 現在のトランザクションIDが返されます.
     */
    public int getTransactionId() {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                if( qstate.isAutoCommit() == true ) {
                    ret = 0 ;
                }
                ret = nowTranID ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * Expire処理.
     * <BR><BR>
     * 対象のQArrayChildがExpire条件である場合、破棄します.
     * <BR>
     * @param ch 対象のQArrayChildを設定します.
     * @return boolean 処理結果が返されます.
     */
    public boolean executionExpire( QArrayChild ch ) {
        
        try {
            synchronized( sync.get() ) {
                
                long exp = qstate.getTable().getExpire( ch ) ;
                
                // 取得されたExpire値が有効で、既に古い情報と認識された場合.
                if( exp <= System.currentTimeMillis() ) {
                    
                    // 取得情報を削除.
                    qstate.getTable().remove( ch ) ;
                    
                    // コミット処理.
                    qstate.getTable().commit( ch ) ;
                    
                    return true ;
                    
                }
            }
        } catch( Exception e ) {
        }
        
        return false ;
    }
    
    /**
     * 送信ID管理オブジェクトを取得.
     * <BR><BR>
     * 送信ID管理オブジェクトを取得します.
     * <BR>
     * @reurn QSendSeq 送信ID管理オブジェクトが返されます.
     */
    public QSendSeq getSendSequence() {
        
        QSendSeq ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = qstate.getTable().getSendSequence() ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在ステータス条件に設定.
     * <BR><BR>
     * 現在のステータス条件に設定します.<BR>
     * このメソッドは、現在のステータスがキューフルや、
     * キュー警告の場合に、その条件を設定します.
     */
    public void checkStatus() {
        try {
            this.checkState( false ) ;
        } catch( Exception e ) {
        }
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized 同期オブジェクトが返されます.
     */
    public Synchronized getSynchronized() {
        
        return sync ;
        
    }
    
    /**
     * コミットデータ存在参考値を取得.
     * <BR><BR>
     * コミットデータ存在参考値を取得します.<BR>
     * この情報は、以前コミット処理により、メッセージ情報が設定された
     * 場合に、[true]がセットされます.<BR>
     * そして、この情報が[true]を返されたとしても、以前データを取得したことに
     * より、コミットされた情報は0件の場合もあるため、参考値扱いとなります.
     * <BR>
     * @return boolean コミットデータ存在参考値が返されます.<BR>
     *                 [true]が返された場合、コミットされたデータが存在するかも
     *                 知れません.<BR>
     *                 [false]が返された場合はコミットされた情報は存在しません.
     */
    public boolean isCommitReference() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                ret = qstate.getTable().isUseCommitDataFlag() ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * コミットされたメッセージ情報が存在するかチェック.
     * <BR><BR>
     * コミットされたメッセージ情報が存在するかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、情報は存在します.<BR>
     *                 [false]が返された場合、情報は存在しません.
     */
    public boolean isCommitMessage() {
        return this.isCommitMessage( null ) ;
    }
    
    /**
     * コミットされたメッセージ情報が存在するかチェック.
     * <BR><BR>
     * コミットされたメッセージ情報が存在するかチェックします.
     * <BR>
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、情報は存在します.<BR>
     *                 [false]が返された場合、情報は存在しません.
     */
    public boolean isCommitMessage( String key ) {
        
        int i ;
        boolean ret = false ;
        
        QTable table = null ;
        QArrayChild ch = null ;
        
        table = qstate.getTable() ;
        
        if( key == null || key.length() <= 0 ) {
            key = null ;
        }
        
        try {
            synchronized( sync.get() ) {
                
                // オートコミットでない場合.
                if( qstate.isAutoCommit() == false ) {
                    
                    // コミット条件の内容を検索.
                    for( i = 0 ;; i ++ ) {
                        
                        // キー情報なしでコミットデータを検索.
                        if( key == null ) {
                            ch = table.getQArrayChild( i ) ;
                        }
                        // キー情報ありでコミットデータを検索.
                        else{
                            ch = table.getQArrayChild( key,i ) ;
                        }
                        
                        // 情報が存在しない場合.
                        if( ch == null || ch.isUse() == false ) {
                            ch = null ;
                            ret = false ;
                            break ;
                        }
                        // 情報がコミット条件の場合.
                        else if( ch.getState() == QArrayChild.STATE_TO_COMMIT ) {
                            ret = true ;
                            break ;
                        }
                    }
                    
                    // キー条件なしコミットデータが存在しない場合.
                    if( ch == null && ret == false ) {
                        
                        // コミットデータ存在参照値をクリア.
                        qstate.getTable().resetUseCommitDataFlag() ;
                        
                    }
                    
                }
                // オートコミットで情報が存在する場合.
                else if( table.getQArrayChild( key,0 ) != null ) {
                    ret = true ;
                }
                
            }
            
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * このキューがオートコミットであるかチェック.
     * <BR><BR>
     * このキューがオートコミットであるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オートコミットです.<BR>
     *                 [false]が返された場合、オートコミットではありません.
     */
    public boolean isAuthCommit() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                ret = qstate.isAutoCommit() ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * このキューが有効であるかチェック.
     * <BR><BR>
     * このキュー情報が有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、このキューは有効です.
     *                 [false]が返された場合、このキューは無効です.
     */
    public boolean isQueue() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                ret = ( qstate != null ) ? true : false ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象キューに設定可能かチェック.
     * <BR><BR>
     * 対象キューに設定可能かチェックします.
     * <BR>
     * @param len 対象の電文長を設定します.
     * @exception MessageOverException メッセージオーバー例外.
     * @exception QueueFullException キューフル例外.
     */
    public final void isPut( int len )
        throws MessageOverException,QueueFullException {
        
        try {
            synchronized( sync.get() ) {
                this.putByState( len ) ;
            }
        } catch( MessageOverException mo ) {
            throw mo ;
        } catch( QueueFullException qf ) {
            throw qf ;
        }
        
    }
    
    /**
     * putステータス変更.
     */
    private final void putByState( int len )
        throws MessageOverException,QueueFullException {
        
        // 最大データ長を越している場合.
        if( qstate.getMaxDataLength() != -1 && 
            len > qstate.getMaxDataLength() ) {
            
            throw new MessageOverException(
                "格納対象データ長[" + len +
                "]は、データ最大長[" + qstate.getMaxDataLength() +
                "]を越しています" ) ;
            
        }
        // ステータスチェック.
        this.checkState() ;
        
    }
    
    /**
     * 現在キューサイズとステータスの関連をチェック.
     */
    private final void checkState()
        throws QueueFullException {
        this.checkState( true ) ;
    }
    
    /**
     * 現在キューサイズとステータスの関連をチェック.
     */
    private final void checkState( boolean mode )
        throws QueueFullException {
        
        // キューサイズに空きが存在する場合.
        if( qstate.getMaxQueue() > qstate.size() ) {
            
            // キューサイズが、警告領域に達している場合.
            if( qstate.getState() != QueueStatus.STATE_WARNING &&
                qstate.getState() != QueueStatus.STATE_FULL &&
                qstate.getWarningQueue() <= qstate.size() ) {
                
                // ワーニングステータスに設定.
                qstate.setState( QueueStatus.STATE_WARNING ) ;
            }
            // ワーニングステータスから、正常に復帰した場合.
            else if(
                ( qstate.getState() == QueueStatus.STATE_WARNING ||
                    qstate.getState() == QueueStatus.STATE_FULL ) &&
                qstate.getWarningQueue() > qstate.size() ) {
                
                // 正常状態に復帰.
                qstate.setState( QueueStatus.STATE_SUCCESS ) ;
                
            }
        }
        // キューフル条件の場合.
        else if( qstate.getState() != QueueStatus.STATE_FULL &&
            qstate.getMaxQueue() <= qstate.size() ) {
            
            // キューフルを示すステータス設定.
            qstate.setState( QueueStatus.STATE_FULL ) ;
            
        }
        
        // exception を返すモード.
        if( mode == true ) {
            // キューフル条件の場合.
            if( qstate.getState() == QueueStatus.STATE_FULL ) {
                throw new QueueFullException( "キューテーブルはFULL状態です" ) ;
            }
        }
        
    }
}

