/*
 * @(#)BaseQueueStateImple.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue.base ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.thread.Synchronized;
import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.status.QueueStatus;
import org.maachang.queue.config.MqDefine;
import org.maachang.queue.main.cache.MqCacheFactory;
import org.maachang.queue.main.queue.QueueDef;
import org.maachang.queue.main.queue.base.core.QTable;

/**
 * 基本キューステータス情報.
 * <BR><BR>
 * 基本キューステータスを表すオブジェクト.
 *  
 * @version 2005/12/05
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class BaseQueueStateImple implements BaseQueueState {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( BaseQueueStateImple.class ) ;
    
    /**
     * キュー警告パーセント : デフォルト値.
     */
    private static final double DEF_WARNING_PERSENT = QueueDef.DEF_WARNING_PERSENT ;
    
    /**
     * キュー警告パーセント : 最小値.
     */
    private static final double MIN_WARNING_PERSEND = QueueDef.MIN_WARNING_PERSEND ;
    
    /**
     * キュー警告パーセント : 最大値.
     */
    private static final double MAX_WARNING_PERSEND = QueueDef.MAX_WARNING_PERSEND ;
    
    /**
     * 最小キュー数.
     */
    private static final int MIN_SIZE = QueueDef.MIN_QUEUE_SIZE ;
    
    /**
     * 最大キュー数.
     */
    private static final int MAX_SIZE = MqDefine.MAX_RECEIVE_QUEUE_LENGTH ;
    
    
    
    /**
     * キュー管理リソースタイプ.
     */
    private ResourceType resourceType = null ;
    
    /**
     * キュータイプ.
     */
    private int queueType = -1 ;
    
    /**
     * 対象キュー名.
     */
    private String name = null ;
    
    /**
     * 管理元キューマネージャ名.
     */
    private String parentManagerName = null ;
    
    /**
     * キュー格納オブジェクト.
     */
    private QTable table = null ;
    
    /**
     * キュー警告パーセント.
     */
    private double warningPersend = DEF_WARNING_PERSENT ;
    
    /**
     * データバイナリ最大長.
     */
    private int maxDataLength = -1 ;
    
    /**
     * キュー格納最大格納数.
     */
    private int maxQueue = -1 ;
    
    /**
     * キュー警告長.
     */
    private int warningQueue = -1 ;
    
    /**
     * 圧縮フラグ.
     */
    private boolean gzipFlag = false ;
    
    /**
     * キューステータス.
     */
    private int state = -1 ;
    
    /**
     * 前回キューステータス.
     */
    private int oldState = -1 ;
    
    /**
     * キャッシュ名.
     */
    private String cacheName = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public BaseQueueStateImple()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト生成.
     * <BR><BR>
     * 対象オブジェクトを生成します.
     * <BR>
     * @param resourceType 対象のリソースタイプを設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @param parentManagerName 管理元のキューマネージャ名を設定します.
     * @param name 対象のキュー名を設定します.
     * @param warning 対象のキュー警告パーセントを設定します.
     * @param maxDataLength データの最大長を設定します.<BR>
     *                      [0]以下を設定した場合、制限を設けません.
     * @param maxQueue 対象のキュー最大格納数を設定します.
     * @param autoCommit オートコミットを設定します.<BR>
     *                   [true]が返された場合、オートコミットはONになります.<BR>
     *                   [false]が返された場合、オートコミットはOFFになります.
     * @param gzipFlag 圧縮条件[GZIP圧縮]を設定します.<BR>
     *                 [true]を設定した場合、メッセージ情報は圧縮されます.<BR>
     *                 [false]を設定した場合、メッセージ情報は圧縮されません.
     * @param cacheName 対象のキャッシュ名を設定します.
     * @exception InputException 入力例外.
     */
    public void create(
        ResourceType resourceType,int queueType,String parentManagerName,String name,
        double warning,int maxDataLength,int maxQueue,boolean autoCommit,boolean gzipFlag,
        String cacheName )
        throws InputException
    {
        if(
            parentManagerName == null || parentManagerName.length() <= 0 ||
            name == null || name.length() <= 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        maxQueue = ( maxQueue <= MIN_SIZE ) ? MIN_SIZE :
        (
            ( maxQueue >= MAX_SIZE ) ? MAX_SIZE : maxQueue 
        ) ;
        
        maxDataLength = ( maxDataLength <= 0 ) ? -1 : maxDataLength ;
        
        this.clear() ;
        sync.create() ;
        
        try{
            synchronized( sync.get() ){
                
                this.resourceType = resourceType ;
                this.queueType = queueType ;
                this.parentManagerName = parentManagerName ;
                this.name = name ;
                this.maxDataLength = maxDataLength ;
                this.maxQueue = maxQueue ;
                this.table = new QTable() ;
                this.table.create( autoCommit ) ;
                this.gzipFlag = gzipFlag ;
                this.state = QueueStatus.STATE_STARTUP ;
                this.oldState = QueueStatus.STATE_DOWN ;
                
                this.setWarningPersend( warning ) ;
                this.setCacheName( cacheName ) ;
                
            }
        }catch( Exception e ){
            this.clear() ;
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear()
    {
        boolean flg = false ;
        
        try{
            synchronized( sync.get() ){
                
                if( this.table != null ){
                    this.table.clear() ;
                }
                
                this.resourceType = null ;
                this.queueType = -1 ;
                this.name = null ;
                this.parentManagerName = null ;
                this.table = null ;
                this.warningPersend = DEF_WARNING_PERSENT ;
                this.maxDataLength = -1 ;
                this.maxQueue = -1 ;
                this.state = QueueStatus.STATE_DOWN ;
                this.oldState = QueueStatus.STATE_DOWN ;
                this.cacheName = null ;
                
                flg = true ;
                
            }
        }catch( Exception e ){
        }
        
        if( flg == false ){
            
            this.resourceType = null ;
            this.queueType = -1 ;
            this.name = null ;
            this.parentManagerName = null ;
            this.table = null ;
            this.warningPersend = DEF_WARNING_PERSENT ;
            this.maxDataLength = -1 ;
            this.maxQueue = -1 ;
            this.state = QueueStatus.STATE_DOWN ;
            this.oldState = QueueStatus.STATE_DOWN ;
            this.cacheName = null ;
            
        }
        
        sync.clear() ;
        
    }
    
    /**
     * 今回ステータスと前回ステータスを交換.
     * <BR><BR>
     * 今回のステータス情報と前回のステータス情報を交換します.
     */
    public void changeState()
    {
        int tmp ;
        
        try{
            synchronized( sync.get() ){
                tmp = this.oldState ;
                this.oldState = this.state ;
                this.state = tmp ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * キュー管理オブジェクトを取得.
     * <BR><BR>
     * 設定されているキュー格納オブジェクトを取得します.
     * <BR>
     * @return QTable キュー格納オブジェクトが返されます.
     */
    public QTable getTable()
    {
        QTable ret = null ;
        
        try{
            synchronized( sync.get() ){
                ret = this.table ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * キューリソースを設定.
     * <BR><BR>
     * 対象のキューリソースタイプを設定します.
     * <BR>
     * @param resourceType 対象のリソースタイプを設定します.
     */
    public void setResourceType( ResourceType resourceType )
    {
        try{
            synchronized( sync.get() ){
                this.resourceType = resourceType ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * キューリソースタイプを取得.
     * <BR><BR>
     * 対象のキューリソースタイプを取得します.
     * <BR>
     * @return ResourceType 対象のリソースタイプが返されます.
     */
    public ResourceType getResourceType()
    {
        ResourceType ret = null ;
        
        try{
            synchronized( sync.get() ){
                ret = this.resourceType ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * キュータイプを設定.
     * <BR><BR>
     * キュータイプを設定します.
     * <BR>
     * @param queueType 対象のキュータイプを設定します.
     */
    public void setQueueType( int queueType )
    {
        try{
            synchronized( sync.get() ) {
                this.queueType = queueType ;
            }
        }catch( Exception e ) {
        }
    }
    
    /**
     * キュータイプを取得.
     * <BR><BR>
     * 対象のキュータイプを取得します.
     * <BR>
     * @return int キュータイプが返されます.
     */
    public int getQueueType()
    {
        int ret ;
        
        try{
            synchronized( sync.get() ) {
                ret = this.queueType ;
            }
        }catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * キュー名を設定.
     * <BR><BR>
     * 対象のキュー名を設定します.
     * <BR>
     * @param name 対象のキュー名を設定します.
     */
    public void setName( String name )
    {
        if( name == null || name.length() <= 0 ){
            return ;
        }
        
        try{
            synchronized( sync.get() ){
                this.name = name ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * キュー名を取得.
     * <BR><BR>
     * 対象のキュー名を取得します.
     * <BR>
     * @return String 対象のキュー名が返されます.
     */
    public String getName()
    {
        String ret = null ;
        
        try{
            synchronized( sync.get() ){
                ret = this.name ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 管理元キューマネージャ名を取得.
     * <BR><BR>
     * 管理元のキューマネージャ名を取得します.
     * <BR>
     * @return String 管理元のキューマネージャ名が返されます.
     */
    public String getParentManagerName()
    {
        String ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.parentManagerName ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在キューに格納されている情報数を取得.
     * <BR><BR>
     * 現在キューに格納されている情報数を取得します.
     * <BR>
     * @return int 現在キューに格納されている情報数が返されます.
     */
    public int size()
    {
        int ret ;
        
        try{
            synchronized( sync.get() ){
                ret = this.table.size() ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * キュー情報管理サイズを設定.
     * <BR><BR>
     * 対象のキュー情報管理サイズを設定します.
     * <BR>
     * @param maxQueue 対象のキュー情報管理サイズを設定します.
     */
    public void setMaxQueue( int maxQueue )
    {
        maxQueue = ( maxQueue <= MIN_SIZE ) ? MIN_SIZE :
        (
            ( maxQueue >= MAX_SIZE ) ? MAX_SIZE : maxQueue 
        ) ;
        
        try{
            synchronized( sync.get() ){
                this.maxQueue = maxQueue ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * キュー情報管理サイズを取得.
     * <BR><BR>
     * 対象のキュー情報管理サイズを取得します.
     * <BR>
     * @return int 対象のキュー情報管理サイズが返されます.
     */
    public int getMaxQueue()
    {
        int ret ;
        
        try{
            synchronized( sync.get() ){
                ret = this.maxQueue ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 警告キューサイズを取得.
     * <BR><BR>
     * 警告キューサイズを取得します.
     * <BR>
     * @returun int 警告キューサイズが返されます.
     */
    public int getWarningQueue() {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.warningQueue ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * データ最大長を設定.
     * <BR><BR>
     * キューに格納するデータの最大長を設定します.
     * <BR>
     * @param maxDataLength データの最大長を設定します.<BR>
     *                      [0]以下を設定した場合、制限は設けられません.
     */
    public void setMaxDataLength( int maxDataLength )
    {
        maxDataLength = ( maxDataLength <= 0 ) ? -1 : maxDataLength ;
        
        try {
            
            synchronized( sync.get() ) {
                
                this.maxDataLength = maxDataLength ;
                
            }
            
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * データ最大長を取得.
     * <BR><BR>
     * キューに格納するデータの最大長を取得します.
     * <BR>
     * @return int データ最大長が返されます.<BR>
     *             [-1]が返された場合、制限はありません.
     */
    public int getMaxDataLength()
    {
        int ret ;
        
        try {
            
            synchronized( sync.get() ) {
                
                ret = this.maxDataLength ;
                
            }
            
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 警告数域値を設定.
     * <BR><BR>
     * キュー格納数に対する警告数域を設定します.
     * <BR>
     * @param warning 対象のキュー警告パーセントを設定します.
     */
    public void setWarningPersend( double warning )
    {
        warning = ( warning <= MIN_WARNING_PERSEND ) ? MIN_WARNING_PERSEND :
        (
            ( warning >= MAX_WARNING_PERSEND ) ? MAX_WARNING_PERSEND : warning
        ) ;
        
        try{
            synchronized( sync.get() ){
                this.warningPersend = warning ;
                this.warningQueue = ( int )( ( double )maxQueue * warning ) ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 警告数域値を取得.
     * <BR><BR>
     * キュー格納数に対する警告数域を取得します.
     * <BR>
     * @return double 警告数域が返されます.
     */
    public double getWarningPersend()
    {
        double ret ;
        
        try{
            synchronized( sync.get() ){
                ret = this.warningPersend ;
            }
        }catch( Exception e ){
            ret = -1f ;
        }
        
        return ret ;
    }
    
    /**
     * 圧縮条件取得.
     * <BR><BR>
     * メッセージ情報を圧縮(GZIP)する条件を取得します.
     * <BR>
     * @return boolean 圧縮条件が返されます.<BR>
     *                 [true]が返された場合、メッセージ情報は圧縮されます.<BR>
     *                 [false]が返された場合、メッセージ情報は圧縮されません.
     */
    public boolean isGzipFlag() {
        
        boolean ret ;
        
        try {
            synchronized( sync.get() ) {
                ret = gzipFlag ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュ名を設定.
     * <BR><BR>
     * キャッシュ名を設定します.
     * <BR>
     * @param name 対象のキャッシュ名を設定します.
     */
    public void setCacheName( String name ) {
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                if( MqCacheFactory.get( name ) != null ) {
                    cacheName = name ;
                }
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * キャッシュ名を取得.
     * <BR><BR>
     * キャッシュ名を取得します.
     * <BR>
     * @return String 対象のキャッシュ名が返されます.
     */
    public String getCacheName() {
        String ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = cacheName ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * キューステータスを前回ステータスに戻す.
     * <BR><BR>
     * キューステータスを前回ステータスに戻します.
     */
    public void stateToOldState() {
        try{
            synchronized( sync.get() ){
                this.state = this.oldState ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * キューステータスを設定.
     * <BR><BR>
     * 対象のキューステータスを設定します.
     * <BR>
     * @param state 対象のキューステータスを設定します.
     */
    public void setState( int state )
    {
        if(
            state != QueueStatus.STATE_SUCCESS &&
            state != QueueStatus.STATE_DOWN &&
            state != QueueStatus.STATE_STARTUP &&
            state != QueueStatus.STATE_SHUTDOWN &&
            state != QueueStatus.STATE_ERROR &&
            state != QueueStatus.STATE_WARNING &&
            state != QueueStatus.STATE_FULL
        )
        {
            return ;
        }
        
        try{
            synchronized( sync.get() ){
                if( this.state != state ) {
                    LOG.info(
                        new StringBuffer().
                        append( ( ( this.queueType == MaachangQAccessDefine.TYPE_SEND ) ?
                            "送信" : "受信" ) ).
                        append( "キュー(queue:" ).
                        append( name ).
                        append( " manager:" ).
                        append( parentManagerName ).
                        append( ")のステータスは " ).
                        append( QueueStatus.getStateByString( this.state ) ).
                        append( " から " ).
                        append( QueueStatus.getStateByString( state ) ).
                        append( " に変更されました" ).
                        toString() ) ;
                    this.state = state ;
                }
                this.state = state ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * キューステータスを取得.
     * <BR><BR>
     * 対象のキューステータスを取得します.
     * <BR>
     * @return int 対象のキューステータス情報が返されます.<BR>
     * @return int 対象のキューステータス情報が返されます.<BR>
     *             [STATE_SUCCESS]が返された場合、「正常」です.<BR>
     *             [STATE_DOWN]が返された場合、「停止」です.<BR>
     *             [STATE_STARTUP]が返された場合、「起動中」です.<BR>
     *             [STATE_SHUTDOWN]が返された場合、「停止」です.<BR>
     *             [STATE_ERROR]が返された場合、「異常」です.<BR>
     *             [STATE_WARNING]が返された場合、「キュー格納数指定量による警告」です.<BR>
     *             [STATE_FULL]が返された場合、「キュー満杯」です.
     */
    public int getState()
    {
        int ret ;
        
        try{
            synchronized( sync.get() ){
                return this.state ;
            }
        }catch( Exception e ){
            ret = QueueStatus.STATE_DOWN ;
        }
        
        return ret ;
    }
    
    /**
     * 前回キューステータスを設定.
     * <BR><BR>
     * 対象の前回キューステータスを設定します.
     * <BR>
     * @param state 対象のキューステータスを設定します.
     */
    public void setOldState( int state )
    {
        if(
            state != QueueStatus.STATE_SUCCESS &&
            state != QueueStatus.STATE_DOWN &&
            state != QueueStatus.STATE_STARTUP &&
            state != QueueStatus.STATE_SHUTDOWN &&
            state != QueueStatus.STATE_ERROR &&
            state != QueueStatus.STATE_WARNING &&
            state != QueueStatus.STATE_FULL
        )
        {
            return ;
        }
        
        try{
            synchronized( sync.get() ){
                this.oldState = state ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 前回キューステータスを取得.
     * <BR><BR>
     * 対象の前回キューステータスを取得します.
     * <BR>
     * @return int 対象の前回キューステータス情報が返されます.<BR>
     *             [STATE_SUCCESS]が返された場合、「正常」です.<BR>
     *             [STATE_DOWN]が返された場合、「停止」です.<BR>
     *             [STATE_STARTUP]が返された場合、「起動中」です.<BR>
     *             [STATE_SHUTDOWN]が返された場合、「停止」です.<BR>
     *             [STATE_ERROR]が返された場合、「異常」です.<BR>
     *             [STATE_WARNING]が返された場合、「キュー格納数指定量による警告」です.<BR>
     *             [STATE_FULL]が返された場合、「キュー満杯」です.
     */
    public int getOldState()
    {
        int ret ;
        
        try{
            synchronized( sync.get() ){
                ret = this.oldState ;
            }
        }catch( Exception e ){
            ret = QueueStatus.STATE_DOWN ;
        }
        
        return ret ;
    }
    
    /**
     * オートコミット情報を取得.
     * <BR><BR>
     * このオブジェクトがオートコミットであるか取得します.
     * <BR>
     * @return boolean オートコミットであるか取得します.<BR>
     *                 [true]が返された場合、オートコミットです.<BR>
     *                 [false]が返された場合、オートコミットではありません.
     */
    public boolean isAutoCommit()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = this.table.isAutoCommit() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
}

