/*
 * @(#)QueueEx.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue.base ;

import java.io.Serializable;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;
import org.maachang.queue.main.queue.MessageOverException;
import org.maachang.queue.main.queue.QueueFullException;
import org.maachang.queue.main.queue.QueueValue;
import org.maachang.queue.main.queue.base.core.QArrayChild;
import org.maachang.queue.main.queue.base.core.QSendSeq;

/**
 * １つのキューを表す拡張インターフェイス.
 * <BR><BR>
 * 1つのキュー情報を表す拡張インターフェイスです.
 *  
 * @version 2006/09/03
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public interface QueueEx
{
    
    /**
     * メッセージ情報を強制コミットで設定.
     * <BR><BR>
     * 対象のメッセージ情報を強制コミット条件で設定します.
     * <BR>
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @param processId 対象のキー条件を設定します.<BR>
     *                  [null]や[""]を設定した場合、プロセスID条件を無視します.
     * @param message 対象のメッセージ情報を設定します.
     * @param priority 実際に設定する優先順位を設定します.<BR>
     *                 設定できる最小値は[0]です.<BR>
     *                 設定できる最大値は[9500]です.
     * @param expire 対象のExpire値を設定します.<BR>
     *               [0]以下を設定した場合、未設定となります.
     * @return int トランザクションIDが返されます.<BR>
     *             [-1]が返された場合、トランザクションが存在します.
     * @exception InputException 入力例外.
     * @exception MessageOverException メッセージ上限越え例外.
     * @exception QueueFullException キュー満杯例外.
     */
    public int putByAutoCommit( String key,String processId,Object message,int priority,long expire )
        throws InputException,MessageOverException,QueueFullException ;
    
    /**
     * メッセージ情報を強制コミットで取得.
     * <BR><BR>
     * 対象のメッセージ情報を強制コミットで取得します.
     * <BR>
     * @param qmsg キューメッセージで取得するか設定します.<BR>
     *             [true]を設定した場合、キューメッセージで取得します.
     *             [false]を設定した場合、キューメッセージで取得しません.
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]や[0以下]を設定した場合、情報が
     *                受信されるまで、待ちます.
     * @return int トランザクションIDが返されます.
     *             [-1]が返された場合、情報は存在しません.
     */
    public int getByAutoCommit( boolean qmsg,QueueValue out,int timeout ) ;
    
    /**
     * メッセージ情報を強制コミットで取得.
     * <BR><BR>
     * 対象のメッセージ情報を強制コミットで取得します.
     * <BR>
     * @param qmsg キューメッセージで取得するか設定します.<BR>
     *             [true]を設定した場合、キューメッセージで取得します.
     *             [false]を設定した場合、キューメッセージで取得しません.
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]や[0以下]を設定した場合、情報が
     *                受信されるまで、待ちます.
     * @return int トランザクションIDが返されます.
     *             [-1]が返された場合、情報は存在しません.
     */
    public int getByAutoCommit( boolean qmsg,QueueValue out,String key,int timeout ) ;
    
    /**
     * メッセージ情報を強制コミットで取得.
     * <BR><BR>
     * 対象のメッセージ情報を強制コミットで取得します.
     * <BR>
     * @param qmsg キューメッセージで取得するか設定します.<BR>
     *             [true]を設定した場合、キューメッセージで取得します.
     *             [false]を設定した場合、キューメッセージで取得しません.
     * @param commit コミットデータのみを取得する条件を設定します.<BR>
     *               [true]を設定した場合、コミットデータのみを取得します.<BR>
     *               [false]を設定した場合、条件を指定せず取得します.
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]や[0以下]を設定した場合、情報が
     *                受信されるまで、待ちます.
     * @return int トランザクションIDが返されます.
     *             [-1]が返された場合、情報は存在しません.
     */
    public int getByAutoCommit( boolean commit,boolean qmsg,QueueValue out,int timeout ) ;
    
    /**
     * メッセージ情報を強制コミットで取得.
     * <BR><BR>
     * 対象のメッセージ情報を強制コミットで取得します.
     * <BR>
     * @param commit コミットデータのみを取得する条件を設定します.<BR>
     *               [true]を設定した場合、コミットデータのみを取得します.<BR>
     *               [false]を設定した場合、条件を指定せず取得します.
     * @param qmsg キューメッセージで取得するか設定します.<BR>
     *             [true]を設定した場合、キューメッセージで取得します.
     *             [false]を設定した場合、キューメッセージで取得しません.
     * @param out 取得されたメッセージ情報を格納するオブジェクト.
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]や[0以下]を設定した場合、情報が
     *                受信されるまで、待ちます.
     * @return int トランザクションIDが返されます.
     *             [-1]が返された場合、情報は存在しません.
     */
    public int getByAutoCommit( boolean commit,boolean qmsg,QueueValue out,String key,int timeout ) ;
    
    /**
     * 対象キュー配列要素を取得.
     * <BR><BR>
     * 対象のキュー配列要素を取得します.
     * <BR>
     * @param commit コミットデータのみを取得する条件を設定します.<BR>
     *               [true]を設定した場合、コミットデータのみを取得します.<BR>
     *               [false]を設定した場合、条件を指定せず取得します.
     * @param no 取得対象の情報項番を設定します.
     * @return QArrayChild 対象のキュー配列要素が返されます.<BR>
     *                     [null]が返された場合、情報は存在しません.
     */
    public QArrayChild getQArrayChild( boolean commit,int no ) ;
    
    /**
     * 対象キュー配列要素を取得.
     * <BR><BR>
     * 対象のキュー配列要素を取得します.
     * <BR>
     * @param commit コミットデータのみを取得する条件を設定します.<BR>
     *               [true]を設定した場合、コミットデータのみを取得します.<BR>
     *               [false]を設定した場合、条件を指定せず取得します.
     * @param key 取得対象のキー名を設定します.
     * @param no 取得対象の情報項番を設定します.
     * @return QArrayChild 対象のキュー配列要素が返されます.<BR>
     *                     [null]が返された場合、情報は存在しません.
     */
    public QArrayChild getQArrayChild( boolean commit,String key,int no ) ;
    
    /**
     * 対象のキュー配列要素を削除.
     * <BR><BR>
     * 対象のキュー配列要素を削除します.
     * <BR>
     * @param ch 削除対象のキュー配列要素を指定します.
     */
    public void removeQArrayChild( QArrayChild ch ) ;
    
    /**
     * 現在の格納電文長を取得.
     * <BR><BR>
     * 現在の格納電文長を取得します.
     * <BR>
     * @param commit コミット条件のみを取得するか設定します.
     *               [true]を設定した場合、コミットデータのみを取得します.<BR>
     *               [false]を設定した場合、条件を指定せず取得します.
     * @return int 格納されている電文長が返されます.
     */
    public int size( boolean commit ) ;
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized 同期オブジェクトが返されます.
     */
    public Synchronized getSynchronized() ;
    
    /**
     * オプションを設定.
     * <BR><BR>
     * 対象のオプションを設定します.
     * <BR>
     * @param option 対象のオプションを設定します.
     */
    public void setOption( Serializable option ) ;
    
    /**
     * オプションを取得.
     * <BR><BR>
     * 対象のオプションを取得します.
     * <BR>
     * @return Serializable 対象のオプションが返されます.
     */
    public Serializable getOption() ;
    
    /**
     * Expire処理.
     * <BR><BR>
     * 対象のQArrayChildがExpire条件である場合、破棄します.
     * <BR>
     * @param ch 対象のQArrayChildを設定します.
     * @return boolean 処理結果が返されます.
     */
    public boolean executionExpire( QArrayChild ch ) ;
    
    /**
     * 送信ID管理オブジェクトを取得.
     * <BR><BR>
     * 送信ID管理オブジェクトを取得します.
     * <BR>
     * @reurn QSendSeq 送信ID管理オブジェクトが返されます.
     */
    public QSendSeq getSendSequence() ;
    
    /**
     * 現在ステータス条件に設定.
     * <BR><BR>
     * 現在のステータス条件に設定します.<BR>
     * このメソッドは、現在のステータスがキューフルや、
     * キュー警告の場合に、その条件を設定します.
     */
    public void checkStatus() ;
    
    /**
     * コミットデータ存在参考値を取得.
     * <BR><BR>
     * コミットデータ存在参考値を取得します.<BR>
     * この情報は、以前コミット処理により、メッセージ情報が設定された
     * 場合に、[true]がセットされます.<BR>
     * そして、この情報が[true]を返されたとしても、以前データを取得したことに
     * より、コミットされた情報は0件の場合もあるため、参考値扱いとなります.
     * <BR>
     * @return boolean コミットデータ存在参考値が返されます.<BR>
     *                 [true]が返された場合、コミットされたデータが存在するかも
     *                 知れません.<BR>
     *                 [false]が返された場合はコミットされた情報は存在しません.
     */
    public boolean isCommitReference() ;
    
    /**
     * コミットされたメッセージ情報が存在するかチェック.
     * <BR><BR>
     * コミットされたメッセージ情報が存在するかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、情報は存在します.<BR>
     *                 [false]が返された場合、情報は存在しません.
     */
    public boolean isCommitMessage() ;
    
    /**
     * コミットされたメッセージ情報が存在するかチェック.
     * <BR><BR>
     * コミットされたメッセージ情報が存在するかチェックします.
     * <BR>
     * @param key 対象のキー条件を設定します.<BR>
     *            [null]や[""]を設定した場合、キー条件を無視します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、情報は存在します.<BR>
     *                 [false]が返された場合、情報は存在しません.
     */
    public boolean isCommitMessage( String key ) ;
    
    /**
     * 対象キューに設定可能かチェック.
     * <BR><BR>
     * 対象キューに設定可能かチェックします.
     * <BR>
     * @param len 対象の電文長を設定します.
     * @exception MessageOverException メッセージオーバー例外.
     * @exception QueueFullException キューフル例外.
     */
    public void isPut( int len )
        throws MessageOverException,QueueFullException ;
}

