/*
 * @(#)QSendSeq.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue.base.core ;

import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.SequenceID;

/**
 * キュー送信ID管理.
 *  
 * @version 2007/01/30
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class QSendSeq {
    
    /**
     * ID管理.
     */
    private SequenceID sequence = null ;
    
    /**
     * 今回ID.
     */
    private int nowId = 0 ;
    
    /**
     * 次ID.
     */
    private int nextId = 0 ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public QSendSeq() {
        sync.create() ;
        sequence = new SequenceID( 1,99999999 ) ;
        nowId = 0 ;
        nextId = 0 ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    public void finalize() throws Exception {
        nextId = 0 ;
        nowId = 0 ;
        sequence = null ;
        sync.clear() ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear() {
        try {
            synchronized( sync.get() ) {
                if( sequence != null ) {
                    nextId = 0 ;
                    nowId = 0 ;
                    sequence.clear() ;
                }
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * 新しいシーケンスIDを発行.
     * <BR><BR>
     * 新しいシーケンスIDを発行します.
     * <BR>
     * @return int 新たに発行されたID情報が返されます.
     */
    public int sequenceId() {
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                
                // 送信開始時は、NowIDは0番指定.
                // （無条件受け取り）
                if( nextId == 0 ) {
                    nowId = sequence.getID() ;
                    nextId = sequence.getID() ;
                }
                // 送信中は、次IDを今回IDとし、
                // 次IDに新しいシーケンスIDをセット.
                else {
                    nowId = nextId ;
                    nextId = sequence.getID() ;
                }
                
                ret = nowId ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 今回ID情報を取得.
     * <BR><BR>
     * 今回ID情報を取得します.
     * <BR>
     * @return int 今回ID情報が返されます.
     */
    public int getNowId() {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                ret = nowId ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 次ID情報を取得.
     * <BR><BR>
     * 次ID情報を取得します.
     * <BR>
     * @return int 次ID情報が返されます.
     */
    public int getNextId() {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                ret = nextId ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 新たにIDを発行して、複合ID情報を取得.
     * <BR><BR>
     * 新たにIDを発行して、複合ID情報を取得します.
     * <BR>
     * @return long 複合ID情報が返されます.
     */
    public long sequenceNowIdAll() {
        
        long ret = -1L ;
        
        try {
            synchronized( sync.get() ) {
                ret = ( long )(
                    ( ( long )this.sequenceId() & 0x000000007fffffffL ) |
                    ( ( ( long )this.getNextId() & 0x000000007fffffffL ) << 32L ) ) ;
            }
        } catch( Exception e ) {
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * 複合ID情報を取得.
     * <BR><BR>
     * 複合ID情報を取得します.
     * <BR>
     * @return long 複合ID情報が返されます.
     */
    public long getNowIdAll() {
        
        long ret = -1L ;
        
        try {
            synchronized( sync.get() ) {
                ret = ( long )(
                    ( ( long )nowId & 0x000000007fffffffL ) |
                    ( ( ( long )nextId & 0x000000007fffffffL ) << 32L ) ) ;
            }
        } catch( Exception e ) {
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * 複合IDから、今回IDを取得.
     * <BR><BR>
     * 複合IDから、今回IDを取得します.
     * <BR>
     * @param nowNextId 複合IDを設定します.
     * @return int 今回ID情報が返されます.
     */
    public static final int getNowNextIdByNowId( long nowNextId ) {
        
        return ( int )( nowNextId & 0x000000007fffffffL ) ;
        
    }
    
    /**
     * 複合IDから、次IDを取得.
     * <BR><BR>
     * 複合IDから、次IDを取得します.
     * <BR>
     * @param nowNextId 複合IDを設定します.
     * @return int 次ID情報が返されます.
     */
    public static final int getNowNextIdByNextId( long nowNextId ) {
        
        return ( int )( ( nowNextId & 0x7fffffff00000000L ) >> 32L ) ;
        
    }
    
}

