/*
 * @(#)MaachangQManagerImple.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access ;

import java.net.InetAddress;
import java.util.ArrayList;

import org.maachang.queue.access.net.ConnectObject;


/**
 * キューマネージャ接続.
 * <BR><BR>
 * 接続中のキューマネージャを表すオブジェクト.
 *  
 * @version 2007/01/02
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class MaachangQManagerImple implements MaachangQManager {
    
    /**
     * 通信オブジェクト.
     */
    private ConnectObject connect = null ;
    
    /**
     * 接続キューマネージャ.
     */
    private String manager = null ;
    
    /**
     * コネクション中キューオブジェクト.
     */
    private ArrayList connectList = null ;
    
    
    
    /**
     * コンストラクタ.
     */
    private MaachangQManagerImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続条件を設定して、オブジェクトを生成します.
     * <BR>
     * @param connect 対象の通信オブジェクトを設定します.
     * @param manager 対象のキューマネージャを設定します.
     * @exception Exception 例外.
     */
    public MaachangQManagerImple( ConnectObject connect,String manager )
        throws Exception {
        
        if( connect == null || connect.isClose() == true ||
            manager == null || manager.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        this.connect = connect ;
        this.manager = manager ;
        this.connectList = new ArrayList() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.close() ;
    }
    
    /**
     * MQマネージャクローズ.
     * <BR><BR>
     * 接続中のMQマネージャをクローズします.
     */
    public void close() {
        if( this.connect != null ) {
            try {
                this.rollback() ;
            } catch( Exception e ) {
            }
            try {
                this.connect.destroy() ;
            } catch( Exception e ) {
            }
        }
        this.connect = null ;
        this.manager = null ;
        this.connectList = null ;
    }
    
    /**
     * コミット処理.
     * <BR><BR>
     * コミット処理を行います.
     * <BR>
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized void commit() throws MaachangQException {
        this.commitOrRollback( true ) ;
    }
    
    /**
     * ロールバック処理.
     * <BR><BR>
     * ロールバック処理を行います.
     * <BR>
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized void rollback() throws MaachangQException {
        this.commitOrRollback( false ) ;
    }
    
    /**
     * 対象送信キューにコネクション.
     * <BR><BR>
     * オープン中のキューマネージャ内の送信キューに接続します.
     * <BR>
     * @param name コネクション対象のキュー名を設定します.
     * @return ConnectMq 接続された送信Mqオブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized ConnectSendMq connectSendQueue( String name )
        throws MaachangQException {
        
        if( this.isOpen() == false ) {
            throw new MaachangQException(
                "対象キューマネージャは既にクローズしています",
                MaachangQErrorCode.ERROR_CLOSE_QUEUE_MANAGER ) ;
        }
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new MaachangQException( "キュー名は不正です" ) ;
        }
        
        ConnectSendMq ret = null ;
        int no = this.checkMq( name,MaachangQAccessDefine.TYPE_SEND ) ;
        
        if( no == -1 ) {
            
            boolean mode = ExecutionClientMq.checkQueue(
                connect,manager,name,MaachangQAccessDefine.TYPE_SEND ) ;
            
            ret = new ConnectSendMqImple( this,connect,manager,name,mode ) ;
            this.addMq( ret ) ;
            
        }
        else {
            ret = ( ConnectSendMq )connectList.get( no ) ;
        }
        
        return ret ;
    }
    
    /**
     * 対象受信キューにコネクション.
     * <BR><BR>
     * オープン中のキューマネージャ内の受信キューに接続します.
     * <BR>
     * @param name コネクション対象の受信キュー名を設定します.
     * @return ConnectMq 接続された受信Mqオブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized ConnectReceiveMq connectReceiveQueue( String name )
        throws MaachangQException {
        
        if( this.isOpen() == false ) {
            throw new MaachangQException(
                "対象キューマネージャは既にクローズしています",
                MaachangQErrorCode.ERROR_CLOSE_QUEUE_MANAGER ) ;
        }
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new MaachangQException( "キュー名は不正です" ) ;
        }
        
        boolean mode ;
        
        try {
            mode = ExecutionClientMq.checkQueue(
                connect,manager,name,MaachangQAccessDefine.TYPE_RECEIVE ) ;
        } catch( MaachangQException fa ) {
            if( fa.getErrorCode() == MaachangQErrorCode.ERROR_NOT_QUEUE ) {
                int no = this.checkMq( name,MaachangQAccessDefine.TYPE_RECEIVE ) ;
                if( no != -1 ) {
                    connectList.remove( no ) ;
                }
            }
            throw fa ;
        }
        
        ConnectReceiveMq ret = null ;
        int no = this.checkMq( name,MaachangQAccessDefine.TYPE_RECEIVE ) ;
        
        if( no == -1 ) {
            
            ret = new ConnectReceiveMqImple( this,connect,manager,name,mode ) ;
            this.addMq( ret ) ;
            
        }
        else {
            ret = ( ConnectReceiveMq )connectList.get( no ) ;
        }
        
        return ret ;
    }
    
    /**
     * 対象キューマネージャ名を取得.
     * <BR><BR>
     * 接続中のキューマネージャ名を取得します.
     * <BR>
     * @return String 対象キューマネージャ名が返されます.
     */
    public synchronized String getManagerName() {
        
        return manager ;
        
    }
    
    /**
     * 接続先アドレスを取得.
     * <BR><BR>
     * 接続先のアドレスを取得します.
     * <BR>
     * @return InetAddress 接続先アドレスが返されます.
     */
    public synchronized InetAddress getInetAddress() {
        
        try {
            return connect.getRemoteAddress() ;
        } catch( Exception e ) {
            return null ;
        }
        
    }
    
    /**
     * 接続先ポート番号を取得.
     * <BR><BR>
     * 接続先ポート番号を取得します.
     * <BR>
     * @return int 接続先ポート番号が返されます.
     */
    public synchronized int getPort() {
        
        try {
            return connect.getRemotePort() ;
        } catch( Exception e ) {
            return -1 ;
        }
        
    }
    
    /**
     * 通信状態を取得.
     * <BR><BR>
     * 通信状態を取得します.<BR>
     * この情報は１秒あたりの受信バイト値を表す情報です.
     * <BR>
     * @return long １秒あたりの受信バイト値が返されます.
     */
    public synchronized long getSecondToByte() {
        try {
            return connect.getSecondByReceiveByte() ;
        } catch( Exception e ) {
            return 0L ;
        }
    }
    
    /**
     * キューマネージャがオープン中かチェック.
     * <BR><RB>
     * 対象のキューマネージャがオープン中であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オープン中です.<BR>
     *                 [false]が返された場合、クローズしています.
     */
    public synchronized boolean isOpen() {
        
        if( connect != null ) {
            return ( connect.isClose() == false ) ? true : false ;
        }
        return false ;
        
    }
    
    /**
     * 新しいMQを追加.
     */
    private final void addMq( ConnectMq mq ) {
        
        connectList.add( mq ) ;
        
    }
    
    /**
     * 対象MQが存在するかチェック.
     */
    private final int checkMq( String name,int type ) {
        
        ConnectMq conn = null ;
        int len = connectList.size() ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            
            conn = ( ConnectMq )connectList.get( i ) ;
            if( name.equals( conn.getName() ) == true &&
                type == conn.getQueueType() ) {
                return i ;
            }
            
        }
        
        return -1 ;
    }
    
    /**
     * コミット or ロールバック.
     */
    private final void commitOrRollback( boolean mode )
        throws MaachangQException {
        
        ConnectMq conn = null ;
        int len = connectList.size() ;
        MaachangQException lastException = null ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            
            conn = ( ConnectMq )connectList.get( i ) ;
            if( conn.isTransaction() == true ) {
                try {
                    if( mode == true ) {
                        conn.commit() ;
                    }
                    else {
                        conn.rollback() ;
                    }
                } catch( MaachangQException fe ) {
                    lastException = fe ;
                }
            }
            
        }
        
        if( lastException != null ) {
            throw lastException ;
        }
        
    }
    
}

