/*
 * @(#)MessageTelegram.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access ;

import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.MessageBean;
import org.maachang.queue.access.protocol.MessageProtocol;

/**
 * データ電文処理.
 *  
 * @version 2007/01/10
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class MessageTelegram {
    
    /**
     * ダミーメッセージ.
     */
    private static final byte[] DUMMY = new byte[] {
        ( byte )0xfe
    } ;
    
    /**
     * マネージャ存在確認電文を作成.
     * <BR><BR>
     * マネージャ存在確認電文を生成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param manager 対象のマネージャ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createCheckManager( int[] out,String manager )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return MessageProtocol.createProtocol(
                out[ 0 ],                               // 電文ID.
                MessageProtocol.TYPE_CONNECT_MANAGER,   // 処理タイプ.
                manager,                                // キューマネージャ名.
                CommonProtocol.DUMMY,                   // not.
                MaachangQAccessDefine.TYPE_SEND,           // not.
                null,                                   // not.
                null,                                   // not.
                DUMMY,                                  // not.
                -1,                                     // not.
                -1                                      // not.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * キュー存在確認電文を作成.
     * <BR><BR>
     * キュー存在確認電文を生成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param manager 対象のマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createCheckQueue( int[] out,String manager,
        String queue,int queueType )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return MessageProtocol.createProtocol(
                out[ 0 ],                           // 電文ID.
                MessageProtocol.TYPE_CONNECT_QUEUE, // 処理タイプ.
                manager,                            // キューマネージャ名.
                queue,                              // キュー名.
                queueType,                          // キュータイプ.
                null,                               // not.
                null,                               // not.
                DUMMY,                              // not.
                -1,                                 // not.
                -1                                  // not.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 送信メッセージ情報電文を作成.
     * <BR><BR>
     * 送信メッセージ情報電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @param key 対象のキーコードを設定します.
     * @param processId 対象のプロセスIDを設定します.
     * @param message 対象のメッセージ情報を設定します.
     * @param expire 対象のExpire情報を設定します.
     * @param priority 対象のプライオリティ値を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createSendMessage( int[] out,String manager,
        String queue,int queueType,String key,String processId,byte[] message,
        int expire,int priority )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return MessageProtocol.createProtocol(
                out[ 0 ],                       // 電文ID.
                MessageProtocol.TYPE_SEND,      // 処理タイプ.
                manager,                        // キューマネージャ名.
                queue,                          // キュー名.
                queueType,                      // キュータイプ.
                key,                            // メッセージKey.
                processId,                      // プロセスID.
                message,                        // メッセージ.
                expire,                         // expire.
                priority                        // 優先順位.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 受信メッセージ情報電文を作成.
     * <BR><BR>
     * 受信メッセージ情報電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param key 対象のキーコードを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createReceiveMessage( int[] out,String manager,
        String queue,String key )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return MessageProtocol.createProtocol(
                out[ 0 ],                           // 電文ID.
                MessageProtocol.TYPE_RECEIVE,       // 処理タイプ.
                manager,                            // キューマネージャ名.
                queue,                              // キュー名.
                MaachangQAccessDefine.TYPE_RECEIVE,    // 受信キュータイプ.
                key,                                // メッセージKey.
                null,                               // not.
                DUMMY,                              // not.
                -1,                                 // not.
                -1                                  // not.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * トランザクション電文を作成.
     * <BR><BR>
     * トランザクション電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createTransaction( int[] out,String manager,
        String queue,int queueType )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return MessageProtocol.createProtocol(
                out[ 0 ],                           // 電文ID.
                MessageProtocol.TYPE_TRANSACTION,   // 処理タイプ.
                manager,                            // キューマネージャ名.
                queue,                              // キュー名.
                queueType,                          // キュータイプ.
                null,                               // not.
                null,                               // not.
                DUMMY,                              // not.
                -1,                                 // not.
                -1                                  // not.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * コミット情報電文を作成.
     * <BR><BR>
     * コミット情報電文を作成します.
     * <BR>
     * @param mode 対象の処理モードを設定します.<BR>
     *             [true]を設定した場合、コミットします.<BR>
     *             [false]を設定した場合、ロールバックします.
     * @param out 対象の電文IDが格納されます.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createCommitOrRollback( boolean mode,int[] out,String manager,
        String queue,int queueType )
        throws Exception {
        
        int execType = ( mode == true ) ?
            MessageProtocol.TYPE_COMMIT :
            MessageProtocol.TYPE_ROLLBACK ;
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return MessageProtocol.createProtocol(
                out[ 0 ],       // 電文ID.
                execType,       // 処理タイプ.
                manager,        // キューマネージャ名.
                queue,          // キュー名.
                queueType,      // キュータイプ.
                null,           // not.
                null,           // not.
                DUMMY,          // not.
                -1,             // not.
                -1              // not.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * データ電文解析.
     * <BR><BR>
     * 受信電文をデータ電文として解析します.
     * <BR>
     * @param out メッセージBeanを設定します.
     * @param telegram 受信電文を設定します.
     * @exception Exception 例外.
     */
    public static final void analysisDataTelegram( MessageBean out,byte[] telegram )
        throws Exception {
        
        if( out == null ) {
            throw new IllegalArgumentException( "out引数は不正です" ) ;
        }
        
        int[] outId = new int[ 1 ] ;
        byte[] bin = CommonProtocol.analysisProtocol( null,outId,telegram ) ;
        MessageProtocol.analysisProtocol( out,bin ) ;
        
        out.setId( outId[ 0 ] ) ;
    }
    
}

