/*
 * @(#)AdminCacheImple.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.CacheInfo;
import org.maachang.queue.access.admin.info.CreateCacheInfo;


/**
 * キャッシュ管理情報.
 * <BR><BR>
 * キャッシュ管理情報を表すインターフェイス.
 *  
 * @version 2007/01/02
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class AdminCacheImple implements AdminCache {
    
    /**
     * コネクションオブジェクト.
     */
    private MaachangQConnectAdminImple connect = null ;
    
    /**
     * コンストラクタ.
     */
    private AdminCacheImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * @param connect 対象オブジェクトを設定します.
     */
    protected AdminCacheImple( MaachangQConnectAdminImple connect ) {
        this.connect = connect ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.connect = null ;
    }
    
    /**
     * キャッシュ情報を追加.
     * <BR><BR>
     * 新しいキャッシュ情報を生成します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param info キャッシュ情報生成条件を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void addCache( CreateCacheInfo info )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionCache.addCache(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            info ) ;
    }
    
    /**
     * キャッシュを削除.
     * <BR><BR>
     * キャッシュを削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 削除対象のキャッシュ名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeCache( String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionCache.removeCache(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name ) ;
    }
    
    /**
     * キャッシュ情報を取得.
     * <BR><BR>
     * キャッシュ情報を取得します.
     * <BR>
     * @param name 対象のキャッシュ名を設定します.
     * @return CacheInfo 対象のキャッシュ情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public CacheInfo getCache( String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        
        return ExecutionCache.getCache(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name ) ;
    }
    
    /**
     * キャッシュ名一覧を取得.
     * <BR><BR>
     * キャッシュ名一覧を取得します.
     * <BR>
     * @return String[] キャッシュ名一覧が返されます.
     */
    public String[] getNames() {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        
        try {
            return ExecutionCache.getNames(
                connect.getConnect(),
                connect.getUserId(),
                connect.isRootOwner() ) ;
        } catch( Exception e ) {
            return null ;
        }
    }
    
    /**
     * キャッシュ情報サイズを取得.
     * <BR><BR>
     * キャッシュ情報サイズを取得します.
     * <BR>
     * @return int キャッシュ情報数が返されます.
     */
    public int size() {
        if( connect == null || connect.isOpen() == false ) {
            return 0 ;
        }
        
        try {
            return ExecutionCache.getSize(
                connect.getConnect(),
                connect.getUserId(),
                connect.isRootOwner() ) ;
        } catch( Exception e ) {
            return 0 ;
        }
    }
    
    /**
     * キャッシュ名が存在するかチェック.
     * <BR><BR>
     * キャッシュ名が存在するかチェックします.
     * <BR>
     * @param name チェック対象のキャッシュ名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、存在します.<BR>
     *                 [false]が返された場合、存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isCache( String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return false ;
        }
        
        return ExecutionCache.isCache(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name ) ;
    }
    
    /**
     * 管理コネクションオブジェクトを取得.
     * <BR><BR>
     * 管理コネクションオブジェクトを取得します.
     * <BR>
     * @return MaachangQConnectAdmin 管理コネクションオブジェクトが返されます.
     */
    public MaachangQConnectAdmin getMaachangQConnectAdmin() {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        return connect ;
    }
    
}

