/*
 * @(#)AdminChannelImple.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.ChannelInfo;
import org.maachang.queue.access.admin.info.ChannelKeyInfo;
import org.maachang.queue.access.admin.info.CreateChannelInfo;
import org.maachang.queue.access.admin.info.SendChannelInfo;


/**
 * チャネル管理情報.
 * <BR><BR>
 * チャネル管理情報を表すインターフェイス.
 *  
 * @version 2007/01/03
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class AdminChannelImple implements AdminChannel {
    
    /**
     * コネクションオブジェクト.
     */
    private MaachangQConnectAdminImple connect = null ;
    
    /**
     * コンストラクタ.
     */
    private AdminChannelImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * @param connect 対象オブジェクトを設定します.
     */
    protected AdminChannelImple( MaachangQConnectAdminImple connect ) {
        this.connect = connect ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.connect = null ;
    }
    
    /**
     * チャネル情報を追加.
     * <BR><BR>
     * 新しいチャネル情報を追加します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param info チャネル情報生成条件を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void addChannel( CreateChannelInfo info )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionChannel.addChannel(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            info ) ;
    }
    
    /**
     * 送信チャネル情報を変更.
     * <BR><BR>
     * 送信チャネル情報を変更します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param info 送信チャネル情報生成条件を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void setSendChannel( SendChannelInfo info )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionChannel.setSendChannel(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            info ) ;
    }
    
    /**
     * 送信チャネル情報を削除.
     * <BR><BR>
     * 送信チャネル情報を削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 削除対象の送信チャネル名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeSendChannel( String name )
        throws MaachangQException {
        this.removeChannel(
            MaachangQAccessDefine.TYPE_SEND,name ) ;
    }
    
    /**
     * 受信チャネル情報を削除.
     * <BR><BR>
     * 受信チャネル情報を削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 削除対象の受信チャネル名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeReceiveChannel( String name )
        throws MaachangQException {
        this.removeChannel(
            MaachangQAccessDefine.TYPE_RECEIVE,name ) ;
    }
    
    /**
     * チャネル情報を削除.
     * <BR><BR>
     * チャネル情報を削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param type 削除対象のチャネルタイプを設定します.
     * @param name 削除対象のチャネル名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeChannel( int type,String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionChannel.removeChannel(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name,type ) ;
    }
    
    /**
     * 送信チャネル情報を停止.
     * <BR><BR>
     * 送信チャネル情報を停止します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 対象の送信チャネル名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void stopSendChannel( String name )
        throws MaachangQException {
        this.stopChannel(
            MaachangQAccessDefine.TYPE_SEND,name ) ;
    }
    
    /**
     * 受信チャネル情報を停止.
     * <BR><BR>
     * 受信チャネル情報を停止します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 対象の受信チャネル名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void stopReceiveChannel( String name )
        throws MaachangQException {
        this.stopChannel(
            MaachangQAccessDefine.TYPE_RECEIVE,name ) ;
    }
    
    /**
     * 指定チャネルのステータスを停止に設定.
     * <BR><BR>
     * 指定チャネルのステータスを停止に設定します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param type 対象のチャネルタイプを設定します.
     * @param name 対象のチャネル名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void stopChannel( int type,String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionChannel.stopChannel(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name,type ) ;
    }
    
    /**
     * 送信チャネル情報を開始.
     * <BR><BR>
     * 送信チャネル情報を開始します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 対象の送信チャネル名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void startSendChannel( String name )
        throws MaachangQException {
        this.startChannel(
            MaachangQAccessDefine.TYPE_SEND,name ) ;
    }
    
    /**
     * 受信チャネル情報を開始.
     * <BR><BR>
     * 受信チャネル情報を開始します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 対象の受信チャネル名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void startReceiveChannel( String name )
        throws MaachangQException {
        this.startChannel(
            MaachangQAccessDefine.TYPE_RECEIVE,name ) ;
    }
    
    /**
     * 指定チャネルのステータスを開始.
     * <BR><BR>
     * 停止状態のチャネルを開始します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param type 対象のチャネルタイプを設定します.
     * @param name 対象のチャネル名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void startChannel( int type,String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionChannel.startChannel(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name,type ) ;
    }
    
    /**
     * 送信チャネル情報を取得.
     * <BR><BR>
     * 送信チャネル情報を取得します.
     * <BR>
     * @param name 対象の送信チャネル名を設定します.
     * @return SendChannelInfo 対象の送信チャネル情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public SendChannelInfo getSendChannel( String name )
        throws MaachangQException {
        
        ChannelInfo info = this.getChannel(
            MaachangQAccessDefine.TYPE_SEND,name ) ;
        
        if( info != null && info instanceof SendChannelInfo ) {
            return ( SendChannelInfo )info ;
        }
        
        return null ;
    }
    
    /**
     * 受信チャネル情報を取得.
     * <BR><BR>
     * 受信チャネル情報を取得します.
     * <BR>
     * @param name 対象の受信チャネル名を設定します.
     * @return ChannelInfo 対象の受信チャネル情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public ChannelInfo getReceiveChannel( String name )
        throws MaachangQException {
        return this.getChannel(
            MaachangQAccessDefine.TYPE_RECEIVE,name ) ;
    }
    
    /**
     * チャネル情報を取得.
     * <BR><BR>
     * チャネル情報を取得します.
     * <BR>
     * @param type 対象のチャネルタイプを設定します.
     * @param name 対象のチャネル名を設定します.
     * @return ChannelInfo 対象のチャネル情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public ChannelInfo getChannel( int type,String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        
        return ExecutionChannel.getChannel(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name,type ) ;
    }
    
    /**
     * 指定送信チャネル名のステータスを取得.
     * <BR><BR>
     * 指定送信チャネル名のステータスを取得します.
     * <BR>
     * @param name 対象の送信チャネル名を設定します.
     * @return int 対象チャネルのステータスが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public int getSendByState( String name )
        throws MaachangQException {
        return this.getState(
            MaachangQAccessDefine.TYPE_SEND,name ) ;
    }
    
    /**
     * 指定受信チャネル名のステータスを取得.
     * <BR><BR>
     * 指定受信チャネル名のステータスを取得します.
     * <BR>
     * @param name 対象の受信チャネル名を設定します.
     * @return int 対象チャネルのステータスが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public int getReceiveByState( String name )
        throws MaachangQException {
        return this.getState(
            MaachangQAccessDefine.TYPE_RECEIVE,name ) ;
    }
    
    /**
     * 指定チャネル名のステータスを取得.
     * <BR><BR>
     * 指定チャネル名のステータスを取得します.
     * <BR>
     * @param type 対象のチャネルタイプを設定します.
     * @param name 対象のチャネル名を設定します.
     * @return int 対象チャネルのステータスが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public int getState( int type,String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return -1 ;
        }
        
        ChannelInfo info = ExecutionChannel.getChannel(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name,type ) ;
        
        if( info != null ) {
            return info.getState() ;
        }
        
        return -1 ;
    }
    
    /**
     * チャネル情報一覧を取得.
     * <BR><BR>
     * 対象のチャネル情報一覧が返されます.
     * <BR>
     * @return ChannelKeyInfo[] チャネル情報一覧が返されます.
     */
    public ChannelKeyInfo[] getKeys() {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        
        try {
            return ExecutionChannel.getKeys(
                connect.getConnect(),
                connect.getUserId(),
                connect.isRootOwner() ) ;
        } catch( Exception e ) {
            return null ;
        }
    }
    
    /**
     * チャネル情報数を取得.
     * <BR><BR>
     * 対象のチャネル情報数を取得します.
     * <BR>
     * @return int チャネル情報数が返されます.
     */
    public int size() {
        if( connect == null || connect.isOpen() == false ) {
            return 0 ;
        }
        
        try {
            return ExecutionChannel.getSize(
                connect.getConnect(),
                connect.getUserId(),
                connect.isRootOwner() ) ;
        } catch( Exception e ) {
            return 0 ;
        }
    }
    
    /**
     * 対象送信チャネル名が存在するかチェック.
     * <BR><BR>
     * 対象の送信チャネル名が存在するかチェックします.
     * <BR>
     * @param name 対象の送信チャネル名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象チャネルは存在します.<BR>
     *                 [false]が返された場合、対象チャネルは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isSendChannel( String name )
        throws MaachangQException {
        return this.isChannel(
            MaachangQAccessDefine.TYPE_SEND,name ) ;
    }
    
    /**
     * 対象受信チャネル名が存在するかチェック.
     * <BR><BR>
     * 対象の受信チャネル名が存在するかチェックします.
     * <BR>
     * @param name 対象の受信チャネル名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象チャネルは存在します.<BR>
     *                 [false]が返された場合、対象チャネルは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isReceiveChannel( String name )
        throws MaachangQException {
        return this.isChannel(
            MaachangQAccessDefine.TYPE_RECEIVE,name ) ;
    }
    
    /**
     * 対象チャネル名が存在するかチェック.
     * <BR><BR>
     * 対象のチャネル名が存在するかチェックします.
     * <BR>
     * @param type 対象のチャネルタイプを設定します.
     * @param name 対象のチャネル名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象チャネルは存在します.<BR>
     *                 [false]が返された場合、対象チャネルは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isChannel( int type,String name )
        throws MaachangQException  {
        if( connect == null || connect.isOpen() == false ) {
            return false ;
        }
        
        return ExecutionChannel.isChannel(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name,type ) ;
    }
    
    /**
     * 管理コネクションオブジェクトを取得.
     * <BR><BR>
     * 管理コネクションオブジェクトを取得します.
     * <BR>
     * @return MaachangQConnectAdmin 管理コネクションオブジェクトが返されます.
     */
    public MaachangQConnectAdmin getMaachangQConnectAdmin() {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        return connect ;
    }
    
}

