/*
 * @(#)AdminQueue.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.CreateQueueInfo;
import org.maachang.queue.access.admin.info.QueueInfo;
import org.maachang.queue.access.admin.info.QueueKeyInfo;
import org.maachang.queue.access.admin.info.SendQueueInfo;


/**
 * キュー管理情報.
 * <BR><BR>
 * キュー管理情報を表すインターフェイス.
 *  
 * @version 2007/01/02
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public interface AdminQueue {
    
    /**
     * キュー情報を追加.
     * <BR><BR>
     * キュー情報を追加します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param info キュー情報生成条件を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void addQueue( CreateQueueInfo info )
        throws MaachangQException ;
    
    /**
     * 送信キュー情報に対する内容を変更.
     * <BR><BR>
     * 送信キュー情報に対する内容を変更します.
     * <BR>
     * @param info 変更対象のキュー情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void setQueue( QueueInfo info )
        throws MaachangQException ;
    
    /**
     * 送信キュー情報を削除.
     * <BR><BR>
     * 送信キュー情報を削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 削除対象の送信キュー名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeSendQueue( String name )
        throws MaachangQException ;
    
    /**
     * 受信キュー情報を削除.
     * <BR><BR>
     * 受信キュー情報を削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 削除対象の受信キュー名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeReceiveQueue( String name )
        throws MaachangQException ;
    
    /**
     * キュータイプを設定して、キュー情報を削除.
     * <BR><BR>
     * キュータイプを設定して、キュー情報を削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param type 削除対象のキュータイプを設定します.
     * @param name 削除対象のキュー名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeQueue( int type,String name )
        throws MaachangQException ;
    
    /**
     * 送信キュー情報を取得.
     * <BR><BR>
     * 送信キュー情報を取得します.
     * <BR>
     * @param name 対象の送信キュー名を設定します.
     * @return SendQueueInfo 対象の送信キュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public SendQueueInfo getSendQueue( String name )
        throws MaachangQException ;
    
    /**
     * 受信キュー情報を取得.
     * <BR><BR>
     * 受信キュー情報を取得します.
     * <BR>
     * @param name 対象の受信キュー名を設定します.
     * @return QueueInfo 対象の受信キュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public QueueInfo getReceiveQueue( String name )
        throws MaachangQException ;
    
    /**
     * キュータイプを設定して、キュー情報を取得.
     * <BR><BR>
     * キュータイプを設定して、キュー情報を取得します.
     * <BR>
     * @param type 対象のキュータイプを設定します.
     * @param name 対象のキュー名を設定します.
     * @return QueueInfo 対象のキュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public QueueInfo getQueue( int type,String name )
        throws MaachangQException ;
    
    /**
     * 指定送信キュー名のステータスを取得.
     * <BR><BR>
     * 指定送信キュー名のステータスを取得します.
     * <BR>
     * @param name 対象の送信キュー名を設定します.
     * @return int 対象送信キューのステータスが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public int getSendQueueByState( String name )
        throws MaachangQException ;
    
    /**
     * 指定受信キュー名のステータスを取得.
     * <BR><BR>
     * 指定受信キュー名のステータスを取得します.
     * <BR>
     * @param name 対象の受信キュー名を設定します.
     * @return int 対象受信キューのステータスが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public int getReceiveQueueByState( String name )
        throws MaachangQException ;
    
    /**
     * 指定キュー名のステータスを取得.
     * <BR><BR>
     * 指定キュー名のステータスを取得します.
     * <BR>
     * @param type 対象のキュータイプを設定します.
     * @param name 対象のキュー名を設定します.
     * @return int 対象キューのステータスが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public int getState( int type,String name )
        throws MaachangQException ;
    
    /**
     * キュー情報一覧を取得.
     * <BR><BR>
     * 対象のキュー情報一覧が返されます.
     * <BR>
     * @return QueueKeyInfo[] キュー情報一覧が返されます.
     */
    public QueueKeyInfo[] getKeys() ;
    
    /**
     * キュー情報数を取得.
     * <BR><BR>
     * 対象のキュー情報数を取得します.
     * <BR>
     * @return int キュー情報数が返されます.
     */
    public int size() ;
    
    /**
     * キューマネージャ名を取得.
     * <BR><BR>
     * 参照元のキューマネージャ名を取得します.
     * <BR>
     * @return String キューマネージャ名が返されます.
     */
    public String getQueueManagerName() ;
    
    /**
     * キューマネージャを取得.
     * <BR><BR>
     * 参照元のキューマネージャを取得します.
     * <BR>
     * @return AdminQueueManager キューマネージャが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public AdminQueueManager getQueueManager()
        throws MaachangQException ;
    
    /**
     * 対象送信キュー名が存在するかチェック.
     * <BR><BR>
     * 対象の送信キュー名が存在するかチェックします.
     * <BR>
     * @param name 対象の送信キュー名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象送信キューは存在します.<BR>
     *                 [false]が返された場合、対象送信キューは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isSendQueue( String name )
        throws MaachangQException ;
    
    /**
     * 対象受信キュー名が存在するかチェック.
     * <BR><BR>
     * 対象の受信キュー名が存在するかチェックします.
     * <BR>
     * @param name 対象の受信キュー名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象受信キューは存在します.<BR>
     *                 [false]が返された場合、対象受信キューは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isReceiveQueue( String name )
        throws MaachangQException ;
    
    /**
     * 対象キュー名が存在するかチェック.
     * <BR><BR>
     * 対象のキュー名が存在するかチェックします.
     * <BR>
     * @param type 対象のキュータイプを設定します.
     * @param name 対象のキュー名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象キューは存在します.<BR>
     *                 [false]が返された場合、対象キューは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isQueue( int type,String name )
        throws MaachangQException ;
    
    /**
     * 管理コネクションオブジェクトを取得.
     * <BR><BR>
     * 管理コネクションオブジェクトを取得します.
     * <BR>
     * @return MaachangQConnectAdmin 管理コネクションオブジェクトが返されます.
     */
    public MaachangQConnectAdmin getMaachangQConnectAdmin() ;
    
}

