/*
 * @(#)AdminQueueImple.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.CreateQueueInfo;
import org.maachang.queue.access.admin.info.QueueInfo;
import org.maachang.queue.access.admin.info.QueueKeyInfo;
import org.maachang.queue.access.admin.info.SendQueueInfo;


/**
 * キュー管理情報.
 * <BR><BR>
 * キュー管理情報を表すオブジェクト.
 *  
 * @version 2007/01/02
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class AdminQueueImple implements AdminQueue {
    
    /**
     * コネクションオブジェクト.
     */
    private MaachangQConnectAdminImple connect = null ;
    
    /**
     * キューマネージャ名.
     */
    private String manager = null ;
    
    /**
     * コンストラクタ.
     */
    private AdminQueueImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * @param connect 対象オブジェクトを設定します.
     */
    protected AdminQueueImple( MaachangQConnectAdminImple connect,String manager ) {
        this.connect = connect ;
        this.manager = manager ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.connect = null ;
        this.manager = null ;
    }
    
    /**
     * キュー情報を追加.
     * <BR><BR>
     * キュー情報を追加します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param info キュー情報生成条件を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void addQueue( CreateQueueInfo info )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionQueue.addQueue(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            manager,info ) ;
    }
    
    /**
     * 送信キュー情報に対する内容を変更.
     * <BR><BR>
     * 送信キュー情報に対する内容を変更します.
     * <BR>
     * @param info 変更対象のキュー情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void setQueue( QueueInfo info )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionQueue.setQueue(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            manager,info ) ;
    }
    
    /**
     * 送信キュー情報を削除.
     * <BR><BR>
     * 送信キュー情報を削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 削除対象の送信キュー名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeSendQueue( String name )
        throws MaachangQException {
        this.removeQueue(
            MaachangQAccessDefine.TYPE_SEND,name ) ;
    }
    
    /**
     * 受信キュー情報を削除.
     * <BR><BR>
     * 受信キュー情報を削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 削除対象の受信キュー名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeReceiveQueue( String name )
        throws MaachangQException {
        this.removeQueue(
            MaachangQAccessDefine.TYPE_RECEIVE,name ) ;
    }
    
    /**
     * キュータイプを設定して、キュー情報を削除.
     * <BR><BR>
     * キュータイプを設定して、キュー情報を削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param type 削除対象のキュータイプを設定します.
     * @param name 削除対象のキュー名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeQueue( int type,String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionQueue.removeQueue(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            manager,name,type ) ;
    }
    
    /**
     * 送信キュー情報を取得.
     * <BR><BR>
     * 送信キュー情報を取得します.
     * <BR>
     * @param name 対象の送信キュー名を設定します.
     * @return SendQueueInfo 対象の送信キュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public SendQueueInfo getSendQueue( String name )
        throws MaachangQException {
        QueueInfo info = this.getQueue(
            MaachangQAccessDefine.TYPE_SEND,name ) ;
        
        if( info != null && info instanceof SendQueueInfo ) {
            return ( SendQueueInfo )info ;
        }
        return null ;
    }
    
    /**
     * 受信キュー情報を取得.
     * <BR><BR>
     * 受信キュー情報を取得します.
     * <BR>
     * @param name 対象の受信キュー名を設定します.
     * @return QueueInfo 対象の受信キュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public QueueInfo getReceiveQueue( String name )
        throws MaachangQException {
        return this.getQueue(
            MaachangQAccessDefine.TYPE_RECEIVE,name ) ;
    }
    
    /**
     * キュータイプを設定して、キュー情報を取得.
     * <BR><BR>
     * キュータイプを設定して、キュー情報を取得します.
     * <BR>
     * @param type 対象のキュータイプを設定します.
     * @param name 対象のキュー名を設定します.
     * @return QueueInfo 対象のキュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public QueueInfo getQueue( int type,String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        
        return ExecutionQueue.getQueue(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            manager,name,type ) ;
    }
    
    /**
     * 指定送信キュー名のステータスを取得.
     * <BR><BR>
     * 指定送信キュー名のステータスを取得します.
     * <BR>
     * @param name 対象の送信キュー名を設定します.
     * @return int 対象送信キューのステータスが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public int getSendQueueByState( String name )
        throws MaachangQException {
        return this.getState(
            MaachangQAccessDefine.TYPE_SEND,name ) ;
    }
    
    /**
     * 指定受信キュー名のステータスを取得.
     * <BR><BR>
     * 指定受信キュー名のステータスを取得します.
     * <BR>
     * @param name 対象の受信キュー名を設定します.
     * @return int 対象受信キューのステータスが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public int getReceiveQueueByState( String name )
        throws MaachangQException {
        return this.getState(
            MaachangQAccessDefine.TYPE_RECEIVE,name ) ;
    }
    
    /**
     * 指定キュー名のステータスを取得.
     * <BR><BR>
     * 指定キュー名のステータスを取得します.
     * <BR>
     * @param type 対象のキュータイプを設定します.
     * @param name 対象のキュー名を設定します.
     * @return int 対象キューのステータスが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public int getState( int type,String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return -1 ;
        }
        
        QueueInfo ret =  ExecutionQueue.getQueue(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            manager,name,type ) ;
        
        if( ret != null ) {
            return ret.getState() ;
        }
        
        return -1 ;
    }
    
    /**
     * キュー情報一覧を取得.
     * <BR><BR>
     * 対象のキュー情報一覧が返されます.
     * <BR>
     * @return QueueKeyInfo[] キュー情報一覧が返されます.
     */
    public QueueKeyInfo[] getKeys() {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        
        try {
            return ExecutionQueue.getKeys(
                connect.getConnect(),
                connect.getUserId(),
                connect.isRootOwner(),
                manager ) ;
        } catch( Exception e ) {
            return null ;
        }
    }
    
    /**
     * キュー情報数を取得.
     * <BR><BR>
     * 対象のキュー情報数を取得します.
     * <BR>
     * @return int キュー情報数が返されます.
     */
    public int size() {
        if( connect == null || connect.isOpen() == false ) {
            return 0 ;
        }
        
        try {
            return ExecutionQueue.getSize(
                connect.getConnect(),
                connect.getUserId(),
                connect.isRootOwner(),
                manager ) ;
        } catch( Exception e ) {
            return 0 ;
        }
    }
    
    /**
     * キューマネージャ名を取得.
     * <BR><BR>
     * 参照元のキューマネージャ名を取得します.
     * <BR>
     * @return String キューマネージャ名が返されます.
     */
    public String getQueueManagerName() {
        return manager ;
    }
    
    /**
     * キューマネージャを取得.
     * <BR><BR>
     * 参照元のキューマネージャを取得します.
     * <BR>
     * @return AdminQueueManager キューマネージャが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public AdminQueueManager getQueueManager()
        throws MaachangQException {
        if( connect != null && connect.isOpen() == true ) {
            return connect.getAdminQueueManager() ;
        }
        return null ;
    }
    
    /**
     * 対象送信キュー名が存在するかチェック.
     * <BR><BR>
     * 対象の送信キュー名が存在するかチェックします.
     * <BR>
     * @param name 対象の送信キュー名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象送信キューは存在します.<BR>
     *                 [false]が返された場合、対象送信キューは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isSendQueue( String name )
        throws MaachangQException {
        return this.isQueue(
            MaachangQAccessDefine.TYPE_SEND,name ) ;
    }
    
    /**
     * 対象受信キュー名が存在するかチェック.
     * <BR><BR>
     * 対象の受信キュー名が存在するかチェックします.
     * <BR>
     * @param name 対象の受信キュー名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象受信キューは存在します.<BR>
     *                 [false]が返された場合、対象受信キューは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isReceiveQueue( String name )
        throws MaachangQException {
        return this.isQueue(
            MaachangQAccessDefine.TYPE_RECEIVE,name ) ;
    }
    
    /**
     * 対象キュー名が存在するかチェック.
     * <BR><BR>
     * 対象のキュー名が存在するかチェックします.
     * <BR>
     * @param type 対象のキュータイプを設定します.
     * @param name 対象のキュー名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象キューは存在します.<BR>
     *                 [false]が返された場合、対象キューは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isQueue( int type,String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return false ;
        }
        
        return ExecutionQueue.isQueue(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            manager,name,type ) ;
    }
    
    /**
     * 管理コネクションオブジェクトを取得.
     * <BR><BR>
     * 管理コネクションオブジェクトを取得します.
     * <BR>
     * @return MaachangQConnectAdmin 管理コネクションオブジェクトが返されます.
     */
    public MaachangQConnectAdmin getMaachangQConnectAdmin() {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        return connect ;
    }
    
}

