/*
 * @(#)AdminQueueManagerImple.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.CreateQueueManagerInfo;
import org.maachang.queue.access.admin.info.QueueManagerInfo;


/**
 * キューマネージャ管理情報.
 * <BR><BR>
 * キューマネージャ管理情報を表すオブジェクト.
 *  
 * @version 2007/01/02
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class AdminQueueManagerImple implements AdminQueueManager {
    
    /**
     * コネクションオブジェクト.
     */
    private MaachangQConnectAdminImple connect = null ;
    
    /**
     * コンストラクタ.
     */
    private AdminQueueManagerImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * @param connect 対象オブジェクトを設定します.
     */
    protected AdminQueueManagerImple( MaachangQConnectAdminImple connect ) {
        this.connect = connect ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.connect = null ;
    }
    
    /**
     * キューマネージャを追加.
     * <BR><BR>
     * 新しいキューマネージャを追加します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param info 対象のキューマネージャ生成情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void addManager( CreateQueueManagerInfo info )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionQueueManager.addQueueManager(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            info ) ;
    }
    
    /**
     * キューマネージャを削除.
     * <BR><BR>
     * 対象のキューマネージャを削除します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 削除対象のキューマネージャを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void removeManager( String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionQueueManager.removeQueueManager(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name ) ;
        
        // キュー管理情報を削除.
        connect.removeQueueManagerTo( name ) ;
    }
    
    /**
     * 指定キューマネージャのステータスを停止に設定.
     * <BR><BR>
     * 指定キューマネージャのステータスを停止に設定します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 対象のキューマネージャ名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void stopManager( String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionQueueManager.stopQueueManager(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name ) ;
    }
    
    /**
     * 指定キューマネージャのステータスを開始.
     * <BR><BR>
     * 停止状態のキューマネージャを開始します.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @param name 対象のキューマネージャ名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void startManager( String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return ;
        }
        
        ExecutionQueueManager.startQueueManager(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name ) ;
    }
    
    /**
     * キューマネージャ情報の取得.
     * <BR><BR>
     * キューマネージャ情報を取得します.
     * <BR>
     * @param name 対象のキューマネージャ名を設定します.
     * @return QueueManagerInfo キューマネージャ情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public QueueManagerInfo get( String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        
        return ExecutionQueueManager.getQueueManager(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name ) ;
    }
    
    /**
     * 指定キューマネージャのステータスを取得.
     * <BR><BR>
     * 指定キューマネージャのステータスを取得します.
     * <BR>
     * @param name キューマネージャ名を取得します.
     * @return int キューマネージャステータスが返されます.<BR>
     *             [QueueManagerStatus.STATE_SUCCESS]が返された場合、「正常」です.<BR>
     *             [QueueManagerStatus.STATE_DOWN]が返された場合、「停止」です.<BR>
     *             [QueueManagerStatus.STATE_STARTUP]が返された場合、「起動中」です.<BR>
     *             [QueueManagerStatus.STATE_SHUTDOWN]が返された場合、「停止」です.<BR>
     *             [QueueManagerStatus.STATE_ERROR]が返された場合、「異常」です.<BR>
     *             [QueueManagerStatus.STATE_PLANNED_STOP]が返された場合、「計画停止」です.
     * @exception MaachangQException MaachangQ例外.
     */
    public int getState( String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return -1 ;
        }
        
        QueueManagerInfo info = ExecutionQueueManager.getQueueManager(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name ) ;
        
        if( info != null ) {
            return info.getState() ;
        }
        
        return -1 ;
    }
    
    /**
     * キューマネージャ名群を取得.
     * <BR><BR>
     * キューマネージャ名群を取得します.
     * <BR>
     * @return String[] キューマネージャ名群が返されます.
     */
    public String[] getNames() {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        
        try {
            return ExecutionQueueManager.getNames(
                connect.getConnect(),
                connect.getUserId(),
                connect.isRootOwner() ) ;
        } catch( Exception e ) {
            return null ;
        }
    }
    
    /**
     * キューマネージャ数を取得.
     * <BR><BR>
     * キューマネージャ数を取得します.
     * <BR>
     * @return int キューマネージャ数が返されます.
     */
    public int size() {
        if( connect == null || connect.isOpen() == false ) {
            return 0 ;
        }
        
        try {
            return ExecutionQueueManager.getSize(
                connect.getConnect(),
                connect.getUserId(),
                connect.isRootOwner() ) ;
        } catch( Exception e ) {
            return 0 ;
        }
    }
    
    /**
     * キュー管理オブジェクトを取得.
     * <BR><BR>
     * キュー管理オブジェクトを取得します.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @return AdminQueue キュー管理オブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public AdminQueue getAdminQueue( String manager )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        
        return connect.getAdminQueue( manager ) ;
    }
    
    /**
     * キューマネージャが存在するかチェック.
     * <BR><BR>
     * 指定されたキューマネージャ名が存在するかチェックします.
     * <BR>
     * @param name 対象のキューマネージャ名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、キューマネージャは存在します.<BR>
     *                 [false]が返された場合、キューマネージャは存在しません.
     * @exception MaachangQException MaachangQ例外.
     */
    public boolean isQueueManager( String name )
        throws MaachangQException {
        if( connect == null || connect.isOpen() == false ) {
            return false ;
        }
        
        return ExecutionQueueManager.isQueueManager(
            connect.getConnect(),
            connect.getUserId(),
            connect.isRootOwner(),
            name ) ;
    }
    
    /**
     * 管理コネクションオブジェクトを取得.
     * <BR><BR>
     * 管理コネクションオブジェクトを取得します.
     * <BR>
     * @return MaachangQConnectAdmin 管理コネクションオブジェクトが返されます.
     */
    public MaachangQConnectAdmin getMaachangQConnectAdmin() {
        if( connect == null || connect.isOpen() == false ) {
            return null ;
        }
        return connect ;
    }
    
}

