/*
 * @(#)CacheTelegram.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.admin.info.CreateCacheInfo;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.admin.AdminCacheBean;
import org.maachang.queue.access.protocol.admin.AdminCacheProtocol;


/**
 * 管理者キャッシュ用電文.
 *  
 * @version 2007/01/12
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class CacheTelegram {
    
    /**
     * 新しいキャッシュを追加する電文を作成.
     * <BR><BR>
     * 新しいキャッシュを追加する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param info 対象キャッシュ情報を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createAddCache(
        int[] out,boolean loginUserOwner,int userId,CreateCacheInfo info )
        throws Exception {
        
        if( info == null || info.getName() == null ) {
            throw new IllegalArgumentException( "キャッシュ生成用条件は不正です" ) ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminCacheProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                AdminCacheProtocol.TYPE_ADD_CACHE,              // 処理タイプ.
                info.getName(),                                 // キャッシュ名.
                info.getSize(),                                 // キャッシュサイズ.
                -1,                                             // 現在キャッシュサイズなし.
                -1,                                             // サイズなし.
                null                                            // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * キャッシュを削除する電文を作成.
     * <BR><BR>
     * キャッシュを削除する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象キャッシュ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createRemoveCache(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キャッシュ名は不正です" ) ;
        }
        
        return CacheTelegram.createNameByCache(
            out,loginUserOwner,userId,name,
            AdminCacheProtocol.TYPE_REMOVE_CACHE ) ;
        
    }
    
    /**
     * キャッシュを取得する電文を作成.
     * <BR><BR>
     * キャッシュを取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象キャッシュ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetCache(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キャッシュ名は不正です" ) ;
        }
        
        return CacheTelegram.createNameByCache(
            out,loginUserOwner,userId,name,
            AdminCacheProtocol.TYPE_GET_CACHE ) ;
        
    }
    
    /**
     * 全てのキャッシュ名一覧を取得する電文を作成.
     * <BR><BR>
     * 全てのキャッシュ名一覧を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetNamesCache( int[] out,boolean loginUserOwner,int userId )
        throws Exception {
        
        return CacheTelegram.createNameByCache(
            out,loginUserOwner,userId,CommonProtocol.DUMMY,
            AdminCacheProtocol.TYPE_GET_CACHE_NAMES ) ;
        
    }
    
    /**
     * 全てのキャッシュ数を取得する電文を作成.
     * <BR><BR>
     * 全てのキャッシュ数を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetSizeCache( int[] out,boolean loginUserOwner,int userId )
        throws Exception {
        
        return CacheTelegram.createNameByCache(
            out,loginUserOwner,userId,CommonProtocol.DUMMY,
            AdminCacheProtocol.TYPE_GET_CACHE_SIZE ) ;
        
    }
    
    /**
     * キャッシュが存在するか確認する電文を作成.
     * <BR><BR>
     * キャッシュが存在するか確認する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象キャッシュ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createUseCache(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キャッシュ名は不正です" ) ;
        }
        
        return CacheTelegram.createNameByCache(
            out,loginUserOwner,userId,name,
            AdminCacheProtocol.TYPE_IS_CACHE ) ;
        
    }
    
    /**
     * キャッシュ問い合わせ結果を解析.
     * <BR><BR>
     * キャッシュ問い合わせ結果を解析します.
     * <BR>
     * @param out データ受け取り用Beanを設定します.
     * @param telegram 対象の電文を設定します.
     * @exception Exception 例外.
     */
    public static final void analysisTelegram( AdminCacheBean out,byte[] telegram )
        throws Exception {
        
        if( out == null || telegram == null || telegram.length <= 0 ) {
            throw new IllegalArgumentException( "out引数は不正です" ) ;
        }
        
        int[] outId = new int[ 1 ] ;
        byte[] bin = CommonProtocol.analysisProtocol( null,outId,telegram ) ;
        AdminCacheProtocol.analysisProtocol( out,bin ) ;
        
        out.setId( outId[ 0 ] ) ;
        
    }
    
    /**
     * 名前とタイプを設定してキャッシュ電文を生成.
     */
    private static final byte[] createNameByCache(
        int[] out,boolean loginUserOwner,int userId,String name,int type )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminCacheProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                type,                                           // 処理タイプ.
                name,                                           // キャッシュ名.
                -1,                                             // キャッシュサイズなし.
                -1,                                             // 現在キャッシュサイズなし.
                -1,                                             // サイズなし.
                null                                            // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }

}

