/*
 * @(#)ConnectTelegram.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.admin.info.ConnectInfo;
import org.maachang.queue.access.admin.info.CreateConnectInfo;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.admin.AdminConnectBean;
import org.maachang.queue.access.protocol.admin.AdminConnectProtocol;


/**
 * 管理者コネクション用電文.
 *  
 * @version 2007/01/12
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class ConnectTelegram {
    
    /**
     * 新しいコネクションを追加する電文を作成.
     * <BR><BR>
     * 新しいコネクションを追加する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param info 対象コネクション情報を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createAddConnect(
        int[] out,boolean loginUserOwner,int userId,CreateConnectInfo info )
        throws Exception {
        
        if( info == null || info.getName() == null ) {
            throw new IllegalArgumentException( "コネクション生成用条件は不正です" ) ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminConnectProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                AdminConnectProtocol.TYPE_ADD_CONNECT,          // 処理タイプ.
                info.getName(),                                 // コネクション名
                info.getBindAddress(),                          // バインドアドレス.
                info.getBindPort(),                             // バインドポート.
                info.getMqCacheName(),                          // MQキャッシュ名.
                info.getBuffer(),                               // 送受信バッファ長.
                info.isCb32(),                                  // 暗号フラグ.
                -1,                                             // サイズなし.
                info.getCb32Word()                              // 暗号ワード群.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * コネクションを再設定する電文を作成.
     * <BR><BR>
     * コネクションを再設定する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param info 対象コネクション情報を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createSetConnect(
        int[] out,boolean loginUserOwner,int userId,ConnectInfo info )
        throws Exception {
        
        if( info == null || info.getName() == null ) {
            throw new IllegalArgumentException( "コネクション生成用条件は不正です" ) ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminConnectProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                AdminConnectProtocol.TYPE_SET_CONNECT,          // 処理タイプ.
                info.getName(),                                 // コネクション名
                ( String )null,                                 // バインドアドレスなし.
                -1,                                             // バインドポートなし.
                null,                                           // MQキャッシュ名.
                -1,                                             // 送受信バッファ長.
                info.isCb32(),                                  // 暗号フラグ.
                -1,                                             // サイズなし.
                info.getCb32Word()                              // 暗号ワード群.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * コネクションを削除する電文を作成.
     * <BR><BR>
     * コネクションを削除する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象コネクション名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createRemoveConnect(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "コネクション名は不正です" ) ;
        }
        
        return ConnectTelegram.createNameByConnect(
            out,loginUserOwner,userId,name,
            AdminConnectProtocol.TYPE_REMOVE_CONNECT ) ;
        
    }
    
    /**
     * コネクションを取得する電文を作成.
     * <BR><BR>
     * コネクションを取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象コネクション名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetConnect(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "コネクション名は不正です" ) ;
        }
        
        return ConnectTelegram.createNameByConnect(
            out,loginUserOwner,userId,name,
            AdminConnectProtocol.TYPE_GET_CONNECT ) ;
        
    }
    
    /**
     * 全てのコネクション名一覧を取得する電文を作成.
     * <BR><BR>
     * 全てのコネクション名一覧を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetNamesConnect( int[] out,boolean loginUserOwner,int userId )
        throws Exception {
        
        return ConnectTelegram.createNameByConnect(
            out,loginUserOwner,userId,CommonProtocol.DUMMY,
            AdminConnectProtocol.TYPE_GET_CONNECT_NAMES ) ;
        
    }
    
    /**
     * 全てのコネクション数を取得する電文を作成.
     * <BR><BR>
     * 全てのコネクション数を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetSizeConnect( int[] out,boolean loginUserOwner,int userId )
        throws Exception {
        
        return ConnectTelegram.createNameByConnect(
            out,loginUserOwner,userId,CommonProtocol.DUMMY,
            AdminConnectProtocol.TYPE_GET_CONNECT_SIZE ) ;
        
    }
    
    /**
     * コネクションが存在するか確認する電文を作成.
     * <BR><BR>
     * コネクションが存在するか確認する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象コネクション名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createUseConnect(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "コネクション名は不正です" ) ;
        }
        
        return ConnectTelegram.createNameByConnect(
            out,loginUserOwner,userId,name,
            AdminConnectProtocol.TYPE_IS_CONNECT ) ;
        
    }
    
    /**
     * コネクション問い合わせ結果を解析.
     * <BR><BR>
     * コネクション問い合わせ結果を解析します.
     * <BR>
     * @param out データ受け取り用Beanを設定します.
     * @param telegram 対象の電文を設定します.
     * @exception Exception 例外.
     */
    public static final void analysisTelegram( AdminConnectBean out,byte[] telegram )
        throws Exception {
        
        if( out == null || telegram == null || telegram.length <= 0 ) {
            throw new IllegalArgumentException( "out引数は不正です" ) ;
        }
        
        int[] outId = new int[ 1 ] ;
        byte[] bin = CommonProtocol.analysisProtocol( null,outId,telegram ) ;
        AdminConnectProtocol.analysisProtocol( out,bin ) ;
        
        out.setId( outId[ 0 ] ) ;
        
    }
    
    /**
     * 名前とタイプを設定してコネクション電文を生成.
     */
    private static final byte[] createNameByConnect(
        int[] out,boolean loginUserOwner,int userId,String name,int type )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminConnectProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                type,                                           // 処理タイプ.
                name,                                           // コネクション名
                ( String )null,                                 // バインドアドレスなし.
                -1,                                             // バインドポートなし.
                null,                                           // MQキャッシュ名なし.
                -1,                                             // 送受信バッファ長なし.
                false,                                          // 暗号フラグ.
                -1,                                             // サイズなし.
                null                                            // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }

}

