/*
 * @(#)ExecutionChannel.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.ChannelInfo;
import org.maachang.queue.access.admin.info.ChannelKeyInfo;
import org.maachang.queue.access.admin.info.CreateChannelInfo;
import org.maachang.queue.access.admin.info.SendChannelInfo;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.ResultProtocol;
import org.maachang.queue.access.protocol.ResultTelegram;
import org.maachang.queue.access.protocol.admin.AdminChannelBean;
import org.maachang.queue.access.protocol.admin.AdminChannelProtocol;


/**
 * 管理チャネル実行処理.
 *  
 * @version 2007/01/12
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class ExecutionChannel {
    
    /**
     * デフォルトタイムアウト.
     */
    public static final long DEFAULT_TIMEOUT = MaachangQAccessDefine.DEFAULT_TIMEOUT ;
    
    /**
     * チャネル追加処理.
     * <BR><BR>
     * チャネル追加処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param info 対象のチャネル情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void addChannel(
        ConnectObject connect,int userId,boolean userOwner,CreateChannelInfo info )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ChannelTelegram.createAddChannel(
                outId,userOwner,userId,info ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminChannelProtocol.TYPE_ADD_CHANNEL,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 送信チャネル変更処理.
     * <BR><BR>
     * 送信チャネル変更処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param info 対象の送信チャネル情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void setSendChannel(
        ConnectObject connect,int userId,boolean userOwner,SendChannelInfo info )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ChannelTelegram.createSetSendChannel(
                outId,userOwner,userId,info ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminChannelProtocol.TYPE_SET_CHANNEL,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * チャネル削除処理.
     * <BR><BR>
     * チャネル削除処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のチャネル名を設定します.
     * @param type 対象のチャネルタイプを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void removeChannel(
        ConnectObject connect,int userId,boolean userOwner,String name,int type )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ChannelTelegram.createRemoveChannel(
                outId,userOwner,userId,name,type ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminChannelProtocol.TYPE_REMOVE_CHANNEL,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * チャネル停止処理.
     * <BR><BR>
     * チャネル停止処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のチャネル名を設定します.
     * @param type 対象のチャネルタイプを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void stopChannel(
        ConnectObject connect,int userId,boolean userOwner,String name,int type )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ChannelTelegram.createStopChannel(
                outId,userOwner,userId,name,type ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminChannelProtocol.TYPE_STOP_CHANNEL,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * チャネル開始処理.
     * <BR><BR>
     * チャネル開始処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のチャネル名を設定します.
     * @param type 対象のチャネルタイプを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void startChannel(
        ConnectObject connect,int userId,boolean userOwner,String name,int type )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ChannelTelegram.createStartChannel(
                outId,userOwner,userId,name,type ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminChannelProtocol.TYPE_START_CHANNEL,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * チャネル情報取得処理.
     * <BR><BR>
     * チャネル情報取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のチャネル名を設定します.
     * @param type 対象のチャネルタイプを設定します.
     * @return ChannelInfo チャネル情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final ChannelInfo getChannel(
        ConnectObject connect,int userId,boolean userOwner,String name,int type )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ChannelTelegram.createGetChannel(
                outId,userOwner,userId,name,type ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminChannelProtocol.TYPE_GET_CHANNEL,
                    telegram ) ;
                
            }
            // 受信電文がチャネル情報の場合.
            else if( result == AdminChannelProtocol.CATEGORY_TYPE_ADMIN_CHANNEL ) {
                
                // チャネル情報を解析.
                AdminChannelBean bean = new AdminChannelBean() ;
                ChannelTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                if( bean.getName() == null || bean.getName().length() <= 0 ) {
                    throw new MaachangQException(
                        "対象名(" + name + ")のチャネル名は存在しません",
                        MaachangQErrorCode.ERROR_NOT_CHANNEL ) ;
                }
                
                ChannelInfo ret = null ;
                
                // 送信チャネルの場合.
                if( bean.getChannelType() == MaachangQAccessDefine.TYPE_SEND ) {
                    
                    SendChannelInfo send = new SendChannelInfo() ;
                    send.setQueueManager( bean.getQueueManager() ) ;
                    send.setAddress( bean.getInetAddress() ) ;
                    send.setPort( bean.getPort() ) ;
                    send.setConnectName( bean.getConnectName() ) ;
                    send.setUseCb32Word( bean.getUseCb32Word() ) ;
                    
                    ret = send ;
                    
                }
                // 受信チャネルの場合.
                else {
                    ret = new ChannelInfo() ;
                }
                
                ret.setChannelName( bean.getName() ) ;
                ret.setState( bean.getState() ) ;
                
                return ret ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * チャネル一覧取得処理.
     * <BR><BR>
     * チャネル一覧取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @return ChannelKeyInfo[] チャネル一覧が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final ChannelKeyInfo[] getKeys(
        ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ChannelTelegram.createGetNamesChannel(
                outId,userOwner,userId ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminChannelProtocol.TYPE_GET_CHANNEL_NAMES,
                    telegram ) ;
                
            }
            // 受信情報が、チャネル情報の場合.
            else if( result == AdminChannelProtocol.CATEGORY_TYPE_ADMIN_CHANNEL ) {
                
                // キュー情報を解析.
                AdminChannelBean bean = new AdminChannelBean() ;
                ChannelTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                int len = 0 ;
                if( bean.getParams() == null || bean.getNumbers() == null ||
                    bean.getParams().length != bean.getNumbers().length ) {
                    return null ;
                }
                
                len = bean.getParams().length ;
                
                if( len > 0 ) {
                    ChannelKeyInfo[] ret = new ChannelKeyInfo[ len ] ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        
                        ret[ i ] = new ChannelKeyInfo() ;
                        ret[ i ].setChannel( bean.getParams()[ i ] ) ;
                        ret[ i ].setType( bean.getNumbers()[ i ] ) ;
                        
                    }
                    
                    return ret ;
                }
                
                return null ;
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * チャネル数取得処理.
     * <BR><BR>
     * チャネル数取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @return int チャネル数が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final int getSize(
        ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ChannelTelegram.createGetSizeChannel(
                outId,userOwner,userId ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminChannelProtocol.TYPE_GET_CHANNEL_SIZE,
                    telegram ) ;
                
            }
            // 受信情報が、チャネル情報の場合.
            else if( result == AdminChannelProtocol.CATEGORY_TYPE_ADMIN_CHANNEL ) {
                
                // キュー情報を解析.
                AdminChannelBean bean = new AdminChannelBean() ;
                ChannelTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                return bean.getSize() ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * チャネル存在チェック処理.
     * <BR><BR>
     * チャネル存在チェック処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のチャネル名を設定します.
     * @param type 対象のチャネルタイプを設定します.
     * @return boolean チャネル存在結果が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final boolean isChannel(
        ConnectObject connect,int userId,boolean userOwner,String name,int type )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ChannelTelegram.createUseChannel(
                outId,userOwner,userId,name,type ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                try {
                    // 処理戻り条件をチェック.
                    ResultTelegram.checkResult(
                        outId[ 0 ],AdminChannelProtocol.TYPE_IS_CHANNEL,
                        telegram ) ;
                    
                    return true ;
                } catch( Exception e ) {
                    return false ;
                }
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
}

