/*
 * @(#)ExecutionLogin.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.ResultProtocol;
import org.maachang.queue.access.protocol.ResultTelegram;
import org.maachang.queue.access.protocol.login.LoginBean;
import org.maachang.queue.access.protocol.login.LoginProtocol;


/**
 * ログイン実行処理.
 *  
 * @version 2007/01/12
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class ExecutionLogin {
    
    /**
     * デフォルトタイムアウト.
     */
    public static final long DEFAULT_TIMEOUT = MaachangQAccessDefine.DEFAULT_TIMEOUT ;
    
    /**
     * ログイン処理.
     * <BR><BR>
     * ログイン処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @return LoginBean ログイン結果が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final LoginBean login( ConnectObject connect,String user,String passwd )
        throws MaachangQException {
        return ExecutionLogin.loginOrShutdown( connect,user,passwd,false ) ;
    }
    
    /**
     * シャットダウン処理.
     * <BR><BR>
     * シャットダウン処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void shutdown( ConnectObject connect,String user,String passwd )
        throws MaachangQException {
        ExecutionLogin.loginOrShutdown( connect,user,passwd,true ) ;
    }
    
    /**
     * ログイン or シャットダウン処理.
     * <BR><BR>
     * ログイン or シャットダウン処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @param shutdown シャットダウンモードを設定します.<BR>
     *                 [true]を設定した場合、接続先のMaachangQはシャットダウンします.
     * @return LoginBean ログイン結果が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    protected static final LoginBean loginOrShutdown( ConnectObject connect,String user,String passwd,boolean shutdown )
        throws MaachangQException {
        
        LoginBean ret = null ;
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = LoginTelegram.createLoginTelegram( outId,user,passwd,shutdown ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],LoginProtocol.CATEGORY_TYPE_LOGIN,telegram ) ;
                
            }
            // 受信電文が、ログイン結果の場合.
            else if( result == LoginProtocol.CATEGORY_TYPE_RESULT_LOGIN ) {
                
                ret = new LoginBean() ;
                LoginTelegram.analysisTelegram( ret,telegram ) ;
                
                if( ret.getTelegramId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
            }
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * ログアウト処理.
     * <BR><BR>
     * ログアウト処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザオーナを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    protected static final void logout( ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = LoginTelegram.createLogoutTelegram( outId,userId,userOwner ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],LoginProtocol.CATEGORY_TYPE_RESULT_LOGIN,telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
}

