/*
 * @(#)ExecutionUser.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import java.util.Arrays;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.CreateUserInfo;
import org.maachang.queue.access.admin.info.UserInfo;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.ResultProtocol;
import org.maachang.queue.access.protocol.ResultTelegram;
import org.maachang.queue.access.protocol.admin.AdminUserBean;
import org.maachang.queue.access.protocol.admin.AdminUserProtocol;


/**
 * 管理者ユーザ実行処理.
 *  
 * @version 2007/01/12
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class ExecutionUser {
    
    /**
     * デフォルトタイムアウト.
     */
    public static final long DEFAULT_TIMEOUT = MaachangQAccessDefine.DEFAULT_TIMEOUT ;
    
    /**
     * ユーザ追加処理.
     * <BR><BR>
     * ユーザ追加処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param info 対象の追加ユーザ情報を設定します.
     * @param passwd 対象の追加ユーザパスワードを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void addUser(
        ConnectObject connect,int userId,boolean userOwner,CreateUserInfo info,String passwd )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = UserTelegram.createAddUser(
                outId,userOwner,userId,info,passwd ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminUserProtocol.TYPE_ADD_USER,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * ユーザ削除処理.
     * <BR><BR>
     * ユーザ削除処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param user 対象のユーザ名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void removeUser(
        ConnectObject connect,int userId,boolean userOwner,String user )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = UserTelegram.createRemoveUser(
                outId,userOwner,userId,user ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminUserProtocol.TYPE_REMOVE_USER,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 指定ユーザパスワード変更処理.
     * <BR><BR>
     * 指定ユーザパスワード変更処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param user 対象のユーザ名を設定します.
     * @param passwd 新しく設定するパスワードを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void changePasswd(
        ConnectObject connect,int userId,boolean userOwner,String user,String passwd )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = UserTelegram.createChangePasswd(
                outId,userOwner,userId,user,passwd ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminUserProtocol.TYPE_SET_USER_CHANGE_PASSWD,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 現在ログインユーザパスワード変更処理.
     * <BR><BR>
     * 現在ログインユーザパスワード変更処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param oldPasswd 現在のパスワードを設定します.
     * @param newPasswd 新しいパスワードを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void thisChangePasswd(
        ConnectObject connect,int userId,boolean userOwner,String oldPasswd,String newPasswd )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = UserTelegram.createThisChangePasswd(
                outId,userOwner,userId,newPasswd,oldPasswd ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminUserProtocol.TYPE_THIS_USER_CHANGE_PASSWD,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 指定ユーザの権限変更処理.
     * <BR><BR>
     * 指定ユーザの権限変更処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param user 対象のユーザ名を設定します.
     * @param owner 新しく設定する権限を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void changeOwner(
        ConnectObject connect,int userId,boolean userOwner,String user,boolean owner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = UserTelegram.createChangeOwner(
                outId,userOwner,userId,user,owner ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminUserProtocol.TYPE_CHANGE_OWNER,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    
    /**
     * 全ユーザ一覧取得処理.
     * <BR><BR>
     * 指定ユーザの権限変更処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @return UserInfo[] ユーザ情報一覧が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final UserInfo[] getUsers( ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = UserTelegram.createGetUsers(
                outId,userOwner,userId ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminUserProtocol.TYPE_GET_USERS,
                    telegram ) ;
                
            }
            // 受信電文が、ユーザ情報の場合.
            else if( result == AdminUserProtocol.CATEGORY_TYPE_ADMIN_USER ) {
                
                // ユーザ情報を解析.
                AdminUserBean bean = new AdminUserBean() ;
                UserTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                // 取得内容が不正.
                else if( bean.getParams() == null || bean.getFlags() == null ||
                    bean.getParams().length != bean.getFlags().length ) {
                    throw new MaachangQException( "ユーザ情報は存在しないか不正です",
                        MaachangQErrorCode.ERROR_USER ) ;
                }
                
                // データ長を取得.
                int len = bean.getParams().length ;
                if( len > 0 ) {
                    
                    // 戻りデータを生成.
                    UserInfo[] ret = new UserInfo[ len ] ;
                    
                    for( int i = 0 ; i < len ; i ++ ) {
                        UserInfo info = new UserInfo() ;
                        info.setUser( bean.getParams()[ i ] ) ;
                        info.setRootOwner( bean.getFlags()[ i ] ) ;
                        ret[ i ] = info ;
                    }
                    
                    // データソート.
                    Arrays.sort( ret ) ;
                    
                    return ret ;
                }
                
                return null ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            e.printStackTrace() ;
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 対象ユーザ数取得処理.
     * <BR><BR>
     * 対象ユーザ数取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param int ユーザ数が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final int getUserSize( ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = UserTelegram.createGetUserSize(
                outId,userOwner,userId ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminUserProtocol.TYPE_USER_SIZE,
                    telegram ) ;
                
            }
            // 受信電文が、ユーザ情報の場合.
            else if( result == AdminUserProtocol.CATEGORY_TYPE_ADMIN_USER ) {
                
                // ユーザ情報を解析.
                AdminUserBean bean = new AdminUserBean() ;
                UserTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                // ユーザ数を返す.
                return bean.getSize() ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 指定ユーザの権限取得処理.
     * <BR><BR>
     * 指定ユーザの権限取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param user 対象のユーザ名を設定します.
     * @return boolean 権限情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final boolean getOwner(
        ConnectObject connect,int userId,boolean userOwner,String user )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = UserTelegram.createGetUserOwner(
                outId,userOwner,userId,user ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminUserProtocol.TYPE_USER_BY_ROOT_OWNER,
                    telegram ) ;
                
            }
            // 受信電文が、ユーザ情報の場合.
            else if( result == AdminUserProtocol.CATEGORY_TYPE_ADMIN_USER ) {
                
                // ユーザ情報を解析.
                AdminUserBean bean = new AdminUserBean() ;
                UserTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                // 内容が不正な場合.
                if( bean.getFlags() == null || bean.getFlags().length != 1 ) {
                    throw new MaachangQException( "ユーザ情報は存在しないか不正です",
                        MaachangQErrorCode.ERROR_USER ) ;
                }
                
                return bean.getFlags()[ 0 ] ;
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 対象ユーザ存在チェック処理.
     * <BR><BR>
     * 指定ユーザの権限取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param user 対象のユーザ名を設定します.
     * @return boolean ユーザ存在結果が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final boolean isUser(
        ConnectObject connect,int userId,boolean userOwner,String user )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = UserTelegram.createCheckUser(
                outId,userOwner,userId,user ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                try {
                    // 処理戻り条件をチェック.
                    ResultTelegram.checkResult(
                        outId[ 0 ],AdminUserProtocol.TYPE_IS_USER,
                        telegram ) ;
                    
                    return true ;
                } catch( Exception e ) {
                    return false ;
                }
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
}

