/*
 * @(#)QueueManagerTelegram.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.admin.info.CreateQueueManagerInfo;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.admin.AdminQueueManagerBean;
import org.maachang.queue.access.protocol.admin.AdminQueueManagerProtocol;


/**
 * 管理者キューマネージャ用電文.
 *  
 * @version 2007/01/11
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class QueueManagerTelegram {
    
    /**
     * 新しいキューマネージャを追加する電文を作成.
     * <BR><BR>
     * 新しいキューマネージャを追加する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param info 対象キューマネージャ情報を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createAddQueueManager(
        int[] out,boolean loginUserOwner,int userId,CreateQueueManagerInfo info )
        throws Exception {
        
        if( info == null || info.getQueueManager() == null ) {
            throw new IllegalArgumentException( "キューマネージャ生成用条件は不正です" ) ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminQueueManagerProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                AdminQueueManagerProtocol.TYPE_ADD_QMANAGER,    // 処理タイプ.
                info.getQueueManager(),                         // キューマネージャ名
                info.getMaxCacheSize(),                         // 最大キャッシュサイズ
                -1,                                             // 現在キャッシュなし.
                -1,                                             // ステータスなし.
                -1,                                             // サイズなし.
                null                                            // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * キューマネージャを削除する電文を作成.
     * <BR><BR>
     * キューマネージャを削除する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createRemoveQueueManager(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キューマネージャ名は不正です" ) ;
        }
        
        return QueueManagerTelegram.createNameByQueueManager(
            out,loginUserOwner,userId,name,
            AdminQueueManagerProtocol.TYPE_REMOVE_QMANAGER ) ;
        
    }
    
    /**
     * キューマネージャ停止する電文を作成.
     * <BR><BR>
     * キューマネージャを停止する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createStopQueueManager(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キューマネージャ名は不正です" ) ;
        }
        
        return QueueManagerTelegram.createNameByQueueManager(
            out,loginUserOwner,userId,name,
            AdminQueueManagerProtocol.TYPE_STOP_QMANAGER ) ;
        
    }
    
    /**
     * キューマネージャ開始する電文を作成.
     * <BR><BR>
     * キューマネージャを開始する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createStartQueueManager(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キューマネージャ名は不正です" ) ;
        }
        
        return QueueManagerTelegram.createNameByQueueManager(
            out,loginUserOwner,userId,name,
            AdminQueueManagerProtocol.TYPE_START_QMANAGER ) ;
        
    }
    
    /**
     * キューマネージャ情報を取得する電文を作成.
     * <BR><BR>
     * キューマネージャ情報を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetQueueManager(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キューマネージャ名は不正です" ) ;
        }
        
        return QueueManagerTelegram.createNameByQueueManager(
            out,loginUserOwner,userId,name,
            AdminQueueManagerProtocol.TYPE_GET_QMANAGER ) ;
        
    }
    
    /**
     * 存在するキューマネージャ名一覧を取得する電文を作成.
     * <BR><BR>
     * 存在するキューマネージャ名一覧を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetNamesQueueManager(
        int[] out,boolean loginUserOwner,int userId )
        throws Exception {
        
        return QueueManagerTelegram.createNameByQueueManager(
            out,loginUserOwner,userId,CommonProtocol.DUMMY,
            AdminQueueManagerProtocol.TYPE_GET_QMANAGER_NAMES ) ;
        
    }
    
    /**
     * 存在するキューマネージャ数を取得する電文を作成.
     * <BR><BR>
     * 存在するキューマネージャ数を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetSizeQueueManager(
        int[] out,boolean loginUserOwner,int userId )
        throws Exception {
        
        return QueueManagerTelegram.createNameByQueueManager(
            out,loginUserOwner,userId,CommonProtocol.DUMMY,
            AdminQueueManagerProtocol.TYPE_GET_QMANAGER_SIZE ) ;
        
    }
    
    /**
     * キューマネージャ情報存在確認用電文を作成.
     * <BR><BR>
     * キューマネージャ情報存在確認用電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createUseQueueManager(
        int[] out,boolean loginUserOwner,int userId,String name )
        throws Exception {
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キューマネージャ名は不正です" ) ;
        }
        
        return QueueManagerTelegram.createNameByQueueManager(
            out,loginUserOwner,userId,name,
            AdminQueueManagerProtocol.TYPE_IS_QMANAGER ) ;
        
    }
    
    /**
     * キューマネージャ問い合わせ結果を解析.
     * <BR><BR>
     * キューマネージャ問い合わせ結果を解析します.
     * <BR>
     * @param out データ受け取り用Beanを設定します.
     * @param telegram 対象の電文を設定します.
     * @exception Exception 例外.
     */
    public static final void analysisTelegram( AdminQueueManagerBean out,byte[] telegram )
        throws Exception {
        
        if( out == null || telegram == null || telegram.length <= 0 ) {
            throw new IllegalArgumentException( "out引数は不正です" ) ;
        }
        
        int[] outId = new int[ 1 ] ;
        byte[] bin = CommonProtocol.analysisProtocol( null,outId,telegram ) ;
        AdminQueueManagerProtocol.analysisProtocol( out,bin ) ;
        
        out.setId( outId[ 0 ] ) ;
        
    }
    
    /**
     * 名前とタイプを設定してキューマネージャ電文を生成.
     */
    private static final byte[] createNameByQueueManager(
        int[] out,boolean loginUserOwner,int userId,String name,int type )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminQueueManagerProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                type,                                           // 処理タイプ.
                name,                                           // キューマネージャ名.
                -1,                                             // 最大キャッシュなし.
                -1,                                             // 現在キャッシュなし.
                -1,                                             // ステータスなし.
                -1,                                             // サイズなし.
                null                                            // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }

}

