/*
 * @(#)UserTelegram.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.CreateUserInfo;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.admin.AdminUserBean;
import org.maachang.queue.access.protocol.admin.AdminUserProtocol;


/**
 * 管理者ユーザ用電文.
 *  
 * @version 2007/01/11
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class UserTelegram {
    
    /**
     * 新しいユーザを追加する電文を作成.
     * <BR><BR>
     * 新しいユーザを追加する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param info ユーザ情報を設定します.
     * @param passwd パスワードを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createAddUser(
        int[] out,boolean loginUserOwner,int userId,CreateUserInfo info,String passwd )
        throws Exception {
        
        if( info == null ) {
            throw new IllegalArgumentException( "ユーザ生成用条件は不正です" ) ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminUserProtocol.createProtocol(
                out[ 0 ],                             // 電文ID.
                userId,                               // ログインユーザID.
                loginUserOwner,                       // ユーザオーナー.
                AdminUserProtocol.TYPE_ADD_USER,      // 処理タイプ.
                info.getUser(),                       // 生成ユーザ名.
                new boolean[] { info.isRootOwner() }, // フラグなし.
                -1,                                   // サイズなし.
                new String[] { passwd }               // 生成ユーザパスワード.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * ユーザを削除する電文を作成.
     * <BR><BR>
     * ユーザを削除する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param user 削除対象のユーザを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createRemoveUser(
        int[] out,boolean loginUserOwner,int userId,String user )
        throws Exception {
        
        if( user == null || ( user = user.trim() ).length() <= 0 ||
            MaachangQAccessDefine.ROOT_USER.equals( user ) == true ) {
            if( user == null || user.length() <= 0 ) {
                throw new IllegalArgumentException( "指定ユーザは不正です" ) ;
            }
            else {
                throw new MaachangQException(
                    "対象ユーザはスーパーユーザなので、削除できません",
                    MaachangQErrorCode.ERROR_NOT_DELETE_USER ) ;
            }
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminUserProtocol.createProtocol(
                out[ 0 ],                           // 電文ID.
                userId,                             // ログインユーザID.
                loginUserOwner,                     // ユーザオーナー.
                AdminUserProtocol.TYPE_REMOVE_USER, // 処理タイプ.
                user,                               // 削除ユーザ名.
                null,                               // フラグなし.
                -1,                                 // サイズなし.
                null                                // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 対象ユーザのパスワードを変更する電文を作成.
     * <BR><BR>
     * 対象ユーザのパスワードを変更する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param user 対象ユーザ名を設定します.
     * @param passwd 変更パスワードを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createChangePasswd(
        int[] out,boolean loginUserOwner,int userId,String user,String passwd )
        throws Exception {
        
        if( user == null || ( user = user.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "ユーザ生成用条件は不正です" ) ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminUserProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // ユーザオーナー.
                AdminUserProtocol.TYPE_SET_USER_CHANGE_PASSWD,  // 処理タイプ.
                user,                                           // 変更ユーザ名.
                null,                                           // フラグなし.
                -1,                                             // サイズなし.
                new String[] { passwd }                         // 変更ユーザパスワード.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 現在ログインユーザのパスワードを変更する電文を作成.
     * <BR><BR>
     * 現在ログインユーザのパスワードを変更する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param passwd 変更パスワードを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createThisChangePasswd(
        int[] out,boolean loginUserOwner,int userId,String passwd,String oldPasswd )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminUserProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // ユーザオーナー.
                AdminUserProtocol.TYPE_THIS_USER_CHANGE_PASSWD, // 処理タイプ.
                CommonProtocol.DUMMY,                           // ユーザ名なし.
                null,                                           // フラグなし.
                -1,                                             // サイズなし.
                new String[] { passwd,oldPasswd }               // 変更パスワード,前回パスワード.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 対象ユーザのルート権限を変更する電文を作成.
     * <BR><BR>
     * 対象ユーザのルート権限を変更する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param user 対象ユーザ名を設定します.
     * @param newOwner 変更ルートオーナを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createChangeOwner(
        int[] out,boolean loginUserOwner,int userId,String user,boolean newOwner )
        throws Exception {
        
        if( user == null || ( user = user.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "ユーザ生成用条件は不正です" ) ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminUserProtocol.createProtocol(
                out[ 0 ],                               // 電文ID.
                userId,                                 // ログインユーザID.
                loginUserOwner,                         // ユーザオーナー.
                AdminUserProtocol.TYPE_CHANGE_OWNER,    // 処理タイプ.
                user,                                   // 変更ユーザ名.
                new boolean[] { newOwner },             // フラグなし.
                -1,                                     // サイズなし.
                null                                    // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 全ユーザ一覧を取得する電文を作成.
     * <BR><BR>
     * 全ユーザ一覧を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetUsers( int[] out,boolean loginUserOwner,int userId )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminUserProtocol.createProtocol(
                out[ 0 ],                               // 電文ID.
                userId,                                 // ログインユーザID.
                loginUserOwner,                         // ユーザオーナー.
                AdminUserProtocol.TYPE_GET_USERS,       // 処理タイプ.
                CommonProtocol.DUMMY,                   // ユーザ名なし.
                null,                                   // フラグなし.
                -1,                                     // サイズなし.
                null                                    // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 全ユーザ数を取得する電文を作成.
     * <BR><BR>
     * 全ユーザ数を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetUserSize( int[] out,boolean loginUserOwner,int userId )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminUserProtocol.createProtocol(
                out[ 0 ],                               // 電文ID.
                userId,                                 // ログインユーザID.
                loginUserOwner,                         // ユーザオーナー.
                AdminUserProtocol.TYPE_USER_SIZE,       // 処理タイプ.
                CommonProtocol.DUMMY,                   // ユーザ名なし.
                null,                                   // フラグなし.
                -1,                                     // サイズなし.
                null                                    // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 対象ユーザのルートオーナを取得する電文を作成.
     * <BR><BR>
     * 対象ユーザのルートオーナを取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param user 対象ユーザ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetUserOwner(
        int[] out,boolean loginUserOwner,int userId,String user )
        throws Exception {
        
        if( user == null || ( user = user.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "ユーザ生成用条件は不正です" ) ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminUserProtocol.createProtocol(
                out[ 0 ],                                   // 電文ID.
                userId,                                     // ログインユーザID.
                loginUserOwner,                             // ユーザオーナー.
                AdminUserProtocol.TYPE_USER_BY_ROOT_OWNER,  // 処理タイプ.
                user,                                       // 取得対象ユーザ名.
                null,                                       // フラグなし.
                -1,                                         // サイズなし.
                null                                        // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 対象ユーザ存在確認の電文を作成.
     * <BR><BR>
     * 対象ユーザ存在確認の電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param user 対象ユーザ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createCheckUser(
        int[] out,boolean loginUserOwner,int userId,String user )
        throws Exception {
        
        if( user == null || ( user = user.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "ユーザ生成用条件は不正です" ) ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminUserProtocol.createProtocol(
                out[ 0 ],                       // 電文ID.
                userId,                         // ログインユーザID.
                loginUserOwner,                 // ユーザオーナー.
                AdminUserProtocol.TYPE_IS_USER, // 処理タイプ.
                user,                           // 取得対象ユーザ名.
                null,                           // フラグなし.
                -1,                             // サイズなし.
                null                            // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * ユーザ問い合わせ結果を解析.
     * <BR><BR>
     * ユーザ問い合わせ結果を解析します.
     * <BR>
     * @param out データ受け取り用Beanを設定します.
     * @param telegram 対象の電文を設定します.
     * @exception Exception 例外.
     */
    public static final void analysisTelegram( AdminUserBean out,byte[] telegram )
        throws Exception {
        
        if( out == null || telegram == null || telegram.length <= 0 ) {
            throw new IllegalArgumentException( "out引数は不正です" ) ;
        }
        
        int[] outId = new int[ 1 ] ;
        byte[] bin = CommonProtocol.analysisProtocol( null,outId,telegram ) ;
        AdminUserProtocol.analysisProtocol( out,bin ) ;
        
        out.setId( outId[ 0 ] ) ;
        
    }
    
}

