/*
 * @(#)ConnectObject.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.net ;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.InetAddress;
import java.net.Socket;

import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.ByteUtil;


/**
 * TCP/IP通信用オブジェクト.
 *  
 * @version 2007/01/08
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class ConnectObject {
    
    /**
     * バッファサイズ.
     */
    public static final int BUFFER_LENGTH = 512 ;
    
    /**
     * 受信タイムアウト更新カウント.
     */
    public static final int NEW_RECEIVE_TIMEOUT_COUNT = 64 ;
    
    /**
     * ソケット受信タイムアウト.
     */
    public static final int RECV_TIMEOUT = 30 ;
    
    /**
     * 基本バッファサイズ.
     */
    public static final int BASE_BUFFER = 262144 ;
    
    /**
     * ソケットオブジェクト.
     */
    private Socket socket = null ;
    
    /**
     * 送信バッファ.
     */
    private BufferedOutputStream sendBuffer = null ;
    
    /**
     * 受信バッファ.
     */
    private BufferedInputStream receiveBuffer = null ;
    
    /**
     * 受信バッファ.
     */
    private ByteUtil byteBuffer = null ;
    
    /**
     * １秒あたりの受信データ長.
     */
    private long secondToReceiveByte = 0L ;
    
    /**
     * 受信開始中フラグ.
     */
    private boolean executionReceive = false ;
    
    /**
     * 内部同期.
     */
    private final Object privateSync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private ConnectObject() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ソケットオブジェクトを設定して、オブジェクトを
     * 生成します.
     * <BR>
     * @param socket 対象のソケットオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public ConnectObject( Socket socket )
        throws Exception {
        
        if( socket == null || socket.isClosed() == true ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        socket.setSendBufferSize( BASE_BUFFER ) ;
        socket.setReceiveBufferSize( BASE_BUFFER ) ;
        socket.setSoTimeout( RECV_TIMEOUT ) ;
        socket.setSoLinger( true,1 ) ;
        socket.setKeepAlive( true ) ;
        socket.setTcpNoDelay( true ) ;
        socket.setReuseAddress( true ) ;
        
        this.socket = socket ;
        this.sendBuffer = new BufferedOutputStream(
            socket.getOutputStream() ) ;
        this.receiveBuffer = new BufferedInputStream(
            socket.getInputStream() ) ;
        this.byteBuffer = new ByteUtil( BUFFER_LENGTH ) ;
        this.secondToReceiveByte = 0L ;
        this.executionReceive = false ;
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクトを破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public void destroy() {
        if( socket != null ) {
            try {
                sendBuffer.close() ;
            } catch( Exception e ) {
            }
            try {
                receiveBuffer.close() ;
            } catch( Exception e ) {
            }
            try {
                socket.close() ;
            } catch( Exception e ) {
            }
            try {
                byteBuffer.clear() ;
            } catch( Exception e ) {
            }
        }
        socket = null ;
        sendBuffer = null ;
        receiveBuffer = null ;
        byteBuffer = null ;
        secondToReceiveByte = 0L ;
        executionReceive = false ;
    }
    
    /**
     * 電文送信処理.
     * <BR><BR>
     * 電文を送信します.
     * <BR>
     * @param binary 送信対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public void send( byte[] binary )
        throws Exception {
        
        if( this.isClose() == true ) {
            return ;
        }
        
        if( binary == null || binary.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        int len = binary.length ;
        int bufferLen = BUFFER_LENGTH ;
        
        // 指定長毎にデータ送信.
        for( int i = 0 ;; i += BUFFER_LENGTH ) {
            
            if( i + BUFFER_LENGTH >= len ) {
                if( i >= len ) {
                    break ;
                }
                bufferLen = len - i ;
            }
            
            sendBuffer.write( binary,i,bufferLen ) ;
            
        }
        
        sendBuffer.flush() ;
        
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.<BR>
     * このメソッドは受信情報が存在しない場合は、タイムアウト値に
     * 関わらず処理されません.
     * <BR>
     * @return byte[] 対象のバイナリが返されます.
     * @param timeout 対象のタイムアウト値を設定します.<BR>
     *                [0]を設定した場合は、無限に待ちます.
     * @exception Exception 例外.
     */
    public synchronized byte[] receive( long timeout )
        throws Exception {
        
        int buf ;
        
        if( this.isClose() == true ) {
            return null ;
        }
        
        // 受信開始.
        try {
            
            // 受信処理開始.
            this.startReceive() ;
            
            // 最初の情報を受信.
            for( ;; ) {
                
                // 最初の受信情報待ち.
                // ここでは、引数の受信タイムアウト値は、
                // 無視して、処理.
                try {
                    if( ( buf = receiveBuffer.read() ) <= -1 ) {
                        return null ;
                    }
                } catch( InterruptedIOException ii ) {
                    try {
                        Thread.sleep( RECV_TIMEOUT ) ;
                    } catch( Exception e ) {
                    }
                    // 最初の情報受信時に情報が存在しない場合は、
                    // 受信処理を停止.
                    return null ;
                } catch( Exception e ) {
                    throw e ;
                }
                
                // 先頭の情報を受信した可能性がある場合.
                if( ( byte )( buf & 0x000000ff ) == CommonProtocol.HEADER[ 0 ] ) {
                    // 次の処理に遷移.
                    break ;
                }
                
            }
            
            // 受信情報が存在する場合.
            // バッファをクリアして、データ長にしたがって、受信処理.
            byteBuffer.add( ( byte )( buf & 0x000000ff ) ) ;
            long receiveTime = System.currentTimeMillis() ;
            
            // タイムアウト値を取得.
            long tmOut = ( timeout <= 0 ) ?
                Long.MAX_VALUE :
                System.currentTimeMillis() + timeout ;
            
            int maxLength = -1 ;
            
            // 受信処理開始.
            for( int i = 1 ;; ) {
                try {
                    
                    for( ;; i ++ ) {
                        
                        // 受信データがEOFの場合.
                        if( ( buf = receiveBuffer.read() ) <= -1 ) {
                            // バイナリ転送であるため、処理不正とする.
                            return null ;
                        }
                        
                        // 受信データをセット.
                        byteBuffer.add( ( byte )( buf & 0x000000ff ) ) ;
                        
                        // 受信対象電文長がまだ取得できてなく、
                        // セットされている受信データ長が、
                        // 受信対象電文長を含む領域の場合.
                        if( maxLength == -1 &&
                            byteBuffer.size() >= CommonProtocol.TELEGRAM_LENGTH_OFFSET + CommonProtocol.ALL_DATA_LENGTH ) {
                            
                            // 受信対象電文長を取得.
                            maxLength = CommonProtocol.getProtocolLength( byteBuffer ) ;
                            
                            // 受信対象電文長が不正な場合.
                            if( maxLength <= -1 ) {
                                return null ;
                            }
                            
                        }
                        // 受信対象電文長に満たした場合.
                        else if( maxLength != -1 && maxLength <= byteBuffer.size() ) {
                            
                            // 受信電文を取得して、その内容を返す.
                            byte[] ret = byteBuffer.get( 0,maxLength ) ;
                            
                            // 処理開始から終了までの時間を取得.
                            long startEndTime = System.currentTimeMillis() - receiveTime ;
                            if( startEndTime > 0L ) {
                                
                                // １秒当たりの受信バイト長を計算.
                                long srb = ( long )(
                                    ( ( double )maxLength / ( double )startEndTime ) * 1000.0f ) ;
                                
                                synchronized( privateSync ) {
                                    if( secondToReceiveByte <= 0L ) {
                                        secondToReceiveByte = srb ;
                                    }
                                    else {
                                        secondToReceiveByte = ( secondToReceiveByte + srb ) / 2L ;
                                    }
                                }
                                
                            }
                            
                            // 正常受信.
                            return ret ;
                            
                        }
                        
                        // 規定受信量に対するタイムアウト更新.
                        if( i >= NEW_RECEIVE_TIMEOUT_COUNT ) {
                            if( tmOut != Long.MAX_VALUE ) {
                                tmOut = System.currentTimeMillis() + timeout ;
                            }
                            i = 0 ;
                        }
                        
                    }
                    
                } catch( InterruptedIOException ii ) {
                    
                    // タイムアウト値が規定値の場合.
                    if( System.currentTimeMillis() >= tmOut ) {
                        // 受信タイムアウト例外を返す.
                        throw new ReceiveTimeoutException( ii ) ;
                    }
                    
                    i = 0 ;
                    
                } catch( Exception e ) {
                    throw e ;
                }
                
            }
            
        } finally {
            // 受信終了処理.
            this.endReceive() ;
            // 受信バッファをクリア.
            try {
                byteBuffer.clear() ;
            } catch( Exception e ) {
            }
        }
        
    }
    
    /**
     * 受信処理待ち.
     * <BR><BR>
     * 受信処理を待ちます.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param time 対象のタイムアウトを設定します.
     * @return byte[] 受信バイナリが返されます.
     * @exception Exception 例外.
     */
    public static final byte[] receive( ConnectObject connect,long time )
        throws Exception {
        
        long timeout = ( time <= 0 ) ?
            Long.MAX_VALUE : System.currentTimeMillis() + time ;
        
        byte[] ret = null ;
        
        for( ;; ) {
            
            if( connect.isClose() == true ) {
                throw new IOException( "コネクションは切断されています" ) ;
            }
            
            if( ( ret = connect.receive( time ) ) == null ) {
                if( System.currentTimeMillis() >= timeout ) {
                    throw new ReceiveTimeoutException( "タイムアウト例外が発生" ) ;
                }
            }
            else {
                return ret ;
            }
            
        }
        
    }
    
    /**
     * 受信に対する１秒当たりの通信速度を取得.
     * <BR><BR>
     * 受信に対する１秒当たりの通信速度を取得します.
     * <BR>
     * @return long 受信に対する１秒当たりの通信速度が返されます.
     */
    public long getSecondByReceiveByte() {
        long ret = 0L ;
        synchronized( privateSync ) {
            ret = secondToReceiveByte ;
        }
        return ret ;
    }
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインドアドレスを取得します.
     * <BR>
     * @return InetAddress バインドアドレスが返されます.
     * @exception Exception 例外.
     */
    public InetAddress getBindAddress() {
        return this.socket.getLocalAddress() ;
    }
    
    /**
     * バインドポートを取得.
     * <BR><BR>
     * バインドポートを取得します.
     * <BR>
     * @return int バインドポートが返されます.
     * @exception Exception 例外.
     */
    public int getBindPort() {
        return this.socket.getLocalPort() ;
    }
    
    /**
     * 接続先アドレスを取得.
     * <BR><BR>
     * 接続先アドレスを取得します.
     * <BR>
     * @return InetAddress 接続先アドレスが返されます.
     * @exception Exception 例外.
     */
    public InetAddress getRemoteAddress()
        throws Exception {
        return this.socket.getInetAddress() ;
    }
    
    /**
     * 接続先ポートを取得.
     * <BR><BR>
     * 接続先ポートを取得します.
     * <BR>
     * @return int 接続先ポートが返されます.
     * @exception Exception 例外.
     */
    public int getRemotePort() {
        return this.socket.getPort() ;
    }
    
    /**
     * ソケットオブジェクトを取得.
     * <BR><BR>
     * ソケットオブジェクトを取得します.
     * <BR>
     * @return Socket ソケットオブジェクトが返されます.
     */
    public Socket getSocket() {
        return this.socket ;
    }
    
    /**
     * このオブジェクトが受信処理中であるかチェック.
     * <BR><BR>
     * このオブジェクトが受信処理中であるかチェックします.
     * <BR>
     * @return boolean 受信処理中であるかチェックします.<BR>
     *                 [true]が返された場合、受信処理中です.<BR>
     *                 [false]が返された場合、受信処理中ではありません.
     */
    public boolean isReceive() {
        
        boolean ret = false ;
        
        synchronized( privateSync ) {
            ret = executionReceive ;
        }
        
        return ret ;
        
    }
    
    /**
     * 通信不可能かチェック.
     * <BR><BR>
     * 通信が不可能であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                  [true]が返された場合、通信不可能です.<BR>
     *                  [false]が返された場合、通信可能です.
     */
    public boolean isClose() {
        if( this.socket == null ) {
            return true ;
        }
        return this.socket.isClosed() ;
    }
    
    /**
     * 受信処理開始.
     */
    private void startReceive() {
        synchronized( privateSync ) {
            executionReceive = true ;
        }
    }
    
    /**
     * 受信処理終了.
     */
    private void endReceive() {
        synchronized( privateSync ) {
            executionReceive = false ;
        }
    }
    
}
