/*
 * @(#)ConnectReceiveMq.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.net ;

import java.io.FileInputStream;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.security.KeyStore;
import java.security.SecureRandom;

import javax.net.ServerSocketFactory;
import javax.net.SocketFactory;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLServerSocket;
import javax.net.ssl.SSLServerSocketFactory;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManagerFactory;

/**
 * ソケット生成処理を行います.
 *  
 * @version 2007/01/02
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class CreateSocket {
    
    /**
     * デフォルト最大アクセス.
     */
    private static final int DEFAULT_MAX_ACCESS = 50 ;
    
    /**
     * SSLソケットファクトリ.
     */
    private static SSLSocketFactory sslSocketFactory = null ;
    
    /**
     * SSLサーバソケットファクトリ.
     */
    private static SSLServerSocketFactory sslServerSocketFactory = null ;
    
    /**
     * SSLサーバソケット初期化処理.
     * <BR><BR>
     * SSLサーバソケット初期化処理を行うオブジェクトです.
     * <BR>
     * @param option SSLソケットオプションを設定します.
     * @exception Exception 例外.
     */
    public static final void initServerSSL( SSLOption option )
        throws Exception {
        initSSL( true,option ) ;
    }
    
    /**
     * SSLクライアントソケット初期化処理.
     * <BR><BR>
     * SSLクライアントソケット初期化処理を行うオブジェクトです.
     * <BR>
     * @param option SSLソケットオプションを設定します.
     * @exception Exception 例外.
     */
    public static final void initClientSSL( SSLOption option )
        throws Exception {
        initSSL( false,option ) ;
    }
    
    /**
     * SSLソケット初期化処理.
     * <BR><BR>
     * SSLソケット初期化処理を行うオブジェクトです.
     * <BR>
     * @param mode 生成モードを設定します.<BR>
     *             [true]を設定した場合、サーバソケットファクトリを作成します.<BR>
     *             [false]を設定した場合、クライアントソケットファクトリを作成します.
     * @param option SSLソケットオプションを設定します.
     * @exception Exception 例外.
     */
    protected synchronized static final void initSSL(
        boolean mode,SSLOption option )
        throws Exception {
        
        if( option == null ) {
            return ;
        }
        
        // 認証ファクトリを設定.
        KeyManagerFactory keyFactory = null ;
        
        if( option.getKeyStoreFile() != null &&
            option.getKeyStore() != null &&
            option.getKeyManagerAlgorithm() != null ) {
            
            char[] keyStorePasswd = null ;
            if( option.getKeyStorePasswd() != null &&
                option.getKeyStorePasswd().length() > 0 ) {
                keyStorePasswd = option.getKeyStorePasswd().toCharArray() ;
            }
            
            KeyStore keyStore = KeyStore.getInstance( option.getKeyStore() ) ;
            keyStore.load( new FileInputStream( option.getKeyStoreFile() ),keyStorePasswd ) ;
            keyFactory = KeyManagerFactory.getInstance( option.getKeyManagerAlgorithm() ) ;
            keyFactory.init( keyStore,keyStorePasswd ) ;
            
        }
        
        // ピア認証信頼を判断するファクトリを設定.
        TrustManagerFactory trustFactory = null ;
        
        if( option.getTrustFile() != null &&
            option.getTrustStore() != null &&
            option.getTrustKeyManagerAlgorithm() != null ) {
            
            char[] trustPasswd = null ;
            if( option.getTrustPassword() != null &&
                option.getTrustPassword().length() > 0 ) {
                trustPasswd = option.getTrustPassword().toCharArray() ;
            }
            
            KeyStore trust = KeyStore.getInstance( option.getTrustStore() ) ;
            trust.load( new FileInputStream( option.getTrustFile() ),trustPasswd ) ;
            trustFactory = TrustManagerFactory.getInstance( option.getTrustKeyManagerAlgorithm() ) ;
            trustFactory.init( trust ) ;
            
        }
        
        // 乱数アルゴリズム(PRNG)を設定.
        SecureRandom prng = null ;
        
        if( option.getRandomAlgorithm() != null ) {
            prng = SecureRandom.getInstance( option.getRandomAlgorithm() ) ;
        }
        
        // SSLコンテキストを設定.
        SSLContext context = SSLContext.getInstance( option.getSslProtocol() ) ;
        context.init( keyFactory.getKeyManagers(),trustFactory.getTrustManagers(),prng ) ;
        
        // サーバソケットファクトリを生成.
        if( mode == true ) {
            sslServerSocketFactory = context.getServerSocketFactory() ;
        }
        // クライアントソケットファクトリを生成.
        else {
            sslSocketFactory = context.getSocketFactory() ;
        }
        
    }
    
    /**
     * サーバソケット生成処理を行います.
     * <BR><BR>
     * @param mode 対象のモードを設定します.<BR>
     *             [true]を設定した場合、SSLソケットを生成します.
     *             [false]を設定した場合、通常ソケットを生成します.
     * @param clientAuth クライアント認証を必要とするか設定します.<BR>
     *                   [true]を設定した場合は、必要とします.<BR>
     *                   [false]を設定した場合は、必要としません.
     * @param bindPort 対象のバインドポートを設定します.
     * @param bindAddress 対象のバインドアドレスを設定します.<BR>
     *                    [null]の場合、デフォルトアドレスを指定します.
     * @param backlog 最大接続数を設定します.<BR>
     *                [0]以下を設定した場合、デフォルト値を設定します.
     */
    public synchronized static final ServerSocket serverSocket(
        boolean mode,boolean clientAuth,int bindPort,InetAddress bindAddress,int backlog )
        throws Exception {
        
        if( backlog <= 0 ) {
            backlog = DEFAULT_MAX_ACCESS ;
        }
        
        if( mode == true ) {
            ServerSocketFactory ss = null ;
            
            if( sslServerSocketFactory != null ) {
                ss = sslServerSocketFactory ;
            }
            else {
                return null ;
                //ss = SSLServerSocketFactory.getDefault() ;
            }
            
            ServerSocket socket = ss.createServerSocket(
                bindPort,backlog,bindAddress ) ;
            
            // クライアント認証設定の場合.
            if( clientAuth == true ){
                ( ( SSLServerSocket )socket ).setNeedClientAuth( true ) ;
            }
            
            return socket ;
        }
        else {
            return new ServerSocket( bindPort,backlog,bindAddress ) ;
        }
        
    }
    
    /**
     * クライアントソケット生成処理を行います.
     * <BR><BR>
     * @param mode 対象のモードを設定します.<BR>
     *             [true]を設定した場合、SSLソケットを生成します.
     *             [false]を設定した場合、通常ソケットを生成します.
     * @param address 接続先アドレスを設定します.
     * @param port 接続先ポート番号を設定します.
     * @return Socket 対象のソケットオブジェクトが返されます.
     * @exception Exception 例外.
     */
    public synchronized static final Socket clientSocket(
        boolean mode,InetAddress address,int port )
        throws Exception {
        
        if( mode == true ) {
            SocketFactory ss = null ;
            
            if( sslSocketFactory != null ) {
                ss = sslSocketFactory ;
            }
            else {
                return null ;
                //ss = SSLSocketFactory.getDefault() ;
            }
            
            Socket socket = ss.createSocket( address,port ) ;
            
            // SSLハンドシェイク.
            ( ( SSLSocket )socket ).startHandshake() ;
            
            return socket ;
        }
        else {
            return new Socket( address,port ) ;
        }
    }
    
}

