/*
 * @(#)CommonProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol ;

import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.protocol.admin.AdminCacheProtocol;
import org.maachang.queue.access.protocol.admin.AdminChannelProtocol;
import org.maachang.queue.access.protocol.admin.AdminConnectProtocol;
import org.maachang.queue.access.protocol.admin.AdminQueueManagerProtocol;
import org.maachang.queue.access.protocol.admin.AdminQueueProtocol;
import org.maachang.queue.access.protocol.admin.AdminUserProtocol;
import org.maachang.queue.access.protocol.login.LoginProtocol;
import org.maachang.queue.access.util.ByteUtil;
import org.maachang.queue.access.util.ConvertBinary;
import org.maachang.queue.access.util.Encryption;
import org.maachang.queue.access.util.Sequence;


/**
 * 共通プロトコル.
 *  
 * @version 2007/01/04
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class CommonProtocol {
    
    /**
     * ルートオーナ権限処理タイプマスク.
     */
    public static final int MASK_ROOT_OWNER = 0x00001000 ;
    
    /**
     * 処理カテゴリマスク.
     */
    public static final int MASK_TYPE_CATEGORY = 0xffff0000 ;
    
    /**
     * 基本文字列変換.
     */
    public static final String CHARSET = "UTF8" ;
    
    /**
     * 基本Digest情報.
     */
    public static final String DIGEST = "SHA1" ;
    
    /**
     * ダミーコード.
     */
    public static final String DUMMY = "dummy" ;
    
    /**
     * 基本ヘッダ.
     */
    public static final byte[] HEADER = {
        ( byte )0x000000fa,( byte )0x00000099 } ;
    
    /**
     * ヘッダ詳細長.
     */
    public static final int SEC_HEADER_LENGTH = 2 ;
    
    /**
     * 全体電文長.
     */
    public static final int ALL_DATA_LENGTH = 4 ;
    
    /**
     * ID発行オブジェクト.
     */
    private static final Sequence SEQUENCE = new Sequence() ;
    
    /**
     * ヘッダ詳細開始値.
     */
    public static final int SEC_HEADER_OFFSET = 
        HEADER.length +                     // ヘッダ.
        3 ;                                 // プロトコルバージョン.
    
    /**
     * 電文容量取得開始値.
     */
    public static final int TELEGRAM_LENGTH_OFFSET =
        SEC_HEADER_OFFSET +                 // ヘッダ＋プロトコルバージョン.
        SEC_HEADER_LENGTH ;                 // ヘッダ詳細.
    
    /**
     * 暗号開始値.
     */
    public static final int ENCRYPTION_OFFSET =
        TELEGRAM_LENGTH_OFFSET +            // 電文基本ヘッダ.
        4 +                                 // 電文全体の長さ.
        4 +                                 // ID.
        Encryption.ENCRYPION_KEY_LENGTH +   // 暗号キー.
        Encryption.STEP_CODE_LENGTH ;       // ステップコード.
    
    /**
     * バージョンマスク.
     */
    public static final int MASK_VERSION_INFO = 0x00ffff00 ;
    
    /**
     * バージョンコード.
     */
    private static final char[] VERSION_CODE = {
        '0','1','2','3','4','5','6','7',
        '8','9','a','b','c','d','e','f'
    } ;
    
    /**
     * MaachangQAccessライブラリのプロトコルバージョンを取得.
     * <BR><BR>
     * MaachangQAccessライブラリのプロトコルバージョンを取得します.
     * <BR>
     * @return int プロトコルバージョンが返されます.
     */
    public static final int getProtocolVersion() {
        
        int version =
            0x100000 |      // メジャーバージョン.
             0x00000 |      // マイナーバージョン1.
              0x0000 |      // マイナーバージョン2.
               0x000 |      // マイナーバージョン3.
                0x00 |      // 予備1.
                 0x0 ;      // 予備2.
        
        return version ;
        
    }
    
    /**
     * プロトコルバージョンを文字列に変換.
     * <BR><BR>
     * 対象のプロトコルバージョンを文字列に変換します.
     * <BR>
     * @param version 変換対象の数値を設定します.
     * @return String 文字列に変換されたバージョン情報が返されます.
     */
    public static final String getStringVersion( int version ) {
        
        return new StringBuffer().
            append( ( ( version & 0x00f00000 ) >> 20 ) ).
            append( "." ).
            append( ( ( version & 0x000f0000 ) >> 16 ) ).
            append( "." ).
            append( VERSION_CODE[ ( ( version & 0x0000f000 ) >> 12 ) ] ).
            append( "-" ).
            append( ( ( version & 0x00000f00 ) >> 8 ) ).
            append( "." ).
            append( VERSION_CODE[ ( ( version & 0x000000f0 ) >> 4 ) ] ).
            append( "-" ).
            append( VERSION_CODE[ ( ( version & 0x0000000f ) ) ] ).
            toString() ;
            
    }
    
    /**
     * プロトコル情報を生成.
     * <BR><BR>
     * プロトコル情報を生成します.
     * <BR>
     * @param secHeader ヘッダ詳細を設定します.
     * @param id 電文IDを設定します.
     * @param binary バイナリ情報を設定します.
     * @return byte[] バイナリ情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( byte[] secHeader,int id,byte[] binary )
        throws Exception {
        
        int len ;
        int stepPoint ;
        
        if( secHeader == null || secHeader.length != SEC_HEADER_LENGTH ||
            id < 0 || binary == null || ( len = binary.length ) <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        int binaryLength = ENCRYPTION_OFFSET +  // 暗号開始オフセット値.
            4 +                                 // 電文長.
            len ;                               // 電文情報.
        
        byte[] ret = new byte[ binaryLength ] ;
        int pnt = 0 ;
        
        // ヘッダ情報を生成.
        ret[ 0 ] = HEADER[ 0 ] ;
        ret[ 1 ] = HEADER[ 1 ] ;
        
        // プロトコルバージョン.
        int version = CommonProtocol.getProtocolVersion() ;
        ret[ 2 ] = ( byte )( ( version & 0x00ff0000 ) >> 16 ) ;
        ret[ 3 ] = ( byte )( ( version & 0x000000ff ) ) ;
        ret[ 4 ] = ( byte )( ( version & 0x0000ff00 ) >> 8 ) ;
        
        // ヘッダ詳細を生成.
        ret[ 5 ] = secHeader[ 0 ] ;
        ret[ 6 ] = secHeader[ 1 ] ;
        pnt = 7 ;
        
        // 全体電文長.
        ConvertBinary.convertInt( ret,pnt,binaryLength ) ;
        pnt += 4 ;
        
        // ID情報.
        ConvertBinary.convertInt( ret,pnt,id ) ;
        ret[ pnt   ] = ( byte )( ( ~ret[ pnt   ] ) & 0x000000ff ) ;
        ret[ pnt+1 ] = ( byte )( ( ~ret[ pnt+1 ] ) & 0x000000ff ) ;
        ret[ pnt+2 ] = ( byte )( ( ~ret[ pnt+2 ] ) & 0x000000ff ) ;
        ret[ pnt+3 ] = ( byte )( ( ~ret[ pnt+3 ] ) & 0x000000ff ) ;
        pnt += 4 ;
        
        // 暗号コード.
        byte[] encryptionKey = Encryption.getPublicKey() ;
        System.arraycopy( encryptionKey,0,ret,pnt,Encryption.ENCRYPION_KEY_LENGTH ) ;
        pnt += Encryption.ENCRYPION_KEY_LENGTH ;
        
        // ステップコード.
        stepPoint = pnt ;
        ret[ pnt ] = ( byte )0x00000000 ;
        pnt += 1 ;
        
        // 電文長.
        ConvertBinary.convertInt( ret,pnt,len ) ;
        pnt += 4 ;
        
        // 電文情報.
        System.arraycopy( binary,0,ret,pnt,len ) ;
        pnt += len ;
        
        // 暗号処理.
        Encryption enc = new Encryption() ;
        int step = enc.encryption( encryptionKey,ret,ENCRYPTION_OFFSET,
            ret.length - ENCRYPTION_OFFSET ) ;
        
        // ステップコードを設定.
        ret[ stepPoint ] = ( byte )( step & 0x000000ff ) ;
        
        return ret ;
        
    }
    
    /**
     * プロトコル情報から、バイナリ情報を取得.
     * <BR><BR>
     * プロトコル情報から、バイナリ情報を取得します.
     * <BR>
     * @param out ヘッダ詳細を受け取る情報を配置します.<BR>
     *            [null]を設定した場合、情報は配置されません.
     * @param outId ID情報を受け取る情報を配置します.<BR>
     *           [null]を設定した場合、情報はチェックしません.
     * @param protocol 解析対象のプロトコル情報を設定します.
     * @return byte[] 解析されたバイナリ情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] analysisProtocol( byte[] out,int[] outId,byte[] protocol )
        throws Exception {
        
        int pnt ;
        
        if( protocol == null || protocol.length <= 0 ||
            protocol[ 0 ] != HEADER[ 0 ] || protocol[ 1 ] != HEADER[ 1 ] ) {
            if( protocol == null ) {
                throw new IllegalArgumentException( "引数は不正です" ) ;
            }
            throw new IllegalArgumentException( "プロトコル情報は不正です" ) ;
        }
        
        // プロトコルバージョンを取得.
        int version = ( int )(
            ( ( ( int )protocol[ 2 ] & 0x000000ff ) << 16 ) |
            ( ( ( int )protocol[ 3 ] & 0x000000ff ) ) |
            ( ( ( int )protocol[ 4 ] & 0x000000ff ) << 8 )
        ) ;
        
        // 現在のバージョンと比較.
        int nowVersion = CommonProtocol.getProtocolVersion() ;
        if( ( nowVersion & MASK_VERSION_INFO ) != ( version & MASK_VERSION_INFO ) ) {
            throw new MaachangQException( "プロトコルバージョン[now:" + getStringVersion( nowVersion ) +
                " protocol:" + getStringVersion( version ) + "]が一致しません",
                MaachangQErrorCode.ERROR_PROTOCOL_VERSION ) ;
        }
        
        // ヘッダ詳細を受け取る場合.
        if( out != null && out.length == SEC_HEADER_LENGTH ) {
            out[ 0 ] = protocol[ 5 ] ;
            out[ 1 ] = protocol[ 6 ] ;
        }
        
        pnt = 7 ;
        
        // 電文長を取得.
        int binaryLength = ConvertBinary.convertInt( pnt,protocol ) ;
        pnt += 4 ;
        
        // 内部電文長とデータ長が一致しない場合.
        if( binaryLength != protocol.length ) {
            throw new IllegalArgumentException(
                "バイナリ全体長(" + binaryLength +
                ")とプロトコル受信バイナリ長(" + protocol.length +
                ")は一致しません" ) ;
        }
        
        // ID情報を取得.
        protocol[ pnt   ] = ( byte )( ( ~protocol[ pnt   ] ) & 0x000000ff ) ;
        protocol[ pnt+1 ] = ( byte )( ( ~protocol[ pnt+1 ] ) & 0x000000ff ) ;
        protocol[ pnt+2 ] = ( byte )( ( ~protocol[ pnt+2 ] ) & 0x000000ff ) ;
        protocol[ pnt+3 ] = ( byte )( ( ~protocol[ pnt+3 ] ) & 0x000000ff ) ;
        int protocolId = ConvertBinary.convertInt( pnt,protocol ) ;
        pnt += 4 ;
        
        // ID取得対象の場合.
        if( outId != null && outId.length == 1 ) {
            outId[ 0 ] = protocolId ;
        }
        
        // 暗号コードを取得.
        byte[] encryptionKey = new byte[ Encryption.ENCRYPION_KEY_LENGTH ] ;
        System.arraycopy( protocol,pnt,encryptionKey,0,
            Encryption.ENCRYPION_KEY_LENGTH ) ;
        pnt += Encryption.ENCRYPION_KEY_LENGTH ;
        
        // ステップコードを取得.
        int stepCode = ( int )( protocol[ pnt ] & 0x000000ff ) ;
        pnt += 1 ;
        
        // 暗号解析処理.
        Encryption enc = new Encryption() ;
        enc.analysis( encryptionKey,stepCode,protocol,
            ENCRYPTION_OFFSET,protocol.length - ENCRYPTION_OFFSET ) ;
        
        // 電文長を取得.
        int telegramLength = ConvertBinary.convertInt( pnt,protocol ) ;
        pnt += 4 ;
        
        // 電文長が不正な場合.
        if( telegramLength <= 0 || telegramLength >= binaryLength ) {
            throw new IllegalArgumentException(
                "受信電文長(" + telegramLength + ")は不正です" ) ;
        }
        
        // 電文情報を取得.
        byte[] ret = new byte[ telegramLength ] ;
        System.arraycopy( protocol,pnt,ret,0,telegramLength ) ;
        
        return ret ;
    }
    
    /**
     * ID情報取得.
     * <BR><BR>
     * ID情報を取得します.
     * <BR>
     * @return int ID情報が返されます.
     */
    public static final int getSequenceId() {
        return SEQUENCE.getId() ;
    }
    
    /**
     * プロトコルデータ長を取得.
     * <BR><BR>
     * プロトコルデータ長を取得します.
     * <BR>
     * @param byteUtil 対象のバイトユーティリティを設定します.
     * @return int プロトコルデータ長が返されます.
     * @exception Exception 例外.
     */
    public static final int getProtocolLength( ByteUtil byteUtil )
        throws Exception {
        
        if( byteUtil == null || byteUtil.size() < TELEGRAM_LENGTH_OFFSET ) {
            throw new IllegalArgumentException( "指定引数は不正です" ) ;
        }
        
        if( byteUtil.get( 0 ) != HEADER[ 0 ] || byteUtil.get( 1 ) != HEADER[ 1 ] ) {
            return -1 ;
        }
        
        return ( int )(
            ( byteUtil.get( TELEGRAM_LENGTH_OFFSET ) & 0x000000ff ) |
            ( ( int )( byteUtil.get( TELEGRAM_LENGTH_OFFSET+1 ) & 0x000000ff ) << 8 ) |
            ( ( int )( byteUtil.get( TELEGRAM_LENGTH_OFFSET+2 ) & 0x000000ff ) << 16 ) |
            ( ( int )( byteUtil.get( TELEGRAM_LENGTH_OFFSET+3 ) & 0x000000ff ) << 24 ) ) ;
    }
    
    /**
     * プロトコル判別.
     * <BR><BR>
     * 対象プロトコルを判別します.
     * <BR>
     * @param telegram 対象の電文情報を設定します.
     * @return int 判別されたカテゴリ情報が返されます.<BR>
     *             [-1]が返された場合、不正な電文です.
     */
    public static final int protocolByCategory( byte[] telegram ) {
        
        if( telegram.length <= TELEGRAM_LENGTH_OFFSET ||
             telegram[ 0 ] != HEADER[ 0 ] || telegram[ 1 ] != HEADER[ 1 ] ) {
            return -1 ;
        }
        
        // 詳細ヘッダを取得.
        byte[] sec = {
            telegram[ SEC_HEADER_OFFSET ],telegram[ SEC_HEADER_OFFSET + 1 ]
        } ;
        
        // プロトコルがメッセージの場合.
        if( sec[ 0 ] == MessageProtocol.HEADER_MESSAGEE[ 0 ] &&
            sec[ 1 ] == MessageProtocol.HEADER_MESSAGEE[ 1 ] ) {
            return MessageProtocol.CATEGORY_TYPE_MESSAGE ;
        }
        // プロトコルが処理結果の場合.
        else if( sec[ 0 ] == ResultProtocol.HEADER_RESULT[ 0 ] &&
            sec[ 1 ] == ResultProtocol.HEADER_RESULT[ 1 ] ) {
            return ResultProtocol.CATEGORY_TYPE_RESULT ;
        }
        // プロトコルがログインの場合.
        else if( sec[ 0 ] == LoginProtocol.HEADER_ADMIN_LOGIN[ 0 ] &&
            sec[ 1 ] == LoginProtocol.HEADER_ADMIN_LOGIN[ 1 ] ) {
            return LoginProtocol.CATEGORY_TYPE_LOGIN ;
        }
        // プロトコルがログイン結果の場合.
        else if( sec[ 0 ] == LoginProtocol.HEADER_ADMIN_RESULT_LOGIN[ 0 ] &&
            sec[ 1 ] == LoginProtocol.HEADER_ADMIN_RESULT_LOGIN[ 1 ] ) {
            return LoginProtocol.CATEGORY_TYPE_RESULT_LOGIN ;
        }
        // プロトコルがユーザ管理の場合.
        else if( sec[ 0 ] == AdminUserProtocol.HEADER_ADMIN_USER[ 0 ] &&
            sec[ 1 ] == AdminUserProtocol.HEADER_ADMIN_USER[ 1 ] ) {
            return AdminUserProtocol.CATEGORY_TYPE_ADMIN_USER ;
        }
        // プロトコルがキュー管理の場合.
        else if( sec[ 0 ] == AdminQueueProtocol.HEADER_ADMIN_QUEUE[ 0 ] &&
            sec[ 1 ] == AdminQueueProtocol.HEADER_ADMIN_QUEUE[ 1 ] ) {
            return AdminQueueProtocol.CATEGORY_TYPE_ADMIN_QUEUE ;
        }
        // プロトコルがキューマネージャ管理の場合.
        else if( sec[ 0 ] == AdminQueueManagerProtocol.HEADER_ADMIN_QMANAGER[ 0 ] &&
            sec[ 1 ] == AdminQueueManagerProtocol.HEADER_ADMIN_QMANAGER[ 1 ] ) {
            return AdminQueueManagerProtocol.CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ;
        }
        // プロトコルがチャネル管理の場合.
        else if( sec[ 0 ] == AdminChannelProtocol.HEADER_ADMIN_CHANNEL[ 0 ] &&
            sec[ 1 ] == AdminChannelProtocol.HEADER_ADMIN_CHANNEL[ 1 ] ) {
            return AdminChannelProtocol.CATEGORY_TYPE_ADMIN_CHANNEL ;
        }
        // プロトコルがコネクション管理の場合.
        else if( sec[ 0 ] == AdminConnectProtocol.HEADER_ADMIN_CONNECT[ 0 ] &&
            sec[ 1 ] == AdminConnectProtocol.HEADER_ADMIN_CONNECT[ 1 ] ) {
            return AdminConnectProtocol.CATEGORY_TYPE_ADMIN_CONNECT ;
        }
        // プロトコルがキャッシュ管理の場合.
        else if( sec[ 0 ] == AdminCacheProtocol.HEADER_ADMIN_CACHE[ 0 ] &&
            sec[ 1 ] == AdminCacheProtocol.HEADER_ADMIN_CACHE[ 1 ] ) {
            return AdminCacheProtocol.CATEGORY_TYPE_ADMIN_CACHE ;
        }
        
        return -1 ;
    }
    
    
    
    /**
     * 電文チェックコードを取得.
     * <BR><BR>
     * 電文チェックコードを取得します.
     * <BR>
     * @param binary チェックコード生成対象のバイナリを設定します.
     * @return int 対象のチェックコードが返されます.
     */
    public static final int getCheckCode( byte[] binary,int length ) {
        
        int len ;
        int ret ;
        
        if( binary == null || ( len = binary.length ) <= 0 || length < len ) {
            return -1 ;
        }
        
        ret = ( ( ( ~len ) + length ) & 0x3fffffff ) ;
        for( int i = 0 ; i < length ; i ++ ) {
            
            ret = ( ret & 0x3fffffff ) ;
            switch( ( ret & 0x00000007 ) ) {
                case 0 :
                    ret = ( ret + ( ( int )binary[ i ] & 0x000000ff ) ) ;
                    break ;
                case 1 :
                    ret = ( ret ^ ( ~( ( ( int )binary[ i ] & 0x000000ff ) << 1 ) ) ) ;
                    break ;
                case 2 :
                    ret = ( ret + ( ( ( int )binary[ i ] & 0x000000ff ) << 2 ) ) ;
                    break ;
                case 3 :
                    ret = ( ret ^ ( ( ( int )binary[ i ] & 0x000000ff ) << 3 ) ) ;
                    break ;
                case 4 :
                    ret = ( ret + ( ( ( int )binary[ i ] & 0x000000ff ) << 3 ) ) ;
                    break ;
                case 5 :
                    ret = ( ret + ( ~( ( ( int )binary[ i ] & 0x000000ff ) << 2 ) ) ) ;
                    break ;
                case 6 :
                    ret = ( ret + ( ( ( int )binary[ i ] & 0x000000ff ) << 1 ) ) ;
                    break ;
                case 7 :
                    ret = ( ret ^ ( ( int )binary[ i ] & 0x000000ff ) ) ;
                    break ;
            }
            
        }
        
        return ( int )( ret & 0x3fffffff ) ;
        
    }
    
    /**
     * チェックコードを設定.
     * <BR><BR>
     * チェックコードをバイナリに設定します.
     * <BR>
     * @param telegram 設定対象のバイナリを設定します.
     */
    public static final void setCheckCode( byte[] telegram ) {
        int endLength = telegram.length - 4 ;
        int code = CommonProtocol.getCheckCode( telegram,endLength ) ;
        ConvertBinary.convertInt( telegram,endLength,code ) ;
    }
    
    /**
     * チェックコードが正しいかチェック.
     * <BR><BR>
     * チェックコードが正しいかチェックします.
     * <BR>
     * @param telegra チェック対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final void checkCheckCode( byte[] telegram )
        throws Exception {
        
        // チェックコードを判別.
        int endLength = telegram.length - 4 ;
        int code = ConvertBinary.convertInt( endLength,telegram ) ;
        if( code != CommonProtocol.getCheckCode( telegram,endLength ) ) {
            throw new IllegalArgumentException( "チェックコードは不正です" ) ;
        }
        
    }
    
    /**
     * 電文長が一致するかチェック.
     * <BR><BR>
     * 電文長が一致するかチェックします.
     * <BR>
     * @param telegram 電文情報を設定します.
     * @exception Exception 例外.
     */
    public static final void checkTelegramLength( byte[] telegram )
        throws Exception {
        
        // 電文長をチェック.
        int telegramLength = ConvertBinary.convertInt( 0,telegram ) ;
        if( telegramLength != telegram.length ) {
            throw new IllegalArgumentException( "電文長は不正です" ) ;
        }
        
    }
    
    /**
     * 指定タイプがルート権限を必要とするかチェック.
     * <BR><BR>
     * 指定タイプがルート権限を必要とするかチェックします.
     * <BR>
     * @param rootOwner 対象ユーザのルート権限を設定します.
     * @param type 対象の処理タイプを設定します.
     * @exception Exception 例外.
     */
    public static final void checkRootOwnerByType( boolean rootOwner,int type )
        throws Exception {
        
        // 権限マスクでチェック.
        if( rootOwner == false &&
            ( type & CommonProtocol.MASK_ROOT_OWNER ) == CommonProtocol.MASK_ROOT_OWNER ) {
            
            throw new MaachangQException(
                "権限が足りません",
                MaachangQErrorCode.ERROR_LOGIN_NOT_OWNWE ) ;
        }
        
    }
}

