/*
 * @(#)AdminCacheProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol.admin ;

import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.ConvertBinary;


/**
 * 管理者キャッシュ用プロトコル.
 * <BR><BR>
 * 管理者キャッシュ用プロトコルを表すオブジェクト.
 *  
 * @version 2007/01/05
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class AdminCacheProtocol {
    
    /**
     * キャッシュ用ヘッダ.
     */
    public static final byte[] HEADER_ADMIN_CACHE = {
        ( byte )0x00000ac,( byte )0x000000a5 } ;
    
    /**
     * 処理カテゴリ : 管理者用キャッシュ系.
     */
    public static final int CATEGORY_TYPE_ADMIN_CACHE = 0x87000000 ;
    
    /**
     * 処理タイプ : キャッシュ追加.
     */
    public static final int TYPE_ADD_CACHE = 0x00000001 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_CACHE ;
    
    /**
     * 処理タイプ : キャッシュ削除.
     */
    public static final int TYPE_REMOVE_CACHE = 0x00000002 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_CACHE ;
    
    /**
     * 処理タイプ : キャッシュ情報取得.
     */
    public static final int TYPE_GET_CACHE = 0x00000013 |
        CATEGORY_TYPE_ADMIN_CACHE ;
    
    /**
     * 処理タイプ : 存在するキャッシュ名群取得.
     */
    public static final int TYPE_GET_CACHE_NAMES = 0x00000014 |
        CATEGORY_TYPE_ADMIN_CACHE ;
    
    /**
     * 処理タイプ : 存在するキャッシュ数取得.
     */
    public static final int TYPE_GET_CACHE_SIZE = 0x00000015 |
        CATEGORY_TYPE_ADMIN_CACHE ;
    
    /**
     * 処理タイプ : キャッシュが存在するか取得.
     */
    public static final int TYPE_IS_CACHE = 0x00000016 |
        CATEGORY_TYPE_ADMIN_CACHE ;
    
    /**
     * 処理タイプチェック.
     * <BR><BR>
     * 処理タイプをチェックします.
     * <BR>
     * @param type 処理タイプを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                  [true]が返された場合、処理タイプは存在します.<BR>
     *                  [false]が返された場合、処理タイプは存在しません.
     */
    public static final boolean checkType( int type ) {
        switch( type ) {
            case TYPE_ADD_CACHE :
            case TYPE_REMOVE_CACHE :
            case TYPE_GET_CACHE :
            case TYPE_GET_CACHE_NAMES :
            case TYPE_GET_CACHE_SIZE :
            case TYPE_IS_CACHE :
            return true ;
        }
        return false ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコル処理を生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param bean 設定対象のBeanを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol(
        int id,boolean rootOwner,AdminCacheBean bean )
        throws Exception {
        
        if( id < 0 || bean == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        return AdminCacheProtocol.createProtocol(
            id,bean.getUserId(),rootOwner,bean.getType(),
            bean.getName(),bean.getMaxSize(),bean.getUseSize(),
            bean.getSize(),bean.getParams() ) ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param type 処理タイプを設定します.
     * @param name 対象のキャッシュ名を設定します.
     * @param maxSize 最大キャッシュサイズを設定します.
     * @param useSize 利用中キャッシュサイズを設定します.
     * @param size 対象のキャッシュサイズを設定します.
     * @param params 対象のパラメータを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int userId,boolean rootOwner,int type,
        String name,int maxSize,int useSize,
        int size,String[] params )
        throws Exception {
        
        // 処理タイプが不正な場合.
        if( AdminCacheProtocol.checkType( type ) == false ) {
            throw new MaachangQException(
                "処理タイプ(" + type + ")が不正です",
                MaachangQErrorCode.ERROR_NOT_TYPE ) ;
        }
        
        // 電文情報長を計算.
        int binaryLength = 
            4 +                 // 最大キャッシュサイズ.
            4 ;                 // 利用中キャッシュサイズ.
        
        // 電文情報を生成.
        int[] nextPnt = new int[ 1 ] ;
        byte[] telegram = BaseAdminProtocol.createCommonTelegram(
            nextPnt,binaryLength,id,userId,rootOwner,type,name,size,params ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // 最大キャッシュサイズ.
        ConvertBinary.convertInt( telegram,pnt,maxSize ) ;
        pnt += 4 ;
        
        // 利用中キャッシュサイズ.
        ConvertBinary.convertInt( telegram,pnt,useSize ) ;
        pnt += 4 ;
        
        // チェックコード設定.
        CommonProtocol.setCheckCode( telegram ) ;
        
        // 電文情報生成.
        return CommonProtocol.createProtocol( HEADER_ADMIN_CACHE,id,telegram ) ;
        
    }
    
    /**
     * プロトコルを解析.
     * <BR><BR>
     * プロトコルを解析します.<BR>
     * この処理で渡される電文データは、あらかじめ[CommonProtocol.analysisProtocol()]で
     * 解析しておく必要があります.
     * <BR>
     * @param out 解析結果を格納するBeanを設定します.
     * @param telegram 解析対象の電文データを設定します.
     * @exception Exception 例外.
     */
    public static final void analysisProtocol( AdminCacheBean out,byte[] telegram )
        throws Exception {
        
        // 電文情報を解析.
        int[] nextPnt = new int[ 1 ] ;
        BaseAdminProtocol.analysisCommonTelegram( nextPnt,out,telegram ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // 最大キャッシュサイズを取得.
        int maxSize = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 利用中キャッシュサイズを取得.
        int useSize = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // Beanに設定.
        out.setMaxSize( maxSize ) ;
        out.setUseSize( useSize ) ;
        
    }
    
}

