/*
 * @(#)AdminQueueProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.ConvertBinary;


/**
 * 管理者キュー用プロトコル.
 * <BR><BR>
 * 管理者キュー用プロトコルを表すオブジェクト.
 *  
 * @version 2007/01/05
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class AdminQueueProtocol {
    
    /**
     * キュー用ヘッダ.
     */
    public static final byte[] HEADER_ADMIN_QUEUE = {
        ( byte )0x00000a9,( byte )0x00000099 } ;
    
    /**
     * 処理カテゴリ : 管理者用キュー系.
     */
    public static final int CATEGORY_TYPE_ADMIN_QUEUE = 0x83000000 ;
    
    /**
     * 処理タイプ : キュー追加.
     */
    public static final int TYPE_ADD_QUEUE = 0x00000001 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_QUEUE ;
    
    /**
     * 処理タイプ : キュー設定.
     */
    public static final int TYPE_SET_QUEUE = 0x00000002 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_QUEUE ;
    
    /**
     * 処理タイプ : キュー削除.
     */
    public static final int TYPE_REMOVE_QUEUE = 0x00000003 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_QUEUE ;
    
    /**
     * 処理タイプ : キュー情報取得.
     */
    public static final int TYPE_GET_QUEUE = 0x00000014 |
        CATEGORY_TYPE_ADMIN_QUEUE ;
    
    /**
     * 処理タイプ : 存在するキュー名群取得.
     */
    public static final int TYPE_GET_QUEUE_NAMES = 0x00000015 |
        CATEGORY_TYPE_ADMIN_QUEUE ;
    
    /**
     * 処理タイプ : 存在するキュー数取得.
     */
    public static final int TYPE_GET_QUEUE_SIZE = 0x00000016 |
        CATEGORY_TYPE_ADMIN_QUEUE ;
    
    /**
     * 処理タイプ : キューが存在するか取得.
     */
    public static final int TYPE_IS_QUEUE = 0x00000017 |
        CATEGORY_TYPE_ADMIN_QUEUE ;
    
    /**
     * 処理タイプチェック.
     * <BR><BR>
     * 処理タイプをチェックします.
     * <BR>
     * @param type 処理タイプを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                  [true]が返された場合、処理タイプは存在します.<BR>
     *                  [false]が返された場合、処理タイプは存在しません.
     */
    public static final boolean checkType( int type ) {
        switch( type ) {
            case TYPE_ADD_QUEUE :
            case TYPE_SET_QUEUE :
            case TYPE_REMOVE_QUEUE :
            case TYPE_GET_QUEUE :
            case TYPE_GET_QUEUE_NAMES :
            case TYPE_GET_QUEUE_SIZE :
            case TYPE_IS_QUEUE :
            return true ;
        }
        return false ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコル処理を生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param bean 設定対象のBeanを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol(
        int id,boolean rootOwner,AdminQueueBean bean )
        throws Exception {
        
        if( bean == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        return AdminQueueProtocol.createProtocol(
            id,bean.getUserId(),rootOwner,bean.getType(),
            bean.getName(),bean.getManager(),bean.getQueueType(),
            bean.getMaxMessageLength(),bean.getMaxQueue(),bean.getUseQueue(),
            bean.getWarningPersend(),bean.isGzipFlag(),
            bean.isAutoCommit(),bean.getCacheName(),bean.getState(),
            bean.getNumbers(),bean.getSize(),
            bean.getParams() ) ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param type 処理タイプを設定します.
     * @param name 対象のキュー名を設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @param maxMessageLength 最大メッセージ制限値を設定します.
     * @param maxQueue 最大キューサイズを設定します.
     * @param useQueue 現在キューサイズを設定します.
     * @param warningPersend 警告表示パーセントを設定します.
     * @param gzipFlag GZIP圧縮モードを設定します.
     * @param autoCommit オートコミットモードを設定します.
     * @param cacheName 対象のキャッシュ名を設定します.
     * @param state 対象ステータスを設定します.
     * @param numbers 対象の数値パラメータを設定します.
     * @param size 対象のキューサイズを設定します.
     * @param params 対象のパラメータを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int userId,boolean rootOwner,int type,
        String name,String manager,int queueType,int maxMessageLength,int maxQueue,int useQueue,
        int warningPersent,boolean gzipFlag,boolean autoCommit,String cacheName,
        int state,int[] numbers,int size,String[] params )
        throws Exception {
        
        // 引数が不正.
        if( queueType != MaachangQAccessDefine.TYPE_SEND &&
                queueType != MaachangQAccessDefine.TYPE_RECEIVE ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        // 処理タイプが不正な場合.
        if( AdminQueueProtocol.checkType( type ) == false ) {
            throw new MaachangQException(
                "処理タイプ(" + type + ")が不正です",
                MaachangQErrorCode.ERROR_NOT_TYPE ) ;
        }
        
        // 対象キューマネージャ名をバイナリ変換.
        byte[] managerBin = manager.getBytes( CommonProtocol.CHARSET ) ;
        int managerLen = managerBin.length ;
        
        // 数値パラメータ数を取得.
        int numbersLen = 0 ;
        if( numbers != null ) {
            numbersLen = numbers.length ;
        }
        
        // キャッシュ名をバイナリ化.
        byte[] cacheBin = null ;
        int cacheLen = 0 ;
        if( cacheName != null &&
            ( cacheName = cacheName.trim().toLowerCase() ).length() > 0 ) {
            cacheBin = cacheName.getBytes( CommonProtocol.CHARSET ) ;
            cacheLen = cacheBin.length ;
        }
        
        
        // 電文情報長を計算.
        int binaryLength = 
            4 +                 // キュータイプ.
            4 +                 // キューマネージャ名長.
            managerLen +        // キューマネージャ名.
            4 +                 // 最大メッセージ制限値.
            4 +                 // 最大キューサイズ.
            4 +                 // 現在キューサイズ.
            4 +                 // 警告パーセント.
            1 +                 // GZIP圧縮モード.
            1 +                 // オートコミットモード.
            4 +                 // キャッシュ名長.
            cacheLen +          // キャッシュ名.
            4 +                 // ステータス.
            4 +                 // 数値パラメータ数.
            numbersLen * 4 ;    // 数値パラメータ.
        
        // 電文情報を生成.
        int[] nextPnt = new int[ 1 ] ;
        byte[] telegram = BaseAdminProtocol.createCommonTelegram(
            nextPnt,binaryLength,id,userId,rootOwner,type,name,size,params ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // キュータイプ.
        ConvertBinary.convertInt( telegram,pnt,queueType ) ;
        pnt += 4 ;
        
        // キューマネージャ名長.
        ConvertBinary.convertInt( telegram,pnt,managerLen ) ;
        pnt += 4 ;
        
        // キューマネージャ名.
        System.arraycopy( managerBin,0,telegram,pnt,managerLen ) ;
        pnt += managerLen ;
        managerBin = null ;
        
        // 最大メッセージ制限値.
        maxMessageLength = ( maxMessageLength <= -1 ) ? -1 : maxMessageLength ;
        ConvertBinary.convertInt( telegram,pnt,maxMessageLength ) ;
        pnt += 4 ;
        
        // 最大キューサイズ.
        maxQueue = ( maxQueue <= -1 ) ? -1 : maxQueue ;
        ConvertBinary.convertInt( telegram,pnt,maxQueue ) ;
        pnt += 4 ;
        
        // 現在キューサイズ.
        useQueue = ( useQueue <= -1 ) ? -1 : useQueue ;
        ConvertBinary.convertInt( telegram,pnt,useQueue ) ;
        pnt += 4 ;
        
        // 警告パーセント.
        warningPersent = ( warningPersent <= -1 ) ? -1 : warningPersent ;
        ConvertBinary.convertInt( telegram,pnt,warningPersent ) ;
        pnt += 4 ;
        
        // GZIP圧縮モード.
        ConvertBinary.convertBoolean( telegram,pnt,gzipFlag ) ;
        pnt += 1 ;
        
        // オートコミットモード
        ConvertBinary.convertBoolean( telegram,pnt,autoCommit ) ;
        pnt += 1 ;
        
        // キャッシュ名サイズ.
        ConvertBinary.convertInt( telegram,pnt,cacheLen ) ;
        pnt += 4 ;
        
        if( cacheLen > 0 ) {
            
            // キャッシュ名.
            System.arraycopy( cacheBin,0,telegram,pnt,cacheLen ) ;
            pnt += cacheLen ;
            cacheBin = null ;
            
        }
        
        // ステータス.
        state = ( state <= -1 ) ? -1 : state ;
        ConvertBinary.convertInt( telegram,pnt,state ) ;
        pnt += 4 ;
        
        // 数値パラメータ数.
        ConvertBinary.convertInt( telegram,pnt,numbersLen ) ;
        pnt += 4 ;
        
        // 数値パラメータ.
        for( int i = 0 ; i < numbersLen ; i ++ ) {
            ConvertBinary.convertInt( telegram,pnt,numbers[ i ] ) ;
            pnt += 4 ;
        }
        
        // チェックコード設定.
        CommonProtocol.setCheckCode( telegram ) ;
        
        // 電文情報生成.
        return CommonProtocol.createProtocol( HEADER_ADMIN_QUEUE,id,telegram ) ;
        
    }
    
    /**
     * プロトコルを解析.
     * <BR><BR>
     * プロトコルを解析します.<BR>
     * この処理で渡される電文データは、あらかじめ[CommonProtocol.analysisProtocol()]で
     * 解析しておく必要があります.
     * <BR>
     * @param out 解析結果を格納するBeanを設定します.
     * @param telegram 解析対象の電文データを設定します.
     * @exception Exception 例外.
     */
    public static final void analysisProtocol( AdminQueueBean out,byte[] telegram )
        throws Exception {
        
        // 電文情報を解析.
        int[] nextPnt = new int[ 1 ] ;
        BaseAdminProtocol.analysisCommonTelegram( nextPnt,out,telegram ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // キュータイプを取得.
        int queueType = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キューマネージャ名長を取得.
        int managerLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キューマネージャ名を取得.
        byte[] managerBin = new byte[ managerLen ] ;
        System.arraycopy( telegram,pnt,managerBin,0,managerLen ) ;
        String manager = new String( managerBin,CommonProtocol.CHARSET ) ;
        managerBin = null ;
        pnt += managerLen ;
        
        // 最大メッセージ制限値を取得.
        int maxMessageLength = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 最大キューサイズを取得.
        int maxQueue = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 現在キューサイズ取得.
        int useQueue = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 警告パーセント取得.
        int warning = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // GZIP圧縮モードを取得.
        boolean gzipFlag = ConvertBinary.convertBoolean( pnt,telegram ) ;
        pnt += 1 ;
        
        // オートコミットを取得.
        boolean autoCommit = ConvertBinary.convertBoolean( pnt,telegram ) ;
        pnt += 1 ;
        
        // キャッシュ名長を取得.
        int cacheLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キャッシュ名を取得.
        String cacheName = null ;
        if( cacheLen > 0 ) {
            byte[] cacheBin = new byte[ cacheLen ] ;
            System.arraycopy( telegram,pnt,cacheBin,0,cacheLen ) ;
            cacheName = new String( cacheBin,CommonProtocol.CHARSET ) ;
            cacheBin = null ;
            pnt += cacheLen ;
        }
        
        // ステータスを取得.
        int state = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 数値パラメータ数を取得.
        int numbersLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 数値パラメータ.
        int[] numbers = null ;
        if( numbersLen > 0 ) {
            numbers = new int[ numbersLen ] ;
            for( int i = 0 ; i < numbersLen ; i ++ ) {
                numbers[ i ] = ConvertBinary.convertInt( pnt,telegram ) ;
                pnt += 4 ;
            }
        }
        
        // Beanに設定.
        out.setQueueType( queueType ) ;
        out.setManager( manager ) ;
        out.setMaxMessageLength( maxMessageLength ) ;
        out.setMaxQueue( maxQueue ) ;
        out.setUseQueue( useQueue ) ;
        out.setWarningPersend( warning ) ;
        out.setGzipFlag( gzipFlag ) ;
        out.setAutoCommit( autoCommit ) ;
        out.setCacheName( cacheName ) ;
        out.setState( state ) ;
        out.setNumbers( numbers ) ;
        
    }
    
}

