/*
 * @(#)LoginProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol.login ;

import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.ConvertBinary;
import org.maachang.queue.access.util.Digest;


/**
 * ログイン用プロトコル.
 * <BR><BR>
 * ログイン用プロトコルを表すオブジェクト.
 *  
 * @version 2007/01/05
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class LoginProtocol {
    
    /**
     * パスワードなしの条件.
     */
    public static final String DUMMY = CommonProtocol.DUMMY ;
    
    /**
     * パスワード用ダイジェスト.
     */
    public static final String PASSWD_DIGEST = CommonProtocol.DIGEST ;
    
    /**
     * 処理カテゴリ : ログイン系.
     */
    public static final int CATEGORY_TYPE_LOGIN = 0x81100000 ;
    
    /**
     * 処理カテゴリ : ログイン戻り系.
     */
    public static final int CATEGORY_TYPE_RESULT_LOGIN = 0x81200000 ;
    
    /**
     * ログイン用ヘッダ.
     */
    public static final byte[] HEADER_ADMIN_LOGIN = {
        ( byte )0x000000a1,( byte )0x0000009e } ;
    
    /**
     * ログイン戻り用ヘッダ.
     */
    public static final byte[] HEADER_ADMIN_RESULT_LOGIN = {
        ( byte )0x000000F1,( byte )0x0000009e } ;
    
    /**
     * ログインプロトコルを生成.
     * <BR><BR>
     * ログインプロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param user 対象のログインユーザを設定します.
     * @param passwd 対象のパスワードを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,String user,String passwd )
        throws Exception {
        return LoginProtocol.createProtocol( id,user,passwd,false ) ;
    }
    
    /**
     * ログインプロトコルを生成.
     * <BR><BR>
     * ログインプロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param user 対象のログインユーザを設定します.
     * @param passwd 対象のパスワードを設定します.
     * @param shutdown シャットダウンコマンドを送信するか設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,String user,String passwd,boolean shutdown )
        throws Exception {
        
        if( id < 0 || user == null ||
            ( user = user.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        // パスワードなしの場合は、パスワードなしの条件を設定.
        if( passwd == null ||
            ( passwd = LoginProtocol.trimPlus( passwd ) ).length() <= 0 ) {
            passwd = DUMMY ;
        }
        
        // ユーザー・パスワード用バイナリを生成.
        byte[] userBin = user.getBytes( CommonProtocol.CHARSET ) ;
        byte[] passwdBin = Digest.convertBinary(
            PASSWD_DIGEST,passwd.getBytes( CommonProtocol.CHARSET ) ) ;
        
        byte[] telegram = new byte[
            4 +                 // 全体電文長.
            4 +                 // 処理タイプ.
            1 +                 // シャットダウンコマンド.
            4 +                 // ユーザバイナリ格納長.
            4 +                 // パスワードバイナリ格納長.
            passwdBin.length +  // パスワードバイナリ長.
            userBin.length +    // ユーザバイナリ長.
            4                   // チェックコード.
        ] ;
        
        int pnt = 0 ;
        
        // 全体電文長.
        ConvertBinary.convertInt( telegram,pnt,telegram.length ) ;
        pnt += 4 ;
        
        // 処理タイプ.
        ConvertBinary.convertInt( telegram,pnt,CATEGORY_TYPE_LOGIN ) ;
        pnt += 4 ;
        
        // シャットダウンコマンド.
        ConvertBinary.convertBoolean( telegram,pnt,shutdown ) ;
        pnt += 1 ;
        
        // ユーザバイナリ長.
        ConvertBinary.convertInt( telegram,pnt,userBin.length ) ;
        pnt += 4 ;
        
        // パスワードバイナリ長.
        ConvertBinary.convertInt( telegram,pnt,passwdBin.length ) ;
        pnt += 4 ;
        
        // パスワードバイナリ.
        System.arraycopy( passwdBin,0,telegram,pnt,passwdBin.length ) ;
        pnt += passwdBin.length ;
        
        // ユーザバイナリ.
        System.arraycopy( userBin,0,telegram,pnt,userBin.length ) ;
        pnt += userBin.length ;
        
        // チェックコード設定.
        CommonProtocol.setCheckCode( telegram ) ;
        
        // 電文情報生成.
        return CommonProtocol.createProtocol( HEADER_ADMIN_LOGIN,id,telegram ) ;
        
    }
    
    /**
     * ログインプロトコルを解析.
     * <BR><BR>
     * ログインプロトコルを解析します.<BR>
     * この処理で渡される電文データは、あらかじめ[CommonProtocol.analysisProtocol()]で
     * 解析しておく必要があります.
     * <BR>
     * @param out 解析結果を格納するBeanを設定します.
     * @param telegram 解析対象の電文データを設定します.
     * @exception Exception 例外.
     */
    public static final void analysisProtocol( LoginBean out,byte[] telegram )
        throws Exception {
        
        if( out == null || telegram == null || telegram.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        // 電文長をチェック.
        CommonProtocol.checkTelegramLength( telegram ) ;
        int pnt = 4 ;
        
        // チェックコードを判別.
        CommonProtocol.checkCheckCode( telegram ) ;
        
        // 処理タイプを取得.
        int type = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // シャットダウンコマンドを取得.
        boolean shutdown = ConvertBinary.convertBoolean( pnt,telegram ) ;
        pnt += 1 ;
        
        // ユーザバイナリ長を取得.
        int userLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // パスワードバイナリ長を取得.
        int passwdLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // パスワードバイナリを取得.
        byte[] passwdBin = new byte[ passwdLen ] ;
        System.arraycopy( telegram,pnt,passwdBin,0,passwdLen ) ;
        pnt += passwdLen ;
        
        // ユーザバイナリを取得.
        byte[] userBin = new byte[ userLen ] ;
        System.arraycopy( telegram,pnt,userBin,0,userLen ) ;
        pnt += userLen ;
        
        // Beanに設定.
        out.setPasswd( passwdBin ) ;
        out.setUser( new String( userBin,CommonProtocol.CHARSET ) ) ;
        out.setType( type ) ;
        out.setShutdown( shutdown ) ;
        
    }
    
    /**
     * ログイン戻りプロトコルを生成.
     * <BR><BR>
     * ログイン戻りプロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param userId 対象のログインユーザIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createResultProtocol( int id,int userId,boolean rootOwner )
        throws Exception {
        
        if( id < 0 || userId < 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        byte[] telegram = new byte[
            4 +                 // 全体電文長.
            4 +                 // 処理タイプ.
            4 +                 // ユーザID.
            1 +                 // ルートオーナー.
            4                   // チェックコード.
        ] ;
        
        int pnt = 0 ;
        
        // 全体電文長.
        ConvertBinary.convertInt( telegram,pnt,telegram.length ) ;
        pnt += 4 ;
        
        // 処理タイプ.
        ConvertBinary.convertInt( telegram,pnt,CATEGORY_TYPE_RESULT_LOGIN ) ;
        pnt += 4 ;
        
        // ユーザID.
        ConvertBinary.convertInt( telegram,pnt,userId ) ;
        pnt += 4 ;
        
        // ルートオーナー.
        ConvertBinary.convertBoolean( telegram,pnt,rootOwner ) ;
        pnt += 1 ;
        
        // チェックコード設定.
        CommonProtocol.setCheckCode( telegram ) ;
        
        // 電文情報生成.
        return CommonProtocol.createProtocol( HEADER_ADMIN_RESULT_LOGIN,id,telegram ) ;
        
    }
    
    /**
     * ログイン戻りプロトコルを解析.
     * <BR><BR>
     * ログイン戻りプロトコルを解析します.<BR>
     * この処理で渡される電文データは、あらかじめ[CommonProtocol.analysisProtocol()]で
     * 解析しておく必要があります.
     * <BR>
     * @param out 解析結果を格納するBeanを設定します.
     * @param telegram 解析対象の電文データを設定します.
     * @exception Exception 例外.
     */
    public static final void analysisResultProtocol( LoginBean out,byte[] telegram )
        throws Exception {
        
        if( out == null || telegram == null || telegram.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        // 電文長をチェック.
        CommonProtocol.checkTelegramLength( telegram ) ;
        int pnt = 4 ;
        
        // チェックコードを判別.
        CommonProtocol.checkCheckCode( telegram ) ;
        
        // 処理タイプを取得.
        int type = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // ユーザIDを取得.
        int userId = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // ルートオーナーを取得.
        boolean rootOwner = ConvertBinary.convertBoolean( pnt,telegram ) ;
        pnt += 1 ;
        
        // Beanに設定.
        out.setId( userId ) ;
        out.setRootOwner( rootOwner ) ;
        out.setUser( null ) ;
        out.setPasswd( null ) ;
        out.setType( type ) ;
        
    }
    
    /**
     * 情報の前後に存在するスペース、タブ関連情報を削除します。
     * <BR><BR>
     * @param str 削除対象の情報を設定します。
     * @return String 削除された情報が格納されます。
     */
    public static final String trimPlus( String str ) {
        
        int i ;
        int s,e ;
        int len ;
        
        String ret = null ;
        
        char spaceCode ;
        char tabCode ;
        char space2Code ;
        
        char now ;
        
        if( str == null || ( len = str.length() ) <= 0){
            return "" ;
        }
        
        spaceCode = ' ' ;
        tabCode = '\t' ;
        space2Code = '　' ;
        
        for(i = 0,s = -1 ; i < len ; i ++){
            
            now = str.charAt( i ) ;
            
            if(
                now != spaceCode &&
                now != tabCode &&
                now != space2Code &&
                now != 0
            )
            {
                
                s = i ;
                break ;
                
            }
            
        }
        
        for(i = len - 1,e = -1 ; i >= 0 ; i --){
            
            now = str.charAt( i ) ;
            
            if(
                now != spaceCode &&
                now != tabCode &&
                now != space2Code &&
                now != 0
            )
            {
                
                e = i ;
                break ;
                
            }
            
        }
        
        if( ( s == -1 && e == -1 ) ){
            if( s == -1 && e == -1 ){
                return "" ;
            }else{
                return null ;
            }
        }
        s = ( s == -1 ) ? 0 : s ;
        e = ( e == -1 ) ? len-1 : e ;
        
        ret = str.substring( s,e+1 ) ;
        return ret ;
    }
}

