/*
 * @(#)AdminQueueCommand.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.command.admin ;

import org.maachang.command.core.AnalysisCommand;
import org.maachang.command.core.QueueParam;
import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.AdminQueue;
import org.maachang.queue.access.admin.MaachangQConnectAdmin;
import org.maachang.queue.access.admin.info.CreateQueueInfo;
import org.maachang.queue.access.admin.info.CreateSendQueueInfo;
import org.maachang.queue.access.admin.info.QueueInfo;
import org.maachang.queue.access.admin.info.QueueKeyInfo;
import org.maachang.queue.access.admin.info.SendQueueInfo;
import org.maachang.queue.access.status.QueueStatus;

/**
 * 管理者用キューコマンド.
 *  
 * @version 2007/01/25
 * @author  masahito suzuki
 * @since   MaachangQ-Command 1.00
 */
public class AdminQueueCommand {
    
    /**
     * 実行条件.
     */
    private static final String EXEC_HEAD = "x" ;
    
    /**
     * 最大メッセージ長.
     */
    private static final String MAX_MESSAGE = "g" ;
    
    /**
     * 最大キューサイズ.
     */
    private static final String MAX_QUEUE = "m" ;
    
    /**
     * 警告キューパーセント.
     */
    private static final String WARNING_PERSENT = "w" ;
    
    /**
     * 圧縮フラグ.
     */
    private static final String GZIP_FLAG = "z" ;
    
    /**
     * オートコミット.
     */
    private static final String AUTO_COMMIT = "a" ;
    
    /**
     * チャネル名群.
     */
    private static final String CHANNEL_NAMES = "s" ;
    
    /**
     * キャッシュ名.
     */
    private static final String CACHE_NAME = "e" ;
    
    
    
    /**
     * 実行タイプ : リスト表示.
     * list.
     */
    private static final String EXEC_LIST = "l" ;
    
    /**
     * 実行タイプ : キュー生成.
     * create.
     */
    private static final String EXEC_CREATE = "c" ;
    
    /**
     * 実行タイプ : キュー再設定.
     * set.
     */
    private static final String EXEC_SET = "s" ;
    
    /**
     * 実行タイプ : キュー削除.
     * remove.
     */
    private static final String EXEC_DELETE = "r" ;
    
    
    
    /**
     * コンストラクタ.
     */
    private AdminQueueCommand() {
        
    }
    
    /**
     * メイン.
     * <BR><BR>
     * @param args
     * @exception Exception
     */
    public static final void main( String[] args )
        throws Exception {
        try {
            new AdminQueueCommand().execution( args ) ;
        } catch( Exception e ) {
            AnalysisCommand.outputError( e ) ;
        }
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * @param args
     * @exception Exception
     */
    private final void execution( String[] args )
        throws Exception {
        
        String exec = this.execType( args ) ;
        int maxLength = this.getMaxMessage( args ) ;
        int maxQueue = this.getMaxQueue( args ) ;
        int warningPersent = this.getWarningPersent( args ) ;
        boolean gzip = this.getGZipFlag( args ) ;
        boolean autoCommit = this.getAutoCommit( args ) ;
        String[] channels = this.getChannels( args ) ;
        String cacheName = this.getCacheName( args ) ;
        
        QueueParam queueParam = AnalysisCommand.getQueueParam( args ) ;
        
        if( queueParam == null || queueParam.getManager() == null ) {
            throw new MaachangQException(
                "キューマネージャ、キュー名が設定されていません\n" +
                "      パラメータ[-Qmanager@queue:[send/receive]]で設定してください" ) ;
        }
        
        MaachangQConnectAdmin admin = AnalysisCommand.loginToConnect( args ) ;
        
        AdminQueue queue = admin.getAdminQueue( queueParam.getManager() ) ;
        
        // リスト表示.
        if( EXEC_LIST.equals( exec ) ) {
            if( queueParam.getQueue() != null &&
                queueParam.getQueue().length() > 0 ) {
                this.outputList( true,queue,queueParam.getQueue(),
                    queueParam.getType() ) ;
            }
            else {
                QueueKeyInfo[] keys = queue.getKeys() ;
                this.outputNames( true,queueParam.getManager(),keys ) ;
            }
        }
        // 生成.
        else if( EXEC_CREATE.equals( exec ) ) {
            
            if( queueParam.getQueue() == null ||
                queueParam.getQueue().length() <= 0 ) {
                throw new MaachangQException(
                    "対象キュー名は存在しません" ) ;
            }
            
            CreateQueueInfo info = null ;
            if( queueParam.getType() == MaachangQAccessDefine.TYPE_SEND ) {
                
                CreateSendQueueInfo send = new CreateSendQueueInfo() ;
                send.setChannels( channels ) ;
                info = ( CreateQueueInfo )send ;
                
            }
            else {
                info = new CreateQueueInfo() ;
            }
            
            info.setQueue( queueParam.getQueue() ) ;
            info.setMaxMessageLength( maxLength ) ;
            info.setMaxQueueSize( maxQueue ) ;
            info.setWarningPersend( warningPersent ) ;
            info.setGzipFlag( gzip ) ;
            info.setAutoCommit( autoCommit ) ;
            info.setCacheName( cacheName ) ;
            
            queue.addQueue( info ) ;
            
            System.out.println( queueParam.getQueue() +
                " キュー作成に成功しました" ) ;
        }
        // 再設定.
        else if( EXEC_SET.equals( exec ) ) {
            
            if( queueParam.getQueue() == null ||
                queueParam.getQueue().length() <= 0 ) {
                throw new MaachangQException(
                    "対象キュー名は存在しません" ) ;
            }
            
            QueueInfo info = null ;
            if( queueParam.getType() == MaachangQAccessDefine.TYPE_SEND ) {
                
                SendQueueInfo send = new SendQueueInfo() ;
                send.setChannels( channels ) ;
                info = ( QueueInfo )send ;
                
            }
            else {
                info = new QueueInfo() ;
            }
            
            info.setQueue( queueParam.getQueue() ) ;
            info.setMaxQueueSize( maxQueue ) ;
            info.setWarningPersend( warningPersent ) ;
            
            queue.setQueue( info ) ;
            
            System.out.println( queueParam.getQueue() +
                " キュー設定に成功しました" ) ;
        }
        // 削除.
        else if( EXEC_DELETE.equals( exec ) ) {
            
            if( queueParam.getQueue() == null ||
                queueParam.getQueue().length() <= 0 ) {
                throw new MaachangQException(
                    "対象キュー名は存在しません" ) ;
            }
            
            queue.removeQueue( queueParam.getType(),
                queueParam.getQueue() ) ;
            
            System.out.println( queueParam.getQueue() +
                " キュー削除に成功しました" ) ;
        }
        
    }
    
    /**
     * 実行条件を解析.
     */
    private final String execType( String[] args ) {
        String execType = AnalysisCommand.getHeader( args,EXEC_HEAD ) ;
        if( execType == null || ( execType = execType.trim().toLowerCase() ).length() <= 0 ) {
            return EXEC_LIST ;
        }
        else if( execType.startsWith( EXEC_LIST ) == true ) {
            return EXEC_LIST ;
        }
        else if( execType.startsWith( EXEC_CREATE ) == true ) {
            return EXEC_CREATE ;
        }
        else if( execType.startsWith( EXEC_SET ) == true ) {
            return EXEC_SET ;
        }
        else if( execType.startsWith( EXEC_DELETE ) == true ) {
            return EXEC_DELETE ;
        }
        else {
            return EXEC_LIST ;
        }
        
    }
    
    /**
     * 最大メッセージ長を取得.
     */
    private final int getMaxMessage( String[] args ) {
        return AnalysisCommand.getParamInt( args,MAX_MESSAGE ) ;
    }
    
    /**
     * 最大キューサイズを取得.
     */
    private final int getMaxQueue( String[] args ) {
        return AnalysisCommand.getParamInt( args,MAX_QUEUE ) ;
    }
    
    /**
     * 警告キューパーセントを取得.
     */
    private final int getWarningPersent( String[] args ) {
        return AnalysisCommand.getParamInt( args,WARNING_PERSENT ) ;
    }
    
    /**
     * 圧縮フラグを取得.
     */
    private final boolean getGZipFlag( String[] args ) {
        return AnalysisCommand.getParamBoolean( args,GZIP_FLAG ) ;
    }
    
    /**
     * オートコミットを取得.
     */
    private final boolean getAutoCommit( String[] args ) {
        return AnalysisCommand.getParamBoolean( args,AUTO_COMMIT ) ;
    }
    
    /**
     * 送信チャネル名群を取得.
     */
    private final String[] getChannels( String[] args ) {
        return AnalysisCommand.getParamArray( args,CHANNEL_NAMES ) ;
    }
    
    /**
     * キャッシュ名を取得.
     */
    private final String getCacheName( String[] args ) {
        return AnalysisCommand.getParamStringLowerCase( args,CACHE_NAME ) ;
    }
    
    /**
     * リスト表示.
     */
    private final void outputList( boolean mode,AdminQueue queue,
        String name,int type )
        throws Exception {
        
        QueueInfo info = queue.getQueue( type,name ) ;
        
        if( info != null ) {
            System.out.println( AnalysisCommand.CUT_CODE ) ;
            System.out.println( "name    :" + info.getQueue() ) ;
            if( type == MaachangQAccessDefine.TYPE_SEND ) {
                System.out.println( "type    :SendQueue" ) ;
            }
            else {
                System.out.println( "type    :ReceiveQueue" ) ;
            }
            System.out.println( "manager :" + info.getQueueManager() ) ;
            System.out.println( "maxQueue:" + info.getMaxQueueSize() ) ;
            System.out.println( "useQueue:" + info.getUseQueueSize() ) ;
            System.out.println( "warn    :" + info.getWarningPersend() + "%" ) ;
            System.out.println( "maxMsg  :" + info.getMaxMessageLength() ) ;
            System.out.println( "gzip    :" + info.isGzipFlag() ) ;
            System.out.println( "auto    :" + info.isAutoCommit() ) ;
            System.out.println( "state   :" + 
                QueueStatus.getStateByString( info.getState() ) ) ;
            
            if( type == MaachangQAccessDefine.TYPE_SEND ) {
                SendQueueInfo send = ( SendQueueInfo )info ;
                if( send.getChannelSize() > 0 ) {
                    int len = send.getChannelSize() ;
                    StringBuffer buf = new StringBuffer() ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        buf.append( send.getChannel( i ) ) ;
                        buf.append( " " ) ;
                    }
                    System.out.println( "channel :" + buf.toString() ) ;
                }
            }
            
        }
        else if( mode == true ) {
            System.out.println( "not queue." ) ;
        }
    }
    
    /**
     * 一覧名表示.
     */
    private final void outputNames( boolean mode,String manager,QueueKeyInfo[] names ) {
        if( names != null && names.length > 0 ) {
            int len = names.length ;
            System.out.println( AnalysisCommand.CUT_CODE ) ;
            System.out.println( "manager :" + manager ) ;
            System.out.println( AnalysisCommand.CUT_CODE ) ;
            for( int i = 0 ; i < len ; i ++ ) {
                System.out.println( "name    :" + names[ i ].getQueue() ) ;
                if( names[ i ].getType() == MaachangQAccessDefine.TYPE_SEND ) {
                    System.out.println( "type    :SendQueue" ) ;
                }
                else {
                    System.out.println( "type    :ReceiveQueue" ) ;
                }
            }
        }
        else if( mode == true ) {
            System.out.println( "not queue." ) ;
        }
    }
    
}

