package org.maachang.dbm.engine ;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.util.Arrays;

import org.maachang.util.ConvertParam;

/**
 * MemoryMapでのフラグ管理.
 * 
 * @version 2008/01/18
 * @author masahito suzuki
 * @since MaachangDBM 1.01
 */
class FlagMapFile {
    
    /**
     * 残りのファイルスペースマスクを判別する条件.
     */
    protected static final int ETC_MASK = 0x0000001f ;
    
    /**
     * ファイルスペースマスク.
     */
    protected static final int FSS_MASK = ~ETC_MASK ;
    
    /**
     * ビット係数.
     */
    protected static final int RIGHT_SHIFT = 5 ;
    
    /**
     * データオフセット値.
     */
    private static final int OFFSET = 12 ;
    
    /**
     * mapFile.
     */
    private MappedByteBuffer map = null ;
    
    /**
     * データ確保用.
     */
    private final byte[] tmp = new byte[ 4 ] ;
    
    /**
     * コンストラクタ.
     */
    private FlagMapFile() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param fp ランダムアクセスファイルオブジェクトを設定します.
     * @param startPos メモリーマップ開始ポジションを設定します.
     * @param size フラグ管理数を設定します.
     * @exception Exception 例外.
     */
    public FlagMapFile( RandomAccessFile fp,int startPos,int size )
        throws Exception {
        this( false,fp,startPos,size ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param mode 初期化する場合は[true]を設定します.
     * @param fp ランダムアクセスファイルオブジェクトを設定します.
     * @param startPos メモリーマップ開始ポジションを設定します.
     * @param size フラグ管理数を設定します.
     * @exception Exception 例外.
     */
    public FlagMapFile( boolean mode,RandomAccessFile fp,int startPos,int size )
        throws Exception {
        if( fp == null || startPos <= -1 || startPos >= fp.length() || size <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        int max = ( ( size & FSS_MASK ) >> RIGHT_SHIFT ) + ( ( ( size & ETC_MASK ) != 0 ) ? 1 : 0 ) ;
        int maxPos = max * 32 ;
        int all = ( max * 4 ) + OFFSET ;
        FileChannel channel = fp.getChannel() ;
        MappedByteBuffer map = channel.map( MapMode.READ_WRITE,startPos,all ) ;
        // 初期化.
        if( mode == true ) {
            byte[] b = new byte[ all ] ;
            Arrays.fill( b,( byte )0 ) ;
            map.put( b ) ;
        }
        this.map = map ;
        // 初期設定値を設定.
        mfMaxSize( maxPos ) ;
        mfMaxArray( max ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.clear() ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public synchronized void clear() {
        if( this.map != null ) {
            try {
                this.map.force() ;
            } catch( Exception e ) {
            }
        }
        this.map = null ;
    }
    
    /**
     * 情報更新.
     * <BR><BR>
     * 情報を更新します.
     * <BR>
     * @exception Exception 例外.
     */
    public synchronized void flush() throws Exception {
        if( this.map != null ) {
            this.map.force() ;
        }
    }
    
    /**
     * 指定位置のポジションを利用中に設定.
     * <BR><BR>
     * 指定位置のポジションを利用中に設定します.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @exception Exception 例外.
     */
    public synchronized void setPos( int pos ) throws Exception {
        int maxPos = mfMaxSize() ;
        if( pos < 0 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        int thisPos = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ;
        int innerPos = ( pos & ETC_MASK ) ;
        int target = mfArray( thisPos ) ;
        if( ( ( ( 1 << innerPos ) ) & target ) != 0 ) {
            throw new IOException( "指定項番["+pos+"]は既にONです" ) ;
        }
        target = ( 1 << innerPos ) | target ;
        mfArray( thisPos,target ) ;
        mfSize( mfSize() + 1 ) ;
    }
    
    /**
     * 指定位置のポジションを空ける.
     * <BR><BR>
     * 指定位置のポジションを空けます.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @exception Exception 例外.
     */
    public synchronized void removePos( int pos ) throws Exception {
        int maxPos = mfMaxSize() ;
        if( pos < 0 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        int thisPos = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ;
        int innerPos = ( pos & ETC_MASK ) ;
        int target = mfArray( thisPos ) ;
        if( ( ( ( 1 << innerPos ) ) & target ) == 0 ) {
            throw new IOException( "指定項番["+pos+"]は既にOFFです" ) ;
        }
        target = ( ~( 1 << innerPos ) ) & target ;
        mfArray( thisPos,target ) ;
        mfSize( mfSize() - 1 ) ;
    }
    
    /**
     * 指定位置のポジションが空いているか取得.
     * <BR><BR>
     * 指定位置のポジションが空いているか取得します.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @return boolean [false]の場合は、空いています.
     * @exception Exception 例外.
     */
    public synchronized boolean getPos( int pos ) throws Exception {
        int maxPos = mfMaxSize() ;
        if( pos < 0 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジション["+pos+"]は不正です" ) ;
        }
        return targetPos( pos ) ;
    }
    
    private boolean targetPos( int pos ) throws Exception {
        return ( ( ( 1 << ( pos & ETC_MASK ) ) ) & mfArray( ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ) ) != 0 ;
    }
    
    /**
     * 指定ポジションの次の有効な項番を取得.
     * <BR><BR>
     * 指定ポジションの次の有効な項番を取得します.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @return int 次に有効なポジションが返されます.
     * @exception Exception 例外.
     */
    public synchronized int useNextPos( int pos ) throws Exception {
        int maxPos = mfMaxSize() ;
        if( pos < -1 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        pos ++ ;
        for( ;; ) {
            if( pos >= maxPos ) {
                return -1 ;
            }
            if( targetPos( pos ) == true ) {
                return pos ;
            }
            pos ++ ;
        }
    }
    
    /**
     * 空いているポジション位置を取得.
     * <BR><BR>
     * 空いているポジション位置を取得します.
     * <BR>
     * @param pos 検索開始ポジションを設定します.
     * @return int 空いているポジション位置が返されます.<BR>
     *             [-1]が返された場合、空き位置は存在しません.
     * @exception Exception 例外.
     */
    public synchronized int usePos( int pos ) throws Exception {
        if( pos < 0 ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        int len = mfMaxArray() ;
        for( int i = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ; i < len ; i ++ ) {
            int target = mfArray( i ) ;
            if( target != 0xffffffff ) {
                for( int j = 0 ; j < 32 ; j ++ ) {
                    if( ( target & ( 1 << j ) ) == 0 ) {
                        return ( i * 32 ) + j ;
                    }
                }
            }
        }
        return -1 ;
    }
    
    /**
     * 空いているポジション位置を予約して取得.
     * <BR><BR>
     * 空いているポジション位置を予約して取得します.
     * <BR>
     * @param pos 検索開始ポジションを設定します.
     * @return int 空いているポジション位置が返されます.<BR>
     *             [-1]が返された場合、空き位置は存在しません.
     * @exception Exception 例外.
     */
    public synchronized int usePosBySet( int pos ) throws Exception {
        int u = usePos( pos ) ;
        if( u == -1 ) {
            return -1 ;
        }
        setPos( u ) ;
        return u ;
    }
    
    /**
     * 現在有効件数を取得.
     * <BR><BR>
     * 現在の有効件数を取得します.
     * <BR>
     * @return int 現在の有効件数が返されます.
     */
    public synchronized int size() {
        int ret = -1 ;
        try {
            ret = mfSize() ;
        } catch( Exception e ) {
            ret = -1 ;
        }
        return ret ;
    }
    
    /**
     * 現在の最大有効件数を取得.
     * <BR><BR>
     * 現在の最大有効件数が返されます.
     * <BR>
     * @return int 現在の最大有効件数が返されます.
     */
    public synchronized int maxSize() {
        int ret = -1 ;
        try {
            ret = mfMaxSize() ;
        } catch( Exception e ) {
            ret = -1 ;
        }
        return ret ;
    }
    
    private int mfMaxSize() throws Exception {
        this.map.position( 0 ) ;
        byte[] b = tmp ;
        this.map.get( b ) ;
        return ConvertParam.convertInt( 0,b ) ;
    }
    
    private void mfMaxSize( int n ) throws Exception {
        this.map.position( 0 ) ;
        this.map.put( ConvertParam.convertInt( n ) ) ;
    }
    
    private int mfSize() throws Exception {
        this.map.position( 4 ) ;
        byte[] b = tmp ;
        this.map.get( b ) ;
        return ConvertParam.convertInt( 0,b ) ;
    }
    
    private void mfSize( int n ) throws Exception {
        this.map.position( 4 ) ;
        this.map.put( ConvertParam.convertInt( n ) ) ;
    }
    
    private int mfMaxArray() throws Exception {
        this.map.position( 8 ) ;
        byte[] b = tmp ;
        this.map.get( b ) ;
        return ConvertParam.convertInt( 0,b ) ;
    }
    
    private void mfMaxArray( int n ) throws Exception {
        this.map.position( 8 ) ;
        this.map.put( ConvertParam.convertInt( n ) ) ;
    }
    
    private int mfArray( int no ) throws Exception {
        this.map.position( OFFSET + ( no * 4 ) ) ;
        byte[] b = tmp ;
        this.map.get( b ) ;
        return ConvertParam.convertInt( 0,b ) ;
    }
    
    private void mfArray( int no,int n ) throws Exception {
        this.map.position( OFFSET + ( no * 4 ) ) ;
        this.map.put( ConvertParam.convertInt( n ) ) ;
    }

}
