package org.maachang.dbm.engine ;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.util.Arrays;

import org.maachang.dbm.MDbmDefine;
import org.maachang.util.ConvertParam;
import org.maachang.util.FileUtil;

/**
 * Hash管理.
 * 
 * @version 2008/01/16
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class MHash {
    
    // Hash格納長を、環境変数で変更.
    static {
        int n = MDbmDefine.DEF_HASH ;
        if( System.getProperty( MDbmDefine.HASH_PROPERTY ) != null ) {
            try {
                n = Integer.parseInt( System.getProperty( MDbmDefine.HASH_PROPERTY ) ) ;
                if( n >= MDbmDefine.MAX_HASH ) {
                    n = MDbmDefine.MAX_HASH ;
                }
                else if( n <= MDbmDefine.MIN_HASH ) {
                    n = MDbmDefine.MIN_HASH ;
                }
            } catch( Exception e ) {
                n = MDbmDefine.DEF_HASH ;
            }
        }
        int bn ;
        switch( n ) {
            case 16 : bn = 0x0000ffff ; break ;
            case 17 : bn = 0x0001ffff ; break ;
            case 18 : bn = 0x0003ffff ; break ;
            case 19 : bn = 0x0007ffff ; break ;
            case 20 : bn = 0x000fffff ; break ;
            case 21 : bn = 0x001fffff ; break ;
            case 22 : bn = 0x003fffff ; break ;
            case 23 : bn = 0x007fffff ; break ;
            case 24 : bn = 0x00ffffff ; break ;
            case 25 : bn = 0x01ffffff ; break ;
            case 26 : bn = 0x03ffffff ; break ;
            case 27 : bn = 0x07ffffff ; break ;
            case 28 : bn = 0x0fffffff ; break ;
            case 29 : bn = 0x1fffffff ; break ;
            case 30 : bn = 0x3fffffff ; break ;
            case 31 : bn = 0x7fffffff ; break ;
            default : bn = 0x000fffff ; break ;
        }
        MAX_HASH_SIZE = bn + 1 ;
    }
    
    /**
     * Hashサイズ.
     */
    public static final int MAX_HASH_SIZE ;
    
    /**
     * シーケンスID最大数.
     */
    private static final int MAX_SEQUENCE_ID = 64 ;
    
    /**
     * ファイル.
     */
    private RandomAccessFile fp = null ;
    
    /**
     * mapFile.
     */
    private MappedByteBuffer map = null ;
    
    /**
     * ファイル名.
     */
    private String filename = null ;
    
    /**
     * データ確保用.
     */
    private final byte[] tmp = new byte[ 4 ] ;
    
    /**
     * コンストラクタ.
     */
    private MHash() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ファイル名を設定して、オブジェクトを生成します.
     * <BR>
     * @param filename 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public MHash( String filename ) throws Exception {
        if( filename == null || ( filename = filename.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "ファイル名は不正です" ) ;
        }
        filename = FileUtil.getFullPath( filename ) ;
        boolean isFile = FileUtil.isFileExists( filename ) ;
        RandomAccessFile fp = new RandomAccessFile( filename,"rwd" ) ;
        if( isFile == false ) {
            initFile( fp ) ;
        }
        FileChannel channel = fp.getChannel();
        MappedByteBuffer map = channel.map( MapMode.READ_WRITE,0,fp.length() );
        this.fp = fp ;
        this.map = map ;
        this.filename = filename ;
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public synchronized void destroy() {
        if( this.map != null ) {
            try {
                this.map.force() ;
            } catch( Exception e ) {
            }
        }
        if( this.fp != null ) {
            try {
                this.fp.close() ;
            } catch( Exception e ) {
            }
        }
        this.fp = null ;
        this.map = null ;
        this.filename = null ;
    }
    
    /**
     * 強制書き込み.
     * <BR>
     * @exception Exception 例外.
     */
    public synchronized void flush() throws Exception {
        if( this.map != null ) {
            this.map.force() ;
        }
    }
    
    /**
     * Hashポジションを設定.
     * <BR>
     * @param hash 対象のHashコードを設定します.
     * @param pos 対象のポジション値を設定します.
     * @exception Exception 例外.
     */
    public synchronized void put( int hash,int pos ) throws Exception {
        if( check() == false ) {
            throw new IOException( "既に破棄されています" ) ;
        }
        if( hash <= -1 || hash >= MAX_HASH_SIZE ) {
            throw new IllegalArgumentException( "指定Hashコード("+hash+")は範囲外です" ) ;
        }
        this.map.position( hash * 4 ) ;
        this.map.put( ConvertParam.convertInt( pos ) ) ;
    }
    
    /**
     * Hashポジションを削除.
     * <BR>
     * @param hash 対象のHashコードを設定します.
     * @exception Exception 例外.
     */
    public synchronized void remove( int hash ) throws Exception {
        this.put( hash,-1 ) ;
    }
    
    /**
     * Hashポジションを取得.
     * <BR>
     * @param hash 対象のHashコードを設定します.
     * @return int 対象のポジション値が返されます.<BR>
     *              [-1]の場合、無効です.
     * @exception Exception 例外.
     */
    public synchronized int get( int hash ) throws Exception {
        if( check() == false ) {
            throw new IOException( "既に破棄されています" ) ;
        }
        if( hash <= -1 || hash >= MAX_HASH_SIZE ) {
            throw new IllegalArgumentException( "指定Hashコード("+hash+")は範囲外です" ) ;
        }
        this.map.position( hash * 4 ) ;
        byte[] b = tmp ;
        this.map.get( b ) ;
        return ConvertParam.convertInt( 0,b ) ;
    }
    
    /**
     * ファイル名を取得.
     * <BR>
     * @return String ファイル名が返されます.
     */
    public synchronized String getFileName() {
        if( check() == false ) {
            return null ;
        }
        return this.filename ;
    }
    
    /**
     * 新しいシーケンスIDを取得.
     * <BR><BR>
     * 新しいシーケンスIDを取得します.
     * <BR>
     * @param no シーケンスNoを設定します.<BR>
     *           [0-31]まで利用可能です.
     * @return long 新しいシーケンスIDが返されます.
     * @exception Exception 例外.
     */
    public synchronized long sequenceId( int no ) throws Exception {
        if( check() == false ) {
            throw new IOException( "既に破棄されています" ) ;
        }
        if( no < 0 || no >= MAX_SEQUENCE_ID ) {
            return -1L ;
        }
        byte[] b = new byte[ 8 ] ;
        int off = ( MAX_HASH_SIZE * 4 ) + ( no * 8 ) ;
        this.map.position( off ) ;
        this.map.get( b ) ;
        long ret = ConvertParam.convertLong( 0,b ) ;
        b = null ;
        if( ret >= Long.MAX_VALUE ) {
            ret = 0L ;
        }
        else {
            ret += 1L ;
        }
        this.map.position( off ) ;
        this.map.put( ConvertParam.convertLong( ret ) ) ;
        return ret ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public synchronized boolean isUse() {
        return check() ;
    }
    
    private boolean check() {
        if( this.fp == null || this.map == null ) {
            return false ;
        }
        return true ;
    }
    
    /**
     * ファイル初期化.
     */
    private static final void initFile( RandomAccessFile fp ) throws Exception {
        int off = MAX_HASH_SIZE * 4 ;
        byte[] b = new byte[ off + ( 8 * MAX_SEQUENCE_ID ) ] ;
        Arrays.fill( b,( byte )0xff ) ;
        for( int i = 0 ; i < MAX_SEQUENCE_ID ; i ++ ) {
            b[ off ] = 0 ;
            b[ off+1 ] = 0 ;
            b[ off+2 ] = 0 ;
            b[ off+3 ] = 0 ;
            b[ off+4 ] = 0 ;
            b[ off+5 ] = 0 ;
            b[ off+6 ] = 0 ;
            b[ off+7 ] = 0 ;
            off += 8 ;
        }
        fp.write( b ) ;
    }
}
