package org.maachang.dbm.engine ;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.util.Arrays;

import org.maachang.util.ConvertParam;

/**
 * フラグ管理用.
 * 
 * @version 2008/06/05
 * @author masahito suzuki
 * @since MaachangDBM 1.12
 */
public class M2Flag {
    
    /**
     * 残りのファイルスペースマスクを判別する条件.
     */
    protected static final int ETC_MASK = 0x0000001f ;
    
    /**
     * ファイルスペースマスク.
     */
    protected static final int FSS_MASK = ~ETC_MASK ;
    
    /**
     * ビット係数.
     */
    protected static final int RIGHT_SHIFT = 5 ;
    
    /**
     * データオフセット値.
     */
    protected static final int OFFSET = 12 ;
    
    /**
     * mapFile.
     */
    private MappedByteBuffer map = null ;
    
    /**
     * データ確保用.
     */
    private final byte[] tmp = new byte[ 4 ] ;
    
    /**
     * 空き領域が存在する開始ポジション.
     */
    private int useStartPos = 0 ;
    
    /**
     * 最大サイズを保持.
     */
    private int maxBuf = -1 ;
    
    /**
     * 現在のサイズを保持.
     */
    private int sizeBuf = -1 ;
    
    /**
     * 最大配列を保持.
     */
    private int maxAryBuf = -1 ;
    
    /**
     * コンストラクタ.
     */
    private M2Flag() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param fp ランダムアクセスファイルオブジェクトを設定します.
     * @param startPos メモリーマップ開始ポジションを設定します.
     * @param size フラグ管理数を設定します.
     * @exception Exception 例外.
     */
    public M2Flag( RandomAccessFile fp,int startPos,int size )
        throws Exception {
        this( false,fp,startPos,size ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param mode 初期化する場合は[true]を設定します.
     * @param fp ランダムアクセスファイルオブジェクトを設定します.
     * @param startPos メモリーマップ開始ポジションを設定します.
     * @param size フラグ管理数を設定します.
     * @exception Exception 例外.
     */
    public M2Flag( boolean mode,RandomAccessFile fp,int startPos,int size )
        throws Exception {
        if( fp == null || startPos <= -1 || startPos >= fp.length() || size <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        int max = ( ( size & FSS_MASK ) >> RIGHT_SHIFT ) + ( ( ( size & ETC_MASK ) != 0 ) ? 1 : 0 ) ;
        int maxPos = max * 32 ;
        int all = ( max * 4 ) + OFFSET ;
        FileChannel channel = fp.getChannel() ;
        MappedByteBuffer map = channel.map( MapMode.READ_WRITE,startPos,all ) ;
        // 初期化.
        if( mode == true ) {
            byte[] b = new byte[ all ] ;
            Arrays.fill( b,( byte )0 ) ;
            map.put( b ) ;
        }
        this.map = map ;
        // 初期設定値を設定.
        mfMaxSize( maxPos ) ;
        mfMaxArray( max ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.clear() ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public synchronized void clear() {
        if( this.map != null ) {
            try {
                this.map.force() ;
            } catch( Exception e ) {
            }
        }
        this.map = null ;
        this.useStartPos = 0 ;
        this.maxBuf = -1 ;
        this.sizeBuf = -1 ;
        this.maxAryBuf = -1 ;
    }
    
    /**
     * 指定データサイズに対する領域を計算.
     * @param size 対象のサイズを設定します.
     * @return long サイズが返されます.
     */
    public static final long convertRect( int size ) {
        return ( ( ( ( size & FSS_MASK ) >> RIGHT_SHIFT ) + ( ( ( size & ETC_MASK ) != 0 ) ? 1 : 0 ) ) * 4 ) + OFFSET ;
    }
    
    /**
     * 情報更新.
     * <BR><BR>
     * 情報を更新します.
     * <BR>
     * @exception Exception 例外.
     */
    public synchronized void flush() throws Exception {
        if( this.map != null ) {
            this.map.force() ;
        }
    }
    
    /**
     * 指定位置のポジションを利用中に設定.
     * <BR><BR>
     * 指定位置のポジションを利用中に設定します.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @exception Exception 例外.
     */
    public synchronized void setPos( int pos ) throws Exception {
        int maxPos = mfMaxSize() ;
        if( pos < 0 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        int thisPos = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ;
        int innerPos = ( pos & ETC_MASK ) ;
        int target = mfArray( thisPos ) ;
        if( ( ( ( 1 << innerPos ) ) & target ) != 0 ) {
            throw new IOException( "指定項番["+pos+"]は既にONです" ) ;
        }
        target = ( 1 << innerPos ) | target ;
        mfArray( thisPos,target ) ;
        mfSize( mfSize() + 1 ) ;
        
        useStartPos = pos+1 ;
    }
    
    /**
     * 指定位置のポジションを空ける.
     * <BR><BR>
     * 指定位置のポジションを空けます.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @exception Exception 例外.
     */
    public synchronized void removePos( int pos ) throws Exception {
        int maxPos = mfMaxSize() ;
        if( pos < 0 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        int thisPos = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ;
        int innerPos = ( pos & ETC_MASK ) ;
        int target = mfArray( thisPos ) ;
        if( ( ( ( 1 << innerPos ) ) & target ) == 0 ) {
            throw new IOException( "指定項番["+pos+"]は既にOFFです" ) ;
        }
        target = ( ~( 1 << innerPos ) ) & target ;
        mfArray( thisPos,target ) ;
        mfSize( mfSize() - 1 ) ;
        
        useStartPos = 0 ;
    }
    
    /**
     * 指定位置のポジションが空いているか取得.
     * <BR><BR>
     * 指定位置のポジションが空いているか取得します.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @return boolean [false]の場合は、空いています.
     * @exception Exception 例外.
     */
    public synchronized boolean getPos( int pos ) throws Exception {
        int maxPos = mfMaxSize() ;
        if( pos < 0 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジション["+pos+"]は不正です" ) ;
        }
        return targetPos( pos ) ;
    }
    
    private boolean targetPos( int pos ) throws Exception {
        return ( ( ( 1 << ( pos & ETC_MASK ) ) ) & mfArray( ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ) ) != 0 ;
    }
    
    private boolean targetPos( int pos,int one ) throws Exception {
        return ( ( ( 1 << ( pos & ETC_MASK ) ) ) & one ) != 0 ;
    }
    
    /**
     * 指定ポジションの次の有効な項番を取得.
     * <BR><BR>
     * 指定ポジションの次の有効な項番を取得します.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @return int 次に有効なポジションが返されます.
     * @exception Exception 例外.
     */
    public synchronized int useNextPos( int pos ) throws Exception {
        int maxPos = mfMaxSize() ;
        if( pos < -1 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        int one = 0 ;
        pos ++ ;
        if( ( pos & ETC_MASK ) != 0 ) {
            one = mfArray( ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ) ;
        }
        for( ;; ) {
            if( pos >= maxPos ) {
                return -1 ;
            }
            if( ( pos & ETC_MASK ) == 0 ) {
                one = mfArray( ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ) ;
                if( one == 0 ) {
                    pos += 32 ;
                    continue ;
                }
            }
            if( targetPos( pos,one ) == true ) {
                return pos ;
            }
            pos ++ ;
        }
    }
    
    /**
     * 空いているポジション位置を取得.
     * <BR><BR>
     * 空いているポジション位置を取得します.
     * <BR>
     * @param pos 検索開始ポジションを設定します.
     * @return int 空いているポジション位置が返されます.<BR>
     *             [-1]が返された場合、空き位置は存在しません.
     * @exception Exception 例外.
     */
    public synchronized int usePos( int pos ) throws Exception {
        if( pos < 0 ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        else if( pos == 0 ) {
            pos = useStartPos ;
        }
        int len = mfMaxArray() ;
        for( int i = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ; i < len ; i ++ ) {
            int target = mfArray( i ) ;
            if( target != 0xffffffff ) {
                for( int j = 0 ; j < 32 ; j ++ ) {
                    if( ( target & ( 1 << j ) ) == 0 ) {
                        return ( i * 32 ) + j ;
                    }
                }
            }
        }
        //useStartPos = 0 ;
        return -1 ;
    }
    
    /**
     * 空いているポジション位置を予約して取得.
     * <BR><BR>
     * 空いているポジション位置を予約して取得します.
     * <BR>
     * @param pos 検索開始ポジションを設定します.
     * @return int 空いているポジション位置が返されます.<BR>
     *             [-1]が返された場合、空き位置は存在しません.
     * @exception Exception 例外.
     */
    public synchronized int usePosBySet( int pos ) throws Exception {
        int u = usePos( pos ) ;
        if( u == -1 ) {
            return -1 ;
        }
        setPos( u ) ;
        return u ;
    }
    
    /**
     * 現在有効件数を取得.
     * <BR><BR>
     * 現在の有効件数を取得します.
     * <BR>
     * @return int 現在の有効件数が返されます.
     * @exception Exception 例外.
     */
    public synchronized int size() throws Exception {
        return mfSize() ;
    }
    
    /**
     * 現在の最大有効件数を取得.
     * <BR><BR>
     * 現在の最大有効件数が返されます.
     * <BR>
     * @return int 現在の最大有効件数が返されます.
     * @exception Exception 例外.
     */
    public synchronized int maxSize() throws Exception {
        return mfMaxSize() ;
    }
    
    private int mfMaxSize() throws Exception {
        int ret ;
        if( this.maxBuf <= -1 ) {
            this.map.position( 0 ) ;
            byte[] b = tmp ;
            this.map.get( b ) ;
            ret = ConvertParam.convertInt( 0,b ) ;
            this.maxBuf = ret ;
        }
        else {
            ret = this.maxBuf ;
        }
        return ret ;
    }
    
    private void mfMaxSize( int n ) throws Exception {
        this.map.position( 0 ) ;
        this.map.put( ConvertParam.convertInt( n ) ) ;
        this.maxBuf = -1 ;
    }
    
    private int mfSize() throws Exception {
        int ret ;
        if( this.sizeBuf <= -1 ) {
            this.map.position( 4 ) ;
            byte[] b = tmp ;
            this.map.get( b ) ;
            ret = ConvertParam.convertInt( 0,b ) ;
            this.sizeBuf = ret ;
        }
        else {
            ret = this.sizeBuf ;
        }
        return ret ;
    }
    
    private void mfSize( int n ) throws Exception {
        this.map.position( 4 ) ;
        this.map.put( ConvertParam.convertInt( n ) ) ;
        this.sizeBuf = -1 ;
    }
    
    private int mfMaxArray() throws Exception {
        int ret ;
        if( maxAryBuf <= -1 ) {
            this.map.position( 8 ) ;
            byte[] b = tmp ;
            this.map.get( b ) ;
            ret = ConvertParam.convertInt( 0,b ) ;
            this.maxAryBuf = ret ;
        }
        else {
            ret = this.maxAryBuf ;
        }
        return ret ;
    }
    
    private void mfMaxArray( int n ) throws Exception {
        this.map.position( 8 ) ;
        this.map.put( ConvertParam.convertInt( n ) ) ;
        this.maxAryBuf = -1 ;
    }
    
    private int mfArray( int no ) throws Exception {
        this.map.position( OFFSET + ( no * 4 ) ) ;
        byte[] b = tmp ;
        this.map.get( b ) ;
        return ConvertParam.convertInt( 0,b ) ;
    }
    
    private void mfArray( int no,int n ) throws Exception {
        this.map.position( OFFSET + ( no * 4 ) ) ;
        this.map.put( ConvertParam.convertInt( n ) ) ;
    }

}
